{-# LANGUAGE MagicHash #-}
{-# LANGUAGE UnboxedTuples #-}
{-# LANGUAGE CPP #-}
module Main where

#include "MachDeps.h"

import GHC.Prim
import GHC.Word
import GHC.Types

import Data.Bits
import Text.Printf

data Test = T
  { numeratorTop      :: Word#
  , numeratorBottom   :: Word#
  , denominator       :: Word#
  , expectedQuotient  :: Word#
  , expectedRemainder :: Word#
  } deriving (Eq, Ord)

instance Show Test where
  show (T n1 n2 d q r) =
    let formatStr =
          if (maxBound :: Word) == 4294967295
          then "T {-n1-}0x%08x## {-n2-}0x%08x## {-d-}0x%08x## {-e_q-}0x%08x## {-e_r-}0x%08x##"
          else "T {-n1-}0x%016x## {-n2-}0x%016x## {-d-}0x%016x## {-e_q-}0x%016x## {-e_r-}0x%016x##"
    in printf formatStr (W# n1) (W# n2) (W# d) (W# q) (W# r)


main = mapM_ runTest testCases

runTest :: Test -> IO ()
runTest t@(T n1 n2 d eq er) =
  case quotRemWord2# n1 n2 d of
    (# q, r #) ->
      if isTrue# ((neWord# q eq) `orI#` (neWord# r er))
      then do
        print t
        print ("Expected", W# eq, W# er)
        print ("Got     ", W#  q, W#  r)
      else pure ()

{-
-- The following is used to generate the 32 bit values in `testCases` below
find32BitCases :: [Test] -> [Test]
find32BitCases =
  map  (\(T n1 n2 d _ _) ->
        case resultsForQuotRem (to32BitInteger n1 n2) d of (# q, r #) -> T n1 n2 d q r)
  . filter (\(T n1 n2 d _ _) -> is32Bit n1 && is32Bit n2 && is32Bit d)

is32Bit :: Word# -> Bool
is32Bit w = isTrue# (and# w 0xFFFFFFFF00000000## `eqWord#` 0##)

to32BitInteger :: Word# -> Word# -> Integer
to32BitInteger n1 n2 = fromIntegral (W# n1) `shiftL` 32 + fromIntegral (W# n2)

resultsForQuotRem :: Integer -> Word# -> (# Word#, Word# #)
resultsForQuotRem n d = case quotRem n (fromIntegral (W# d)) of
  (q,r) -> case (fromIntegral q, fromIntegral r) of
    (W# q', W# r') -> (# q', r' #)

testFor :: Word -> Word -> Word -> Test
testFor (W# n1) (W# n2) (W# d) = case resultsForQuotRem (to32BitInteger n1 n2) d of
  (# q, r #) -> T n1 n2 d q r

generate32BitCases :: [Test]
generate32BitCases =
  [ r
  | n1 <- numeratorCases
  , n2 <- numeratorCases
  , d <- numeratorCases ++ [3,4,5,6,7,8,9,11,13,17,0x0000_1000,0x0001_0000,0xFAFA_AFAF]
  , let r@(T _ _ _ q _) = testFor n1 n2 d
  , W# q <= fromIntegral (maxBound @Word32)
  ]
    where
      numeratorCases = [0x0000_0001, 0x0000_0002, 0x0000_FFFF, 0x0000_AAAA, 0xFFFF_0000, 0xAAAA_0000, 0xFFFF_FFFF, 0xAAAA_AAAA]
-}

-- 64 bit test cases are from compiler-rt's test suite:
-- https://github.com/llvm/llvm-project/blob/ba702aa067c0659d2263d2144d38666efa4a1750/compiler-rt/test/builtins/Unit/udivmoddi4_test.c
-- These test a n128/n128 division, so divisors with a non-zero top word have been removed.
testCases :: [Test]
testCases =
  [ T 0## 0## 1## 0## 0##
#if WORD_SIZE_IN_BITS == 32
  , T {-n1-}0x00000001## {-n2-}0x00000001## {-d-}0x00000002##  {-e_q-}0x80000000## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0x00000001## {-d-}0x0000ffff##  {-e_q-}0x00010001## {-e_r-}0x00000002##
  , T {-n1-}0x00000001## {-n2-}0x00000001## {-d-}0x0000aaaa##  {-e_q-}0x00018001## {-e_r-}0x00005557##
  , T {-n1-}0x00000001## {-n2-}0x00000001## {-d-}0xffff0000##  {-e_q-}0x00000001## {-e_r-}0x00010001##
  , T {-n1-}0x00000001## {-n2-}0x00000001## {-d-}0xaaaa0000##  {-e_q-}0x00000001## {-e_r-}0x55560001##
  , T {-n1-}0x00000001## {-n2-}0x00000001## {-d-}0xffffffff##  {-e_q-}0x00000001## {-e_r-}0x00000002##
  , T {-n1-}0x00000001## {-n2-}0x00000001## {-d-}0xaaaaaaaa##  {-e_q-}0x00000001## {-e_r-}0x55555557##
  , T {-n1-}0x00000001## {-n2-}0x00000001## {-d-}0x00000003##  {-e_q-}0x55555555## {-e_r-}0x00000002##
  , T {-n1-}0x00000001## {-n2-}0x00000001## {-d-}0x00000004##  {-e_q-}0x40000000## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0x00000001## {-d-}0x00000005##  {-e_q-}0x33333333## {-e_r-}0x00000002##
  , T {-n1-}0x00000001## {-n2-}0x00000001## {-d-}0x00000006##  {-e_q-}0x2aaaaaaa## {-e_r-}0x00000005##
  , T {-n1-}0x00000001## {-n2-}0x00000001## {-d-}0x00000007##  {-e_q-}0x24924924## {-e_r-}0x00000005##
  , T {-n1-}0x00000001## {-n2-}0x00000001## {-d-}0x00000008##  {-e_q-}0x20000000## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0x00000001## {-d-}0x00000009##  {-e_q-}0x1c71c71c## {-e_r-}0x00000005##
  , T {-n1-}0x00000001## {-n2-}0x00000001## {-d-}0x0000000b##  {-e_q-}0x1745d174## {-e_r-}0x00000005##
  , T {-n1-}0x00000001## {-n2-}0x00000001## {-d-}0x0000000d##  {-e_q-}0x13b13b13## {-e_r-}0x0000000a##
  , T {-n1-}0x00000001## {-n2-}0x00000001## {-d-}0x00000011##  {-e_q-}0x0f0f0f0f## {-e_r-}0x00000002##
  , T {-n1-}0x00000001## {-n2-}0x00000001## {-d-}0x00001000##  {-e_q-}0x00100000## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0x00000001## {-d-}0x00010000##  {-e_q-}0x00010000## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0x00000001## {-d-}0xfafaafaf##  {-e_q-}0x00000001## {-e_r-}0x05055052##
  , T {-n1-}0x00000001## {-n2-}0x00000002## {-d-}0x00000002##  {-e_q-}0x80000001## {-e_r-}0x00000000##
  , T {-n1-}0x00000001## {-n2-}0x00000002## {-d-}0x0000ffff##  {-e_q-}0x00010001## {-e_r-}0x00000003##
  , T {-n1-}0x00000001## {-n2-}0x00000002## {-d-}0x0000aaaa##  {-e_q-}0x00018001## {-e_r-}0x00005558##
  , T {-n1-}0x00000001## {-n2-}0x00000002## {-d-}0xffff0000##  {-e_q-}0x00000001## {-e_r-}0x00010002##
  , T {-n1-}0x00000001## {-n2-}0x00000002## {-d-}0xaaaa0000##  {-e_q-}0x00000001## {-e_r-}0x55560002##
  , T {-n1-}0x00000001## {-n2-}0x00000002## {-d-}0xffffffff##  {-e_q-}0x00000001## {-e_r-}0x00000003##
  , T {-n1-}0x00000001## {-n2-}0x00000002## {-d-}0xaaaaaaaa##  {-e_q-}0x00000001## {-e_r-}0x55555558##
  , T {-n1-}0x00000001## {-n2-}0x00000002## {-d-}0x00000003##  {-e_q-}0x55555556## {-e_r-}0x00000000##
  , T {-n1-}0x00000001## {-n2-}0x00000002## {-d-}0x00000004##  {-e_q-}0x40000000## {-e_r-}0x00000002##
  , T {-n1-}0x00000001## {-n2-}0x00000002## {-d-}0x00000005##  {-e_q-}0x33333333## {-e_r-}0x00000003##
  , T {-n1-}0x00000001## {-n2-}0x00000002## {-d-}0x00000006##  {-e_q-}0x2aaaaaab## {-e_r-}0x00000000##
  , T {-n1-}0x00000001## {-n2-}0x00000002## {-d-}0x00000007##  {-e_q-}0x24924924## {-e_r-}0x00000006##
  , T {-n1-}0x00000001## {-n2-}0x00000002## {-d-}0x00000008##  {-e_q-}0x20000000## {-e_r-}0x00000002##
  , T {-n1-}0x00000001## {-n2-}0x00000002## {-d-}0x00000009##  {-e_q-}0x1c71c71c## {-e_r-}0x00000006##
  , T {-n1-}0x00000001## {-n2-}0x00000002## {-d-}0x0000000b##  {-e_q-}0x1745d174## {-e_r-}0x00000006##
  , T {-n1-}0x00000001## {-n2-}0x00000002## {-d-}0x0000000d##  {-e_q-}0x13b13b13## {-e_r-}0x0000000b##
  , T {-n1-}0x00000001## {-n2-}0x00000002## {-d-}0x00000011##  {-e_q-}0x0f0f0f0f## {-e_r-}0x00000003##
  , T {-n1-}0x00000001## {-n2-}0x00000002## {-d-}0x00001000##  {-e_q-}0x00100000## {-e_r-}0x00000002##
  , T {-n1-}0x00000001## {-n2-}0x00000002## {-d-}0x00010000##  {-e_q-}0x00010000## {-e_r-}0x00000002##
  , T {-n1-}0x00000001## {-n2-}0x00000002## {-d-}0xfafaafaf##  {-e_q-}0x00000001## {-e_r-}0x05055053##
  , T {-n1-}0x00000001## {-n2-}0x0000ffff## {-d-}0x00000002##  {-e_q-}0x80007fff## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0x0000ffff## {-d-}0x0000ffff##  {-e_q-}0x00010002## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0x0000ffff## {-d-}0x0000aaaa##  {-e_q-}0x00018003## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0x0000ffff## {-d-}0xffff0000##  {-e_q-}0x00000001## {-e_r-}0x0001ffff##
  , T {-n1-}0x00000001## {-n2-}0x0000ffff## {-d-}0xaaaa0000##  {-e_q-}0x00000001## {-e_r-}0x5556ffff##
  , T {-n1-}0x00000001## {-n2-}0x0000ffff## {-d-}0xffffffff##  {-e_q-}0x00000001## {-e_r-}0x00010000##
  , T {-n1-}0x00000001## {-n2-}0x0000ffff## {-d-}0xaaaaaaaa##  {-e_q-}0x00000001## {-e_r-}0x55565555##
  , T {-n1-}0x00000001## {-n2-}0x0000ffff## {-d-}0x00000003##  {-e_q-}0x5555aaaa## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0x0000ffff## {-d-}0x00000004##  {-e_q-}0x40003fff## {-e_r-}0x00000003##
  , T {-n1-}0x00000001## {-n2-}0x0000ffff## {-d-}0x00000005##  {-e_q-}0x33336666## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0x0000ffff## {-d-}0x00000006##  {-e_q-}0x2aaad555## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0x0000ffff## {-d-}0x00000007##  {-e_q-}0x24926db6## {-e_r-}0x00000005##
  , T {-n1-}0x00000001## {-n2-}0x0000ffff## {-d-}0x00000008##  {-e_q-}0x20001fff## {-e_r-}0x00000007##
  , T {-n1-}0x00000001## {-n2-}0x0000ffff## {-d-}0x00000009##  {-e_q-}0x1c71e38e## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0x0000ffff## {-d-}0x0000000b##  {-e_q-}0x1745e8ba## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0x0000ffff## {-d-}0x0000000d##  {-e_q-}0x13b14ec4## {-e_r-}0x0000000b##
  , T {-n1-}0x00000001## {-n2-}0x0000ffff## {-d-}0x00000011##  {-e_q-}0x0f0f1e1e## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0x0000ffff## {-d-}0x00001000##  {-e_q-}0x0010000f## {-e_r-}0x00000fff##
  , T {-n1-}0x00000001## {-n2-}0x0000ffff## {-d-}0x00010000##  {-e_q-}0x00010000## {-e_r-}0x0000ffff##
  , T {-n1-}0x00000001## {-n2-}0x0000ffff## {-d-}0xfafaafaf##  {-e_q-}0x00000001## {-e_r-}0x05065050##
  , T {-n1-}0x00000001## {-n2-}0x0000aaaa## {-d-}0x00000002##  {-e_q-}0x80005555## {-e_r-}0x00000000##
  , T {-n1-}0x00000001## {-n2-}0x0000aaaa## {-d-}0x0000ffff##  {-e_q-}0x00010001## {-e_r-}0x0000aaab##
  , T {-n1-}0x00000001## {-n2-}0x0000aaaa## {-d-}0x0000aaaa##  {-e_q-}0x00018002## {-e_r-}0x00005556##
  , T {-n1-}0x00000001## {-n2-}0x0000aaaa## {-d-}0xffff0000##  {-e_q-}0x00000001## {-e_r-}0x0001aaaa##
  , T {-n1-}0x00000001## {-n2-}0x0000aaaa## {-d-}0xaaaa0000##  {-e_q-}0x00000001## {-e_r-}0x5556aaaa##
  , T {-n1-}0x00000001## {-n2-}0x0000aaaa## {-d-}0xffffffff##  {-e_q-}0x00000001## {-e_r-}0x0000aaab##
  , T {-n1-}0x00000001## {-n2-}0x0000aaaa## {-d-}0xaaaaaaaa##  {-e_q-}0x00000001## {-e_r-}0x55560000##
  , T {-n1-}0x00000001## {-n2-}0x0000aaaa## {-d-}0x00000003##  {-e_q-}0x55558e38## {-e_r-}0x00000002##
  , T {-n1-}0x00000001## {-n2-}0x0000aaaa## {-d-}0x00000004##  {-e_q-}0x40002aaa## {-e_r-}0x00000002##
  , T {-n1-}0x00000001## {-n2-}0x0000aaaa## {-d-}0x00000005##  {-e_q-}0x33335555## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0x0000aaaa## {-d-}0x00000006##  {-e_q-}0x2aaac71c## {-e_r-}0x00000002##
  , T {-n1-}0x00000001## {-n2-}0x0000aaaa## {-d-}0x00000007##  {-e_q-}0x24926186## {-e_r-}0x00000000##
  , T {-n1-}0x00000001## {-n2-}0x0000aaaa## {-d-}0x00000008##  {-e_q-}0x20001555## {-e_r-}0x00000002##
  , T {-n1-}0x00000001## {-n2-}0x0000aaaa## {-d-}0x00000009##  {-e_q-}0x1c71da12## {-e_r-}0x00000008##
  , T {-n1-}0x00000001## {-n2-}0x0000aaaa## {-d-}0x0000000b##  {-e_q-}0x1745e0f8## {-e_r-}0x00000002##
  , T {-n1-}0x00000001## {-n2-}0x0000aaaa## {-d-}0x0000000d##  {-e_q-}0x13b14834## {-e_r-}0x00000006##
  , T {-n1-}0x00000001## {-n2-}0x0000aaaa## {-d-}0x00000011##  {-e_q-}0x0f0f1919## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0x0000aaaa## {-d-}0x00001000##  {-e_q-}0x0010000a## {-e_r-}0x00000aaa##
  , T {-n1-}0x00000001## {-n2-}0x0000aaaa## {-d-}0x00010000##  {-e_q-}0x00010000## {-e_r-}0x0000aaaa##
  , T {-n1-}0x00000001## {-n2-}0x0000aaaa## {-d-}0xfafaafaf##  {-e_q-}0x00000001## {-e_r-}0x0505fafb##
  , T {-n1-}0x00000001## {-n2-}0xffff0000## {-d-}0x00000002##  {-e_q-}0xffff8000## {-e_r-}0x00000000##
  , T {-n1-}0x00000001## {-n2-}0xffff0000## {-d-}0x0000ffff##  {-e_q-}0x00020001## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0xffff0000## {-d-}0x0000aaaa##  {-e_q-}0x00030001## {-e_r-}0x00005556##
  , T {-n1-}0x00000001## {-n2-}0xffff0000## {-d-}0xffff0000##  {-e_q-}0x00000002## {-e_r-}0x00010000##
  , T {-n1-}0x00000001## {-n2-}0xffff0000## {-d-}0xaaaa0000##  {-e_q-}0x00000003## {-e_r-}0x00010000##
  , T {-n1-}0x00000001## {-n2-}0xffff0000## {-d-}0xffffffff##  {-e_q-}0x00000001## {-e_r-}0xffff0001##
  , T {-n1-}0x00000001## {-n2-}0xffff0000## {-d-}0xaaaaaaaa##  {-e_q-}0x00000002## {-e_r-}0xaaa9aaac##
  , T {-n1-}0x00000001## {-n2-}0xffff0000## {-d-}0x00000003##  {-e_q-}0xaaaa5555## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0xffff0000## {-d-}0x00000004##  {-e_q-}0x7fffc000## {-e_r-}0x00000000##
  , T {-n1-}0x00000001## {-n2-}0xffff0000## {-d-}0x00000005##  {-e_q-}0x66663333## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0xffff0000## {-d-}0x00000006##  {-e_q-}0x55552aaa## {-e_r-}0x00000004##
  , T {-n1-}0x00000001## {-n2-}0xffff0000## {-d-}0x00000007##  {-e_q-}0x49246db6## {-e_r-}0x00000006##
  , T {-n1-}0x00000001## {-n2-}0xffff0000## {-d-}0x00000008##  {-e_q-}0x3fffe000## {-e_r-}0x00000000##
  , T {-n1-}0x00000001## {-n2-}0xffff0000## {-d-}0x00000009##  {-e_q-}0x38e371c7## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0xffff0000## {-d-}0x0000000b##  {-e_q-}0x2e8b8ba2## {-e_r-}0x0000000a##
  , T {-n1-}0x00000001## {-n2-}0xffff0000## {-d-}0x0000000d##  {-e_q-}0x27626276## {-e_r-}0x00000002##
  , T {-n1-}0x00000001## {-n2-}0xffff0000## {-d-}0x00000011##  {-e_q-}0x1e1e0f0f## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0xffff0000## {-d-}0x00001000##  {-e_q-}0x001ffff0## {-e_r-}0x00000000##
  , T {-n1-}0x00000001## {-n2-}0xffff0000## {-d-}0x00010000##  {-e_q-}0x0001ffff## {-e_r-}0x00000000##
  , T {-n1-}0x00000001## {-n2-}0xffff0000## {-d-}0xfafaafaf##  {-e_q-}0x00000002## {-e_r-}0x0a09a0a2##
  , T {-n1-}0x00000001## {-n2-}0xaaaa0000## {-d-}0x00000002##  {-e_q-}0xd5550000## {-e_r-}0x00000000##
  , T {-n1-}0x00000001## {-n2-}0xaaaa0000## {-d-}0x0000ffff##  {-e_q-}0x0001aaab## {-e_r-}0x0000aaab##
  , T {-n1-}0x00000001## {-n2-}0xaaaa0000## {-d-}0x0000aaaa##  {-e_q-}0x00028001## {-e_r-}0x00005556##
  , T {-n1-}0x00000001## {-n2-}0xaaaa0000## {-d-}0xffff0000##  {-e_q-}0x00000001## {-e_r-}0xaaab0000##
  , T {-n1-}0x00000001## {-n2-}0xaaaa0000## {-d-}0xaaaa0000##  {-e_q-}0x00000002## {-e_r-}0x55560000##
  , T {-n1-}0x00000001## {-n2-}0xaaaa0000## {-d-}0xffffffff##  {-e_q-}0x00000001## {-e_r-}0xaaaa0001##
  , T {-n1-}0x00000001## {-n2-}0xaaaa0000## {-d-}0xaaaaaaaa##  {-e_q-}0x00000002## {-e_r-}0x5554aaac##
  , T {-n1-}0x00000001## {-n2-}0xaaaa0000## {-d-}0x00000003##  {-e_q-}0x8e38aaaa## {-e_r-}0x00000002##
  , T {-n1-}0x00000001## {-n2-}0xaaaa0000## {-d-}0x00000004##  {-e_q-}0x6aaa8000## {-e_r-}0x00000000##
  , T {-n1-}0x00000001## {-n2-}0xaaaa0000## {-d-}0x00000005##  {-e_q-}0x55553333## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0xaaaa0000## {-d-}0x00000006##  {-e_q-}0x471c5555## {-e_r-}0x00000002##
  , T {-n1-}0x00000001## {-n2-}0xaaaa0000## {-d-}0x00000007##  {-e_q-}0x3cf3b6db## {-e_r-}0x00000003##
  , T {-n1-}0x00000001## {-n2-}0xaaaa0000## {-d-}0x00000008##  {-e_q-}0x35554000## {-e_r-}0x00000000##
  , T {-n1-}0x00000001## {-n2-}0xaaaa0000## {-d-}0x00000009##  {-e_q-}0x2f6838e3## {-e_r-}0x00000005##
  , T {-n1-}0x00000001## {-n2-}0xaaaa0000## {-d-}0x0000000b##  {-e_q-}0x26c9a2e8## {-e_r-}0x00000008##
  , T {-n1-}0x00000001## {-n2-}0xaaaa0000## {-d-}0x0000000d##  {-e_q-}0x20d20000## {-e_r-}0x00000000##
  , T {-n1-}0x00000001## {-n2-}0xaaaa0000## {-d-}0x00000011##  {-e_q-}0x19190f0f## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0xaaaa0000## {-d-}0x00001000##  {-e_q-}0x001aaaa0## {-e_r-}0x00000000##
  , T {-n1-}0x00000001## {-n2-}0xaaaa0000## {-d-}0x00010000##  {-e_q-}0x0001aaaa## {-e_r-}0x00000000##
  , T {-n1-}0x00000001## {-n2-}0xaaaa0000## {-d-}0xfafaafaf##  {-e_q-}0x00000001## {-e_r-}0xafaf5051##
  , T {-n1-}0x00000001## {-n2-}0xffffffff## {-d-}0x00000002##  {-e_q-}0xffffffff## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0xffffffff## {-d-}0x0000ffff##  {-e_q-}0x00020002## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0xffffffff## {-d-}0x0000aaaa##  {-e_q-}0x00030003## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0xffffffff## {-d-}0xffff0000##  {-e_q-}0x00000002## {-e_r-}0x0001ffff##
  , T {-n1-}0x00000001## {-n2-}0xffffffff## {-d-}0xaaaa0000##  {-e_q-}0x00000003## {-e_r-}0x0001ffff##
  , T {-n1-}0x00000001## {-n2-}0xffffffff## {-d-}0xffffffff##  {-e_q-}0x00000002## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0xffffffff## {-d-}0xaaaaaaaa##  {-e_q-}0x00000003## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0xffffffff## {-d-}0x00000003##  {-e_q-}0xaaaaaaaa## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0xffffffff## {-d-}0x00000004##  {-e_q-}0x7fffffff## {-e_r-}0x00000003##
  , T {-n1-}0x00000001## {-n2-}0xffffffff## {-d-}0x00000005##  {-e_q-}0x66666666## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0xffffffff## {-d-}0x00000006##  {-e_q-}0x55555555## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0xffffffff## {-d-}0x00000007##  {-e_q-}0x49249249## {-e_r-}0x00000000##
  , T {-n1-}0x00000001## {-n2-}0xffffffff## {-d-}0x00000008##  {-e_q-}0x3fffffff## {-e_r-}0x00000007##
  , T {-n1-}0x00000001## {-n2-}0xffffffff## {-d-}0x00000009##  {-e_q-}0x38e38e38## {-e_r-}0x00000007##
  , T {-n1-}0x00000001## {-n2-}0xffffffff## {-d-}0x0000000b##  {-e_q-}0x2e8ba2e8## {-e_r-}0x00000007##
  , T {-n1-}0x00000001## {-n2-}0xffffffff## {-d-}0x0000000d##  {-e_q-}0x27627627## {-e_r-}0x00000004##
  , T {-n1-}0x00000001## {-n2-}0xffffffff## {-d-}0x00000011##  {-e_q-}0x1e1e1e1e## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0xffffffff## {-d-}0x00001000##  {-e_q-}0x001fffff## {-e_r-}0x00000fff##
  , T {-n1-}0x00000001## {-n2-}0xffffffff## {-d-}0x00010000##  {-e_q-}0x0001ffff## {-e_r-}0x0000ffff##
  , T {-n1-}0x00000001## {-n2-}0xffffffff## {-d-}0xfafaafaf##  {-e_q-}0x00000002## {-e_r-}0x0a0aa0a1##
  , T {-n1-}0x00000001## {-n2-}0xaaaaaaaa## {-d-}0x00000002##  {-e_q-}0xd5555555## {-e_r-}0x00000000##
  , T {-n1-}0x00000001## {-n2-}0xaaaaaaaa## {-d-}0x0000ffff##  {-e_q-}0x0001aaac## {-e_r-}0x00005556##
  , T {-n1-}0x00000001## {-n2-}0xaaaaaaaa## {-d-}0x0000aaaa##  {-e_q-}0x00028002## {-e_r-}0x00005556##
  , T {-n1-}0x00000001## {-n2-}0xaaaaaaaa## {-d-}0xffff0000##  {-e_q-}0x00000001## {-e_r-}0xaaabaaaa##
  , T {-n1-}0x00000001## {-n2-}0xaaaaaaaa## {-d-}0xaaaa0000##  {-e_q-}0x00000002## {-e_r-}0x5556aaaa##
  , T {-n1-}0x00000001## {-n2-}0xaaaaaaaa## {-d-}0xffffffff##  {-e_q-}0x00000001## {-e_r-}0xaaaaaaab##
  , T {-n1-}0x00000001## {-n2-}0xaaaaaaaa## {-d-}0xaaaaaaaa##  {-e_q-}0x00000002## {-e_r-}0x55555556##
  , T {-n1-}0x00000001## {-n2-}0xaaaaaaaa## {-d-}0x00000003##  {-e_q-}0x8e38e38e## {-e_r-}0x00000000##
  , T {-n1-}0x00000001## {-n2-}0xaaaaaaaa## {-d-}0x00000004##  {-e_q-}0x6aaaaaaa## {-e_r-}0x00000002##
  , T {-n1-}0x00000001## {-n2-}0xaaaaaaaa## {-d-}0x00000005##  {-e_q-}0x55555555## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0xaaaaaaaa## {-d-}0x00000006##  {-e_q-}0x471c71c7## {-e_r-}0x00000000##
  , T {-n1-}0x00000001## {-n2-}0xaaaaaaaa## {-d-}0x00000007##  {-e_q-}0x3cf3cf3c## {-e_r-}0x00000006##
  , T {-n1-}0x00000001## {-n2-}0xaaaaaaaa## {-d-}0x00000008##  {-e_q-}0x35555555## {-e_r-}0x00000002##
  , T {-n1-}0x00000001## {-n2-}0xaaaaaaaa## {-d-}0x00000009##  {-e_q-}0x2f684bda## {-e_r-}0x00000000##
  , T {-n1-}0x00000001## {-n2-}0xaaaaaaaa## {-d-}0x0000000b##  {-e_q-}0x26c9b26c## {-e_r-}0x00000006##
  , T {-n1-}0x00000001## {-n2-}0xaaaaaaaa## {-d-}0x0000000d##  {-e_q-}0x20d20d20## {-e_r-}0x0000000a##
  , T {-n1-}0x00000001## {-n2-}0xaaaaaaaa## {-d-}0x00000011##  {-e_q-}0x19191919## {-e_r-}0x00000001##
  , T {-n1-}0x00000001## {-n2-}0xaaaaaaaa## {-d-}0x00001000##  {-e_q-}0x001aaaaa## {-e_r-}0x00000aaa##
  , T {-n1-}0x00000001## {-n2-}0xaaaaaaaa## {-d-}0x00010000##  {-e_q-}0x0001aaaa## {-e_r-}0x0000aaaa##
  , T {-n1-}0x00000001## {-n2-}0xaaaaaaaa## {-d-}0xfafaafaf##  {-e_q-}0x00000001## {-e_r-}0xafaffafb##
  , T {-n1-}0x00000002## {-n2-}0x00000001## {-d-}0x0000ffff##  {-e_q-}0x00020002## {-e_r-}0x00000003##
  , T {-n1-}0x00000002## {-n2-}0x00000001## {-d-}0x0000aaaa##  {-e_q-}0x00030003## {-e_r-}0x00000003##
  , T {-n1-}0x00000002## {-n2-}0x00000001## {-d-}0xffff0000##  {-e_q-}0x00000002## {-e_r-}0x00020001##
  , T {-n1-}0x00000002## {-n2-}0x00000001## {-d-}0xaaaa0000##  {-e_q-}0x00000003## {-e_r-}0x00020001##
  , T {-n1-}0x00000002## {-n2-}0x00000001## {-d-}0xffffffff##  {-e_q-}0x00000002## {-e_r-}0x00000003##
  , T {-n1-}0x00000002## {-n2-}0x00000001## {-d-}0xaaaaaaaa##  {-e_q-}0x00000003## {-e_r-}0x00000003##
  , T {-n1-}0x00000002## {-n2-}0x00000001## {-d-}0x00000003##  {-e_q-}0xaaaaaaab## {-e_r-}0x00000000##
  , T {-n1-}0x00000002## {-n2-}0x00000001## {-d-}0x00000004##  {-e_q-}0x80000000## {-e_r-}0x00000001##
  , T {-n1-}0x00000002## {-n2-}0x00000001## {-d-}0x00000005##  {-e_q-}0x66666666## {-e_r-}0x00000003##
  , T {-n1-}0x00000002## {-n2-}0x00000001## {-d-}0x00000006##  {-e_q-}0x55555555## {-e_r-}0x00000003##
  , T {-n1-}0x00000002## {-n2-}0x00000001## {-d-}0x00000007##  {-e_q-}0x49249249## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0x00000001## {-d-}0x00000008##  {-e_q-}0x40000000## {-e_r-}0x00000001##
  , T {-n1-}0x00000002## {-n2-}0x00000001## {-d-}0x00000009##  {-e_q-}0x38e38e39## {-e_r-}0x00000000##
  , T {-n1-}0x00000002## {-n2-}0x00000001## {-d-}0x0000000b##  {-e_q-}0x2e8ba2e8## {-e_r-}0x00000009##
  , T {-n1-}0x00000002## {-n2-}0x00000001## {-d-}0x0000000d##  {-e_q-}0x27627627## {-e_r-}0x00000006##
  , T {-n1-}0x00000002## {-n2-}0x00000001## {-d-}0x00000011##  {-e_q-}0x1e1e1e1e## {-e_r-}0x00000003##
  , T {-n1-}0x00000002## {-n2-}0x00000001## {-d-}0x00001000##  {-e_q-}0x00200000## {-e_r-}0x00000001##
  , T {-n1-}0x00000002## {-n2-}0x00000001## {-d-}0x00010000##  {-e_q-}0x00020000## {-e_r-}0x00000001##
  , T {-n1-}0x00000002## {-n2-}0x00000001## {-d-}0xfafaafaf##  {-e_q-}0x00000002## {-e_r-}0x0a0aa0a3##
  , T {-n1-}0x00000002## {-n2-}0x00000002## {-d-}0x0000ffff##  {-e_q-}0x00020002## {-e_r-}0x00000004##
  , T {-n1-}0x00000002## {-n2-}0x00000002## {-d-}0x0000aaaa##  {-e_q-}0x00030003## {-e_r-}0x00000004##
  , T {-n1-}0x00000002## {-n2-}0x00000002## {-d-}0xffff0000##  {-e_q-}0x00000002## {-e_r-}0x00020002##
  , T {-n1-}0x00000002## {-n2-}0x00000002## {-d-}0xaaaa0000##  {-e_q-}0x00000003## {-e_r-}0x00020002##
  , T {-n1-}0x00000002## {-n2-}0x00000002## {-d-}0xffffffff##  {-e_q-}0x00000002## {-e_r-}0x00000004##
  , T {-n1-}0x00000002## {-n2-}0x00000002## {-d-}0xaaaaaaaa##  {-e_q-}0x00000003## {-e_r-}0x00000004##
  , T {-n1-}0x00000002## {-n2-}0x00000002## {-d-}0x00000003##  {-e_q-}0xaaaaaaab## {-e_r-}0x00000001##
  , T {-n1-}0x00000002## {-n2-}0x00000002## {-d-}0x00000004##  {-e_q-}0x80000000## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0x00000002## {-d-}0x00000005##  {-e_q-}0x66666666## {-e_r-}0x00000004##
  , T {-n1-}0x00000002## {-n2-}0x00000002## {-d-}0x00000006##  {-e_q-}0x55555555## {-e_r-}0x00000004##
  , T {-n1-}0x00000002## {-n2-}0x00000002## {-d-}0x00000007##  {-e_q-}0x49249249## {-e_r-}0x00000003##
  , T {-n1-}0x00000002## {-n2-}0x00000002## {-d-}0x00000008##  {-e_q-}0x40000000## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0x00000002## {-d-}0x00000009##  {-e_q-}0x38e38e39## {-e_r-}0x00000001##
  , T {-n1-}0x00000002## {-n2-}0x00000002## {-d-}0x0000000b##  {-e_q-}0x2e8ba2e8## {-e_r-}0x0000000a##
  , T {-n1-}0x00000002## {-n2-}0x00000002## {-d-}0x0000000d##  {-e_q-}0x27627627## {-e_r-}0x00000007##
  , T {-n1-}0x00000002## {-n2-}0x00000002## {-d-}0x00000011##  {-e_q-}0x1e1e1e1e## {-e_r-}0x00000004##
  , T {-n1-}0x00000002## {-n2-}0x00000002## {-d-}0x00001000##  {-e_q-}0x00200000## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0x00000002## {-d-}0x00010000##  {-e_q-}0x00020000## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0x00000002## {-d-}0xfafaafaf##  {-e_q-}0x00000002## {-e_r-}0x0a0aa0a4##
  , T {-n1-}0x00000002## {-n2-}0x0000ffff## {-d-}0x0000ffff##  {-e_q-}0x00020003## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0x0000ffff## {-d-}0x0000aaaa##  {-e_q-}0x00030004## {-e_r-}0x00005557##
  , T {-n1-}0x00000002## {-n2-}0x0000ffff## {-d-}0xffff0000##  {-e_q-}0x00000002## {-e_r-}0x0002ffff##
  , T {-n1-}0x00000002## {-n2-}0x0000ffff## {-d-}0xaaaa0000##  {-e_q-}0x00000003## {-e_r-}0x0002ffff##
  , T {-n1-}0x00000002## {-n2-}0x0000ffff## {-d-}0xffffffff##  {-e_q-}0x00000002## {-e_r-}0x00010001##
  , T {-n1-}0x00000002## {-n2-}0x0000ffff## {-d-}0xaaaaaaaa##  {-e_q-}0x00000003## {-e_r-}0x00010001##
  , T {-n1-}0x00000002## {-n2-}0x0000ffff## {-d-}0x00000003##  {-e_q-}0xaaaaffff## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0x0000ffff## {-d-}0x00000004##  {-e_q-}0x80003fff## {-e_r-}0x00000003##
  , T {-n1-}0x00000002## {-n2-}0x0000ffff## {-d-}0x00000005##  {-e_q-}0x66669999## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0x0000ffff## {-d-}0x00000006##  {-e_q-}0x55557fff## {-e_r-}0x00000005##
  , T {-n1-}0x00000002## {-n2-}0x0000ffff## {-d-}0x00000007##  {-e_q-}0x4924b6db## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0x0000ffff## {-d-}0x00000008##  {-e_q-}0x40001fff## {-e_r-}0x00000007##
  , T {-n1-}0x00000002## {-n2-}0x0000ffff## {-d-}0x00000009##  {-e_q-}0x38e3aaaa## {-e_r-}0x00000005##
  , T {-n1-}0x00000002## {-n2-}0x0000ffff## {-d-}0x0000000b##  {-e_q-}0x2e8bba2e## {-e_r-}0x00000005##
  , T {-n1-}0x00000002## {-n2-}0x0000ffff## {-d-}0x0000000d##  {-e_q-}0x276289d8## {-e_r-}0x00000007##
  , T {-n1-}0x00000002## {-n2-}0x0000ffff## {-d-}0x00000011##  {-e_q-}0x1e1e2d2d## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0x0000ffff## {-d-}0x00001000##  {-e_q-}0x0020000f## {-e_r-}0x00000fff##
  , T {-n1-}0x00000002## {-n2-}0x0000ffff## {-d-}0x00010000##  {-e_q-}0x00020000## {-e_r-}0x0000ffff##
  , T {-n1-}0x00000002## {-n2-}0x0000ffff## {-d-}0xfafaafaf##  {-e_q-}0x00000002## {-e_r-}0x0a0ba0a1##
  , T {-n1-}0x00000002## {-n2-}0x0000aaaa## {-d-}0x0000ffff##  {-e_q-}0x00020002## {-e_r-}0x0000aaac##
  , T {-n1-}0x00000002## {-n2-}0x0000aaaa## {-d-}0x0000aaaa##  {-e_q-}0x00030004## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0x0000aaaa## {-d-}0xffff0000##  {-e_q-}0x00000002## {-e_r-}0x0002aaaa##
  , T {-n1-}0x00000002## {-n2-}0x0000aaaa## {-d-}0xaaaa0000##  {-e_q-}0x00000003## {-e_r-}0x0002aaaa##
  , T {-n1-}0x00000002## {-n2-}0x0000aaaa## {-d-}0xffffffff##  {-e_q-}0x00000002## {-e_r-}0x0000aaac##
  , T {-n1-}0x00000002## {-n2-}0x0000aaaa## {-d-}0xaaaaaaaa##  {-e_q-}0x00000003## {-e_r-}0x0000aaac##
  , T {-n1-}0x00000002## {-n2-}0x0000aaaa## {-d-}0x00000003##  {-e_q-}0xaaaae38e## {-e_r-}0x00000000##
  , T {-n1-}0x00000002## {-n2-}0x0000aaaa## {-d-}0x00000004##  {-e_q-}0x80002aaa## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0x0000aaaa## {-d-}0x00000005##  {-e_q-}0x66668888## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0x0000aaaa## {-d-}0x00000006##  {-e_q-}0x555571c7## {-e_r-}0x00000000##
  , T {-n1-}0x00000002## {-n2-}0x0000aaaa## {-d-}0x00000007##  {-e_q-}0x4924aaaa## {-e_r-}0x00000004##
  , T {-n1-}0x00000002## {-n2-}0x0000aaaa## {-d-}0x00000008##  {-e_q-}0x40001555## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0x0000aaaa## {-d-}0x00000009##  {-e_q-}0x38e3a12f## {-e_r-}0x00000003##
  , T {-n1-}0x00000002## {-n2-}0x0000aaaa## {-d-}0x0000000b##  {-e_q-}0x2e8bb26c## {-e_r-}0x00000006##
  , T {-n1-}0x00000002## {-n2-}0x0000aaaa## {-d-}0x0000000d##  {-e_q-}0x27628348## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0x0000aaaa## {-d-}0x00000011##  {-e_q-}0x1e1e2828## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0x0000aaaa## {-d-}0x00001000##  {-e_q-}0x0020000a## {-e_r-}0x00000aaa##
  , T {-n1-}0x00000002## {-n2-}0x0000aaaa## {-d-}0x00010000##  {-e_q-}0x00020000## {-e_r-}0x0000aaaa##
  , T {-n1-}0x00000002## {-n2-}0x0000aaaa## {-d-}0xfafaafaf##  {-e_q-}0x00000002## {-e_r-}0x0a0b4b4c##
  , T {-n1-}0x00000002## {-n2-}0xffff0000## {-d-}0x0000ffff##  {-e_q-}0x00030002## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0xffff0000## {-d-}0x0000aaaa##  {-e_q-}0x00048003## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0xffff0000## {-d-}0xffff0000##  {-e_q-}0x00000003## {-e_r-}0x00020000##
  , T {-n1-}0x00000002## {-n2-}0xffff0000## {-d-}0xaaaa0000##  {-e_q-}0x00000004## {-e_r-}0x55570000##
  , T {-n1-}0x00000002## {-n2-}0xffff0000## {-d-}0xffffffff##  {-e_q-}0x00000002## {-e_r-}0xffff0002##
  , T {-n1-}0x00000002## {-n2-}0xffff0000## {-d-}0xaaaaaaaa##  {-e_q-}0x00000004## {-e_r-}0x55545558##
  , T {-n1-}0x00000002## {-n2-}0xffff0000## {-d-}0x00000003##  {-e_q-}0xffffaaaa## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0xffff0000## {-d-}0x00000004##  {-e_q-}0xbfffc000## {-e_r-}0x00000000##
  , T {-n1-}0x00000002## {-n2-}0xffff0000## {-d-}0x00000005##  {-e_q-}0x99996666## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0xffff0000## {-d-}0x00000006##  {-e_q-}0x7fffd555## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0xffff0000## {-d-}0x00000007##  {-e_q-}0x6db6b6db## {-e_r-}0x00000003##
  , T {-n1-}0x00000002## {-n2-}0xffff0000## {-d-}0x00000008##  {-e_q-}0x5fffe000## {-e_r-}0x00000000##
  , T {-n1-}0x00000002## {-n2-}0xffff0000## {-d-}0x00000009##  {-e_q-}0x555538e3## {-e_r-}0x00000005##
  , T {-n1-}0x00000002## {-n2-}0xffff0000## {-d-}0x0000000b##  {-e_q-}0x45d15d17## {-e_r-}0x00000003##
  , T {-n1-}0x00000002## {-n2-}0xffff0000## {-d-}0x0000000d##  {-e_q-}0x3b139d89## {-e_r-}0x0000000b##
  , T {-n1-}0x00000002## {-n2-}0xffff0000## {-d-}0x00000011##  {-e_q-}0x2d2d1e1e## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0xffff0000## {-d-}0x00001000##  {-e_q-}0x002ffff0## {-e_r-}0x00000000##
  , T {-n1-}0x00000002## {-n2-}0xffff0000## {-d-}0x00010000##  {-e_q-}0x0002ffff## {-e_r-}0x00000000##
  , T {-n1-}0x00000002## {-n2-}0xffff0000## {-d-}0xfafaafaf##  {-e_q-}0x00000003## {-e_r-}0x0f0ef0f3##
  , T {-n1-}0x00000002## {-n2-}0xaaaa0000## {-d-}0x0000ffff##  {-e_q-}0x0002aaac## {-e_r-}0x0000aaac##
  , T {-n1-}0x00000002## {-n2-}0xaaaa0000## {-d-}0x0000aaaa##  {-e_q-}0x00040003## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0xaaaa0000## {-d-}0xffff0000##  {-e_q-}0x00000002## {-e_r-}0xaaac0000##
  , T {-n1-}0x00000002## {-n2-}0xaaaa0000## {-d-}0xaaaa0000##  {-e_q-}0x00000004## {-e_r-}0x00020000##
  , T {-n1-}0x00000002## {-n2-}0xaaaa0000## {-d-}0xffffffff##  {-e_q-}0x00000002## {-e_r-}0xaaaa0002##
  , T {-n1-}0x00000002## {-n2-}0xaaaa0000## {-d-}0xaaaaaaaa##  {-e_q-}0x00000003## {-e_r-}0xaaaa0002##
  , T {-n1-}0x00000002## {-n2-}0xaaaa0000## {-d-}0x00000003##  {-e_q-}0xe38e0000## {-e_r-}0x00000000##
  , T {-n1-}0x00000002## {-n2-}0xaaaa0000## {-d-}0x00000004##  {-e_q-}0xaaaa8000## {-e_r-}0x00000000##
  , T {-n1-}0x00000002## {-n2-}0xaaaa0000## {-d-}0x00000005##  {-e_q-}0x88886666## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0xaaaa0000## {-d-}0x00000006##  {-e_q-}0x71c70000## {-e_r-}0x00000000##
  , T {-n1-}0x00000002## {-n2-}0xaaaa0000## {-d-}0x00000007##  {-e_q-}0x61860000## {-e_r-}0x00000000##
  , T {-n1-}0x00000002## {-n2-}0xaaaa0000## {-d-}0x00000008##  {-e_q-}0x55554000## {-e_r-}0x00000000##
  , T {-n1-}0x00000002## {-n2-}0xaaaa0000## {-d-}0x00000009##  {-e_q-}0x4bda0000## {-e_r-}0x00000000##
  , T {-n1-}0x00000002## {-n2-}0xaaaa0000## {-d-}0x0000000b##  {-e_q-}0x3e0f745d## {-e_r-}0x00000001##
  , T {-n1-}0x00000002## {-n2-}0xaaaa0000## {-d-}0x0000000d##  {-e_q-}0x34833b13## {-e_r-}0x00000009##
  , T {-n1-}0x00000002## {-n2-}0xaaaa0000## {-d-}0x00000011##  {-e_q-}0x28281e1e## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0xaaaa0000## {-d-}0x00001000##  {-e_q-}0x002aaaa0## {-e_r-}0x00000000##
  , T {-n1-}0x00000002## {-n2-}0xaaaa0000## {-d-}0x00010000##  {-e_q-}0x0002aaaa## {-e_r-}0x00000000##
  , T {-n1-}0x00000002## {-n2-}0xaaaa0000## {-d-}0xfafaafaf##  {-e_q-}0x00000002## {-e_r-}0xb4b4a0a2##
  , T {-n1-}0x00000002## {-n2-}0xffffffff## {-d-}0x0000ffff##  {-e_q-}0x00030003## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0xffffffff## {-d-}0x0000aaaa##  {-e_q-}0x00048004## {-e_r-}0x00005557##
  , T {-n1-}0x00000002## {-n2-}0xffffffff## {-d-}0xffff0000##  {-e_q-}0x00000003## {-e_r-}0x0002ffff##
  , T {-n1-}0x00000002## {-n2-}0xffffffff## {-d-}0xaaaa0000##  {-e_q-}0x00000004## {-e_r-}0x5557ffff##
  , T {-n1-}0x00000002## {-n2-}0xffffffff## {-d-}0xffffffff##  {-e_q-}0x00000003## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0xffffffff## {-d-}0xaaaaaaaa##  {-e_q-}0x00000004## {-e_r-}0x55555557##
  , T {-n1-}0x00000002## {-n2-}0xffffffff## {-d-}0x00000003##  {-e_q-}0xffffffff## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0xffffffff## {-d-}0x00000004##  {-e_q-}0xbfffffff## {-e_r-}0x00000003##
  , T {-n1-}0x00000002## {-n2-}0xffffffff## {-d-}0x00000005##  {-e_q-}0x99999999## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0xffffffff## {-d-}0x00000006##  {-e_q-}0x7fffffff## {-e_r-}0x00000005##
  , T {-n1-}0x00000002## {-n2-}0xffffffff## {-d-}0x00000007##  {-e_q-}0x6db6db6d## {-e_r-}0x00000004##
  , T {-n1-}0x00000002## {-n2-}0xffffffff## {-d-}0x00000008##  {-e_q-}0x5fffffff## {-e_r-}0x00000007##
  , T {-n1-}0x00000002## {-n2-}0xffffffff## {-d-}0x00000009##  {-e_q-}0x55555555## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0xffffffff## {-d-}0x0000000b##  {-e_q-}0x45d1745d## {-e_r-}0x00000000##
  , T {-n1-}0x00000002## {-n2-}0xffffffff## {-d-}0x0000000d##  {-e_q-}0x3b13b13b## {-e_r-}0x00000000##
  , T {-n1-}0x00000002## {-n2-}0xffffffff## {-d-}0x00000011##  {-e_q-}0x2d2d2d2d## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0xffffffff## {-d-}0x00001000##  {-e_q-}0x002fffff## {-e_r-}0x00000fff##
  , T {-n1-}0x00000002## {-n2-}0xffffffff## {-d-}0x00010000##  {-e_q-}0x0002ffff## {-e_r-}0x0000ffff##
  , T {-n1-}0x00000002## {-n2-}0xffffffff## {-d-}0xfafaafaf##  {-e_q-}0x00000003## {-e_r-}0x0f0ff0f2##
  , T {-n1-}0x00000002## {-n2-}0xaaaaaaaa## {-d-}0x0000ffff##  {-e_q-}0x0002aaad## {-e_r-}0x00005557##
  , T {-n1-}0x00000002## {-n2-}0xaaaaaaaa## {-d-}0x0000aaaa##  {-e_q-}0x00040004## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0xaaaaaaaa## {-d-}0xffff0000##  {-e_q-}0x00000002## {-e_r-}0xaaacaaaa##
  , T {-n1-}0x00000002## {-n2-}0xaaaaaaaa## {-d-}0xaaaa0000##  {-e_q-}0x00000004## {-e_r-}0x0002aaaa##
  , T {-n1-}0x00000002## {-n2-}0xaaaaaaaa## {-d-}0xffffffff##  {-e_q-}0x00000002## {-e_r-}0xaaaaaaac##
  , T {-n1-}0x00000002## {-n2-}0xaaaaaaaa## {-d-}0xaaaaaaaa##  {-e_q-}0x00000004## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0xaaaaaaaa## {-d-}0x00000003##  {-e_q-}0xe38e38e3## {-e_r-}0x00000001##
  , T {-n1-}0x00000002## {-n2-}0xaaaaaaaa## {-d-}0x00000004##  {-e_q-}0xaaaaaaaa## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0xaaaaaaaa## {-d-}0x00000005##  {-e_q-}0x88888888## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0xaaaaaaaa## {-d-}0x00000006##  {-e_q-}0x71c71c71## {-e_r-}0x00000004##
  , T {-n1-}0x00000002## {-n2-}0xaaaaaaaa## {-d-}0x00000007##  {-e_q-}0x61861861## {-e_r-}0x00000003##
  , T {-n1-}0x00000002## {-n2-}0xaaaaaaaa## {-d-}0x00000008##  {-e_q-}0x55555555## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0xaaaaaaaa## {-d-}0x00000009##  {-e_q-}0x4bda12f6## {-e_r-}0x00000004##
  , T {-n1-}0x00000002## {-n2-}0xaaaaaaaa## {-d-}0x0000000b##  {-e_q-}0x3e0f83e0## {-e_r-}0x0000000a##
  , T {-n1-}0x00000002## {-n2-}0xaaaaaaaa## {-d-}0x0000000d##  {-e_q-}0x34834834## {-e_r-}0x00000006##
  , T {-n1-}0x00000002## {-n2-}0xaaaaaaaa## {-d-}0x00000011##  {-e_q-}0x28282828## {-e_r-}0x00000002##
  , T {-n1-}0x00000002## {-n2-}0xaaaaaaaa## {-d-}0x00001000##  {-e_q-}0x002aaaaa## {-e_r-}0x00000aaa##
  , T {-n1-}0x00000002## {-n2-}0xaaaaaaaa## {-d-}0x00010000##  {-e_q-}0x0002aaaa## {-e_r-}0x0000aaaa##
  , T {-n1-}0x00000002## {-n2-}0xaaaaaaaa## {-d-}0xfafaafaf##  {-e_q-}0x00000002## {-e_r-}0xb4b54b4c##
  , T {-n1-}0x0000ffff## {-n2-}0x00000001## {-d-}0xffff0000##  {-e_q-}0x00010000## {-e_r-}0x00000001##
  , T {-n1-}0x0000ffff## {-n2-}0x00000001## {-d-}0xaaaa0000##  {-e_q-}0x00018000## {-e_r-}0x00000001##
  , T {-n1-}0x0000ffff## {-n2-}0x00000001## {-d-}0xffffffff##  {-e_q-}0x0000ffff## {-e_r-}0x00010000##
  , T {-n1-}0x0000ffff## {-n2-}0x00000001## {-d-}0xaaaaaaaa##  {-e_q-}0x00017ffe## {-e_r-}0x55565555##
  , T {-n1-}0x0000ffff## {-n2-}0x00000001## {-d-}0x00010000##  {-e_q-}0xffff0000## {-e_r-}0x00000001##
  , T {-n1-}0x0000ffff## {-n2-}0x00000001## {-d-}0xfafaafaf##  {-e_q-}0x0001051e## {-e_r-}0x0181fe7f##
  , T {-n1-}0x0000ffff## {-n2-}0x00000002## {-d-}0xffff0000##  {-e_q-}0x00010000## {-e_r-}0x00000002##
  , T {-n1-}0x0000ffff## {-n2-}0x00000002## {-d-}0xaaaa0000##  {-e_q-}0x00018000## {-e_r-}0x00000002##
  , T {-n1-}0x0000ffff## {-n2-}0x00000002## {-d-}0xffffffff##  {-e_q-}0x0000ffff## {-e_r-}0x00010001##
  , T {-n1-}0x0000ffff## {-n2-}0x00000002## {-d-}0xaaaaaaaa##  {-e_q-}0x00017ffe## {-e_r-}0x55565556##
  , T {-n1-}0x0000ffff## {-n2-}0x00000002## {-d-}0x00010000##  {-e_q-}0xffff0000## {-e_r-}0x00000002##
  , T {-n1-}0x0000ffff## {-n2-}0x00000002## {-d-}0xfafaafaf##  {-e_q-}0x0001051e## {-e_r-}0x0181fe80##
  , T {-n1-}0x0000ffff## {-n2-}0x0000ffff## {-d-}0xffff0000##  {-e_q-}0x00010000## {-e_r-}0x0000ffff##
  , T {-n1-}0x0000ffff## {-n2-}0x0000ffff## {-d-}0xaaaa0000##  {-e_q-}0x00018000## {-e_r-}0x0000ffff##
  , T {-n1-}0x0000ffff## {-n2-}0x0000ffff## {-d-}0xffffffff##  {-e_q-}0x0000ffff## {-e_r-}0x0001fffe##
  , T {-n1-}0x0000ffff## {-n2-}0x0000ffff## {-d-}0xaaaaaaaa##  {-e_q-}0x00017ffe## {-e_r-}0x55575553##
  , T {-n1-}0x0000ffff## {-n2-}0x0000ffff## {-d-}0x00010000##  {-e_q-}0xffff0000## {-e_r-}0x0000ffff##
  , T {-n1-}0x0000ffff## {-n2-}0x0000ffff## {-d-}0xfafaafaf##  {-e_q-}0x0001051e## {-e_r-}0x0182fe7d##
  , T {-n1-}0x0000ffff## {-n2-}0x0000aaaa## {-d-}0xffff0000##  {-e_q-}0x00010000## {-e_r-}0x0000aaaa##
  , T {-n1-}0x0000ffff## {-n2-}0x0000aaaa## {-d-}0xaaaa0000##  {-e_q-}0x00018000## {-e_r-}0x0000aaaa##
  , T {-n1-}0x0000ffff## {-n2-}0x0000aaaa## {-d-}0xffffffff##  {-e_q-}0x0000ffff## {-e_r-}0x0001aaa9##
  , T {-n1-}0x0000ffff## {-n2-}0x0000aaaa## {-d-}0xaaaaaaaa##  {-e_q-}0x00017ffe## {-e_r-}0x5556fffe##
  , T {-n1-}0x0000ffff## {-n2-}0x0000aaaa## {-d-}0x00010000##  {-e_q-}0xffff0000## {-e_r-}0x0000aaaa##
  , T {-n1-}0x0000ffff## {-n2-}0x0000aaaa## {-d-}0xfafaafaf##  {-e_q-}0x0001051e## {-e_r-}0x0182a928##
  , T {-n1-}0x0000ffff## {-n2-}0xffff0000## {-d-}0xffff0000##  {-e_q-}0x00010001## {-e_r-}0x00000000##
  , T {-n1-}0x0000ffff## {-n2-}0xffff0000## {-d-}0xaaaa0000##  {-e_q-}0x00018001## {-e_r-}0x55550000##
  , T {-n1-}0x0000ffff## {-n2-}0xffff0000## {-d-}0xffffffff##  {-e_q-}0x00010000## {-e_r-}0x00000000##
  , T {-n1-}0x0000ffff## {-n2-}0xffff0000## {-d-}0xaaaaaaaa##  {-e_q-}0x00018000## {-e_r-}0x00000000##
  , T {-n1-}0x0000ffff## {-n2-}0xffff0000## {-d-}0x00010000##  {-e_q-}0xffffffff## {-e_r-}0x00000000##
  , T {-n1-}0x0000ffff## {-n2-}0xffff0000## {-d-}0xfafaafaf##  {-e_q-}0x0001051f## {-e_r-}0x06864ecf##
  , T {-n1-}0x0000ffff## {-n2-}0xaaaa0000## {-d-}0xffff0000##  {-e_q-}0x00010000## {-e_r-}0xaaaa0000##
  , T {-n1-}0x0000ffff## {-n2-}0xaaaa0000## {-d-}0xaaaa0000##  {-e_q-}0x00018001## {-e_r-}0x00000000##
  , T {-n1-}0x0000ffff## {-n2-}0xaaaa0000## {-d-}0xffffffff##  {-e_q-}0x0000ffff## {-e_r-}0xaaaaffff##
  , T {-n1-}0x0000ffff## {-n2-}0xaaaa0000## {-d-}0xaaaaaaaa##  {-e_q-}0x00017fff## {-e_r-}0x5555aaaa##
  , T {-n1-}0x0000ffff## {-n2-}0xaaaa0000## {-d-}0x00010000##  {-e_q-}0xffffaaaa## {-e_r-}0x00000000##
  , T {-n1-}0x0000ffff## {-n2-}0xaaaa0000## {-d-}0xfafaafaf##  {-e_q-}0x0001051e## {-e_r-}0xac2bfe7e##
  , T {-n1-}0x0000ffff## {-n2-}0xffffffff## {-d-}0xffff0000##  {-e_q-}0x00010001## {-e_r-}0x0000ffff##
  , T {-n1-}0x0000ffff## {-n2-}0xffffffff## {-d-}0xaaaa0000##  {-e_q-}0x00018001## {-e_r-}0x5555ffff##
  , T {-n1-}0x0000ffff## {-n2-}0xffffffff## {-d-}0xffffffff##  {-e_q-}0x00010000## {-e_r-}0x0000ffff##
  , T {-n1-}0x0000ffff## {-n2-}0xffffffff## {-d-}0xaaaaaaaa##  {-e_q-}0x00018000## {-e_r-}0x0000ffff##
  , T {-n1-}0x0000ffff## {-n2-}0xffffffff## {-d-}0x00010000##  {-e_q-}0xffffffff## {-e_r-}0x0000ffff##
  , T {-n1-}0x0000ffff## {-n2-}0xffffffff## {-d-}0xfafaafaf##  {-e_q-}0x0001051f## {-e_r-}0x06874ece##
  , T {-n1-}0x0000ffff## {-n2-}0xaaaaaaaa## {-d-}0xffff0000##  {-e_q-}0x00010000## {-e_r-}0xaaaaaaaa##
  , T {-n1-}0x0000ffff## {-n2-}0xaaaaaaaa## {-d-}0xaaaa0000##  {-e_q-}0x00018001## {-e_r-}0x0000aaaa##
  , T {-n1-}0x0000ffff## {-n2-}0xaaaaaaaa## {-d-}0xffffffff##  {-e_q-}0x0000ffff## {-e_r-}0xaaabaaa9##
  , T {-n1-}0x0000ffff## {-n2-}0xaaaaaaaa## {-d-}0xaaaaaaaa##  {-e_q-}0x00017fff## {-e_r-}0x55565554##
  , T {-n1-}0x0000ffff## {-n2-}0xaaaaaaaa## {-d-}0x00010000##  {-e_q-}0xffffaaaa## {-e_r-}0x0000aaaa##
  , T {-n1-}0x0000ffff## {-n2-}0xaaaaaaaa## {-d-}0xfafaafaf##  {-e_q-}0x0001051e## {-e_r-}0xac2ca928##
  , T {-n1-}0x0000aaaa## {-n2-}0x00000001## {-d-}0x0000ffff##  {-e_q-}0xaaaaaaaa## {-e_r-}0x0000aaab##
  , T {-n1-}0x0000aaaa## {-n2-}0x00000001## {-d-}0xffff0000##  {-e_q-}0x0000aaaa## {-e_r-}0xaaaa0001##
  , T {-n1-}0x0000aaaa## {-n2-}0x00000001## {-d-}0xaaaa0000##  {-e_q-}0x00010000## {-e_r-}0x00000001##
  , T {-n1-}0x0000aaaa## {-n2-}0x00000001## {-d-}0xffffffff##  {-e_q-}0x0000aaaa## {-e_r-}0x0000aaab##
  , T {-n1-}0x0000aaaa## {-n2-}0x00000001## {-d-}0xaaaaaaaa##  {-e_q-}0x0000ffff## {-e_r-}0x0000aaab##
  , T {-n1-}0x0000aaaa## {-n2-}0x00000001## {-d-}0x00010000##  {-e_q-}0xaaaa0000## {-e_r-}0x00000001##
  , T {-n1-}0x0000aaaa## {-n2-}0x00000001## {-d-}0xfafaafaf##  {-e_q-}0x0000ae14## {-e_r-}0x01015455##
  , T {-n1-}0x0000aaaa## {-n2-}0x00000002## {-d-}0x0000ffff##  {-e_q-}0xaaaaaaaa## {-e_r-}0x0000aaac##
  , T {-n1-}0x0000aaaa## {-n2-}0x00000002## {-d-}0xffff0000##  {-e_q-}0x0000aaaa## {-e_r-}0xaaaa0002##
  , T {-n1-}0x0000aaaa## {-n2-}0x00000002## {-d-}0xaaaa0000##  {-e_q-}0x00010000## {-e_r-}0x00000002##
  , T {-n1-}0x0000aaaa## {-n2-}0x00000002## {-d-}0xffffffff##  {-e_q-}0x0000aaaa## {-e_r-}0x0000aaac##
  , T {-n1-}0x0000aaaa## {-n2-}0x00000002## {-d-}0xaaaaaaaa##  {-e_q-}0x0000ffff## {-e_r-}0x0000aaac##
  , T {-n1-}0x0000aaaa## {-n2-}0x00000002## {-d-}0x00010000##  {-e_q-}0xaaaa0000## {-e_r-}0x00000002##
  , T {-n1-}0x0000aaaa## {-n2-}0x00000002## {-d-}0xfafaafaf##  {-e_q-}0x0000ae14## {-e_r-}0x01015456##
  , T {-n1-}0x0000aaaa## {-n2-}0x0000ffff## {-d-}0x0000ffff##  {-e_q-}0xaaaaaaab## {-e_r-}0x0000aaaa##
  , T {-n1-}0x0000aaaa## {-n2-}0x0000ffff## {-d-}0xffff0000##  {-e_q-}0x0000aaaa## {-e_r-}0xaaaaffff##
  , T {-n1-}0x0000aaaa## {-n2-}0x0000ffff## {-d-}0xaaaa0000##  {-e_q-}0x00010000## {-e_r-}0x0000ffff##
  , T {-n1-}0x0000aaaa## {-n2-}0x0000ffff## {-d-}0xffffffff##  {-e_q-}0x0000aaaa## {-e_r-}0x0001aaa9##
  , T {-n1-}0x0000aaaa## {-n2-}0x0000ffff## {-d-}0xaaaaaaaa##  {-e_q-}0x0000ffff## {-e_r-}0x0001aaa9##
  , T {-n1-}0x0000aaaa## {-n2-}0x0000ffff## {-d-}0x00010000##  {-e_q-}0xaaaa0000## {-e_r-}0x0000ffff##
  , T {-n1-}0x0000aaaa## {-n2-}0x0000ffff## {-d-}0xfafaafaf##  {-e_q-}0x0000ae14## {-e_r-}0x01025453##
  , T {-n1-}0x0000aaaa## {-n2-}0x0000aaaa## {-d-}0x0000ffff##  {-e_q-}0xaaaaaaab## {-e_r-}0x00005555##
  , T {-n1-}0x0000aaaa## {-n2-}0x0000aaaa## {-d-}0xffff0000##  {-e_q-}0x0000aaaa## {-e_r-}0xaaaaaaaa##
  , T {-n1-}0x0000aaaa## {-n2-}0x0000aaaa## {-d-}0xaaaa0000##  {-e_q-}0x00010000## {-e_r-}0x0000aaaa##
  , T {-n1-}0x0000aaaa## {-n2-}0x0000aaaa## {-d-}0xffffffff##  {-e_q-}0x0000aaaa## {-e_r-}0x00015554##
  , T {-n1-}0x0000aaaa## {-n2-}0x0000aaaa## {-d-}0xaaaaaaaa##  {-e_q-}0x0000ffff## {-e_r-}0x00015554##
  , T {-n1-}0x0000aaaa## {-n2-}0x0000aaaa## {-d-}0x00010000##  {-e_q-}0xaaaa0000## {-e_r-}0x0000aaaa##
  , T {-n1-}0x0000aaaa## {-n2-}0x0000aaaa## {-d-}0xfafaafaf##  {-e_q-}0x0000ae14## {-e_r-}0x0101fefe##
  , T {-n1-}0x0000aaaa## {-n2-}0xffff0000## {-d-}0x0000ffff##  {-e_q-}0xaaabaaaa## {-e_r-}0x0000aaaa##
  , T {-n1-}0x0000aaaa## {-n2-}0xffff0000## {-d-}0xffff0000##  {-e_q-}0x0000aaab## {-e_r-}0xaaaa0000##
  , T {-n1-}0x0000aaaa## {-n2-}0xffff0000## {-d-}0xaaaa0000##  {-e_q-}0x00010001## {-e_r-}0x55550000##
  , T {-n1-}0x0000aaaa## {-n2-}0xffff0000## {-d-}0xffffffff##  {-e_q-}0x0000aaaa## {-e_r-}0xffffaaaa##
  , T {-n1-}0x0000aaaa## {-n2-}0xffff0000## {-d-}0xaaaaaaaa##  {-e_q-}0x00010000## {-e_r-}0x55550000##
  , T {-n1-}0x0000aaaa## {-n2-}0xffff0000## {-d-}0x00010000##  {-e_q-}0xaaaaffff## {-e_r-}0x00000000##
  , T {-n1-}0x0000aaaa## {-n2-}0xffff0000## {-d-}0xfafaafaf##  {-e_q-}0x0000ae15## {-e_r-}0x0605a4a5##
  , T {-n1-}0x0000aaaa## {-n2-}0xaaaa0000## {-d-}0x0000ffff##  {-e_q-}0xaaab5555## {-e_r-}0x00005555##
  , T {-n1-}0x0000aaaa## {-n2-}0xaaaa0000## {-d-}0xffff0000##  {-e_q-}0x0000aaab## {-e_r-}0x55550000##
  , T {-n1-}0x0000aaaa## {-n2-}0xaaaa0000## {-d-}0xaaaa0000##  {-e_q-}0x00010001## {-e_r-}0x00000000##
  , T {-n1-}0x0000aaaa## {-n2-}0xaaaa0000## {-d-}0xffffffff##  {-e_q-}0x0000aaaa## {-e_r-}0xaaaaaaaa##
  , T {-n1-}0x0000aaaa## {-n2-}0xaaaa0000## {-d-}0xaaaaaaaa##  {-e_q-}0x00010000## {-e_r-}0x00000000##
  , T {-n1-}0x0000aaaa## {-n2-}0xaaaa0000## {-d-}0x00010000##  {-e_q-}0xaaaaaaaa## {-e_r-}0x00000000##
  , T {-n1-}0x0000aaaa## {-n2-}0xaaaa0000## {-d-}0xfafaafaf##  {-e_q-}0x0000ae14## {-e_r-}0xabab5454##
  , T {-n1-}0x0000aaaa## {-n2-}0xffffffff## {-d-}0x0000ffff##  {-e_q-}0xaaabaaab## {-e_r-}0x0000aaaa##
  , T {-n1-}0x0000aaaa## {-n2-}0xffffffff## {-d-}0xffff0000##  {-e_q-}0x0000aaab## {-e_r-}0xaaaaffff##
  , T {-n1-}0x0000aaaa## {-n2-}0xffffffff## {-d-}0xaaaa0000##  {-e_q-}0x00010001## {-e_r-}0x5555ffff##
  , T {-n1-}0x0000aaaa## {-n2-}0xffffffff## {-d-}0xffffffff##  {-e_q-}0x0000aaab## {-e_r-}0x0000aaaa##
  , T {-n1-}0x0000aaaa## {-n2-}0xffffffff## {-d-}0xaaaaaaaa##  {-e_q-}0x00010000## {-e_r-}0x5555ffff##
  , T {-n1-}0x0000aaaa## {-n2-}0xffffffff## {-d-}0x00010000##  {-e_q-}0xaaaaffff## {-e_r-}0x0000ffff##
  , T {-n1-}0x0000aaaa## {-n2-}0xffffffff## {-d-}0xfafaafaf##  {-e_q-}0x0000ae15## {-e_r-}0x0606a4a4##
  , T {-n1-}0x0000aaaa## {-n2-}0xaaaaaaaa## {-d-}0x0000ffff##  {-e_q-}0xaaab5556## {-e_r-}0x00000000##
  , T {-n1-}0x0000aaaa## {-n2-}0xaaaaaaaa## {-d-}0xffff0000##  {-e_q-}0x0000aaab## {-e_r-}0x5555aaaa##
  , T {-n1-}0x0000aaaa## {-n2-}0xaaaaaaaa## {-d-}0xaaaa0000##  {-e_q-}0x00010001## {-e_r-}0x0000aaaa##
  , T {-n1-}0x0000aaaa## {-n2-}0xaaaaaaaa## {-d-}0xffffffff##  {-e_q-}0x0000aaaa## {-e_r-}0xaaab5554##
  , T {-n1-}0x0000aaaa## {-n2-}0xaaaaaaaa## {-d-}0xaaaaaaaa##  {-e_q-}0x00010000## {-e_r-}0x0000aaaa##
  , T {-n1-}0x0000aaaa## {-n2-}0xaaaaaaaa## {-d-}0x00010000##  {-e_q-}0xaaaaaaaa## {-e_r-}0x0000aaaa##
  , T {-n1-}0x0000aaaa## {-n2-}0xaaaaaaaa## {-d-}0xfafaafaf##  {-e_q-}0x0000ae14## {-e_r-}0xababfefe##
  , T {-n1-}0xffff0000## {-n2-}0x00000001## {-d-}0xffffffff##  {-e_q-}0xffff0000## {-e_r-}0xffff0001##
  , T {-n1-}0xffff0000## {-n2-}0x00000002## {-d-}0xffffffff##  {-e_q-}0xffff0000## {-e_r-}0xffff0002##
  , T {-n1-}0xffff0000## {-n2-}0x0000ffff## {-d-}0xffffffff##  {-e_q-}0xffff0001## {-e_r-}0x00000000##
  , T {-n1-}0xffff0000## {-n2-}0x0000aaaa## {-d-}0xffffffff##  {-e_q-}0xffff0000## {-e_r-}0xffffaaaa##
  , T {-n1-}0xffff0000## {-n2-}0xffff0000## {-d-}0xffffffff##  {-e_q-}0xffff0001## {-e_r-}0xfffe0001##
  , T {-n1-}0xffff0000## {-n2-}0xaaaa0000## {-d-}0xffffffff##  {-e_q-}0xffff0001## {-e_r-}0xaaa90001##
  , T {-n1-}0xffff0000## {-n2-}0xffffffff## {-d-}0xffffffff##  {-e_q-}0xffff0001## {-e_r-}0xffff0000##
  , T {-n1-}0xffff0000## {-n2-}0xaaaaaaaa## {-d-}0xffffffff##  {-e_q-}0xffff0001## {-e_r-}0xaaa9aaab##
  , T {-n1-}0xaaaa0000## {-n2-}0x00000001## {-d-}0xffff0000##  {-e_q-}0xaaaaaaaa## {-e_r-}0xaaaa0001##
  , T {-n1-}0xaaaa0000## {-n2-}0x00000001## {-d-}0xffffffff##  {-e_q-}0xaaaa0000## {-e_r-}0xaaaa0001##
  , T {-n1-}0xaaaa0000## {-n2-}0x00000001## {-d-}0xaaaaaaaa##  {-e_q-}0xffff0000## {-e_r-}0xaaaa0001##
  , T {-n1-}0xaaaa0000## {-n2-}0x00000001## {-d-}0xfafaafaf##  {-e_q-}0xae140106## {-e_r-}0x77c432e7##
  , T {-n1-}0xaaaa0000## {-n2-}0x00000002## {-d-}0xffff0000##  {-e_q-}0xaaaaaaaa## {-e_r-}0xaaaa0002##
  , T {-n1-}0xaaaa0000## {-n2-}0x00000002## {-d-}0xffffffff##  {-e_q-}0xaaaa0000## {-e_r-}0xaaaa0002##
  , T {-n1-}0xaaaa0000## {-n2-}0x00000002## {-d-}0xaaaaaaaa##  {-e_q-}0xffff0000## {-e_r-}0xaaaa0002##
  , T {-n1-}0xaaaa0000## {-n2-}0x00000002## {-d-}0xfafaafaf##  {-e_q-}0xae140106## {-e_r-}0x77c432e8##
  , T {-n1-}0xaaaa0000## {-n2-}0x0000ffff## {-d-}0xffff0000##  {-e_q-}0xaaaaaaaa## {-e_r-}0xaaaaffff##
  , T {-n1-}0xaaaa0000## {-n2-}0x0000ffff## {-d-}0xffffffff##  {-e_q-}0xaaaa0000## {-e_r-}0xaaaaffff##
  , T {-n1-}0xaaaa0000## {-n2-}0x0000ffff## {-d-}0xaaaaaaaa##  {-e_q-}0xffff0001## {-e_r-}0x00005555##
  , T {-n1-}0xaaaa0000## {-n2-}0x0000ffff## {-d-}0xfafaafaf##  {-e_q-}0xae140106## {-e_r-}0x77c532e5##
  , T {-n1-}0xaaaa0000## {-n2-}0x0000aaaa## {-d-}0xffff0000##  {-e_q-}0xaaaaaaaa## {-e_r-}0xaaaaaaaa##
  , T {-n1-}0xaaaa0000## {-n2-}0x0000aaaa## {-d-}0xffffffff##  {-e_q-}0xaaaa0000## {-e_r-}0xaaaaaaaa##
  , T {-n1-}0xaaaa0000## {-n2-}0x0000aaaa## {-d-}0xaaaaaaaa##  {-e_q-}0xffff0001## {-e_r-}0x00000000##
  , T {-n1-}0xaaaa0000## {-n2-}0x0000aaaa## {-d-}0xfafaafaf##  {-e_q-}0xae140106## {-e_r-}0x77c4dd90##
  , T {-n1-}0xaaaa0000## {-n2-}0xffff0000## {-d-}0xffff0000##  {-e_q-}0xaaaaaaab## {-e_r-}0xaaaa0000##
  , T {-n1-}0xaaaa0000## {-n2-}0xffff0000## {-d-}0xffffffff##  {-e_q-}0xaaaa0001## {-e_r-}0xaaa90001##
  , T {-n1-}0xaaaa0000## {-n2-}0xffff0000## {-d-}0xaaaaaaaa##  {-e_q-}0xffff0002## {-e_r-}0x5553aaac##
  , T {-n1-}0xaaaa0000## {-n2-}0xffff0000## {-d-}0xfafaafaf##  {-e_q-}0xae140107## {-e_r-}0x7cc88337##
  , T {-n1-}0xaaaa0000## {-n2-}0xaaaa0000## {-d-}0xffff0000##  {-e_q-}0xaaaaaaab## {-e_r-}0x55550000##
  , T {-n1-}0xaaaa0000## {-n2-}0xaaaa0000## {-d-}0xffffffff##  {-e_q-}0xaaaa0001## {-e_r-}0x55540001##
  , T {-n1-}0xaaaa0000## {-n2-}0xaaaa0000## {-d-}0xaaaaaaaa##  {-e_q-}0xffff0001## {-e_r-}0xaaa95556##
  , T {-n1-}0xaaaa0000## {-n2-}0xaaaa0000## {-d-}0xfafaafaf##  {-e_q-}0xae140107## {-e_r-}0x27738337##
  , T {-n1-}0xaaaa0000## {-n2-}0xffffffff## {-d-}0xffff0000##  {-e_q-}0xaaaaaaab## {-e_r-}0xaaaaffff##
  , T {-n1-}0xaaaa0000## {-n2-}0xffffffff## {-d-}0xffffffff##  {-e_q-}0xaaaa0001## {-e_r-}0xaaaa0000##
  , T {-n1-}0xaaaa0000## {-n2-}0xffffffff## {-d-}0xaaaaaaaa##  {-e_q-}0xffff0002## {-e_r-}0x5554aaab##
  , T {-n1-}0xaaaa0000## {-n2-}0xffffffff## {-d-}0xfafaafaf##  {-e_q-}0xae140107## {-e_r-}0x7cc98336##
  , T {-n1-}0xaaaa0000## {-n2-}0xaaaaaaaa## {-d-}0xffff0000##  {-e_q-}0xaaaaaaab## {-e_r-}0x5555aaaa##
  , T {-n1-}0xaaaa0000## {-n2-}0xaaaaaaaa## {-d-}0xffffffff##  {-e_q-}0xaaaa0001## {-e_r-}0x5554aaab##
  , T {-n1-}0xaaaa0000## {-n2-}0xaaaaaaaa## {-d-}0xaaaaaaaa##  {-e_q-}0xffff0001## {-e_r-}0xaaaa0000##
  , T {-n1-}0xaaaa0000## {-n2-}0xaaaaaaaa## {-d-}0xfafaafaf##  {-e_q-}0xae140107## {-e_r-}0x27742de1##
  , T {-n1-}0xaaaaaaaa## {-n2-}0x00000001## {-d-}0xffff0000##  {-e_q-}0xaaab5555## {-e_r-}0x55550001##
  , T {-n1-}0xaaaaaaaa## {-n2-}0x00000001## {-d-}0xffffffff##  {-e_q-}0xaaaaaaaa## {-e_r-}0xaaaaaaab##
  , T {-n1-}0xaaaaaaaa## {-n2-}0x00000001## {-d-}0xfafaafaf##  {-e_q-}0xae14af1a## {-e_r-}0x78c5873b##
  , T {-n1-}0xaaaaaaaa## {-n2-}0x00000002## {-d-}0xffff0000##  {-e_q-}0xaaab5555## {-e_r-}0x55550002##
  , T {-n1-}0xaaaaaaaa## {-n2-}0x00000002## {-d-}0xffffffff##  {-e_q-}0xaaaaaaaa## {-e_r-}0xaaaaaaac##
  , T {-n1-}0xaaaaaaaa## {-n2-}0x00000002## {-d-}0xfafaafaf##  {-e_q-}0xae14af1a## {-e_r-}0x78c5873c##
  , T {-n1-}0xaaaaaaaa## {-n2-}0x0000ffff## {-d-}0xffff0000##  {-e_q-}0xaaab5555## {-e_r-}0x5555ffff##
  , T {-n1-}0xaaaaaaaa## {-n2-}0x0000ffff## {-d-}0xffffffff##  {-e_q-}0xaaaaaaaa## {-e_r-}0xaaabaaa9##
  , T {-n1-}0xaaaaaaaa## {-n2-}0x0000ffff## {-d-}0xfafaafaf##  {-e_q-}0xae14af1a## {-e_r-}0x78c68739##
  , T {-n1-}0xaaaaaaaa## {-n2-}0x0000aaaa## {-d-}0xffff0000##  {-e_q-}0xaaab5555## {-e_r-}0x5555aaaa##
  , T {-n1-}0xaaaaaaaa## {-n2-}0x0000aaaa## {-d-}0xffffffff##  {-e_q-}0xaaaaaaaa## {-e_r-}0xaaab5554##
  , T {-n1-}0xaaaaaaaa## {-n2-}0x0000aaaa## {-d-}0xfafaafaf##  {-e_q-}0xae14af1a## {-e_r-}0x78c631e4##
  , T {-n1-}0xaaaaaaaa## {-n2-}0xffff0000## {-d-}0xffff0000##  {-e_q-}0xaaab5556## {-e_r-}0x55550000##
  , T {-n1-}0xaaaaaaaa## {-n2-}0xffff0000## {-d-}0xffffffff##  {-e_q-}0xaaaaaaab## {-e_r-}0xaaa9aaab##
  , T {-n1-}0xaaaaaaaa## {-n2-}0xffff0000## {-d-}0xfafaafaf##  {-e_q-}0xae14af1b## {-e_r-}0x7dc9d78b##
  , T {-n1-}0xaaaaaaaa## {-n2-}0xaaaa0000## {-d-}0xffff0000##  {-e_q-}0xaaab5556## {-e_r-}0x00000000##
  , T {-n1-}0xaaaaaaaa## {-n2-}0xaaaa0000## {-d-}0xffffffff##  {-e_q-}0xaaaaaaab## {-e_r-}0x5554aaab##
  , T {-n1-}0xaaaaaaaa## {-n2-}0xaaaa0000## {-d-}0xfafaafaf##  {-e_q-}0xae14af1b## {-e_r-}0x2874d78b##
  , T {-n1-}0xaaaaaaaa## {-n2-}0xffffffff## {-d-}0xffff0000##  {-e_q-}0xaaab5556## {-e_r-}0x5555ffff##
  , T {-n1-}0xaaaaaaaa## {-n2-}0xffffffff## {-d-}0xffffffff##  {-e_q-}0xaaaaaaab## {-e_r-}0xaaaaaaaa##
  , T {-n1-}0xaaaaaaaa## {-n2-}0xffffffff## {-d-}0xfafaafaf##  {-e_q-}0xae14af1b## {-e_r-}0x7dcad78a##
  , T {-n1-}0xaaaaaaaa## {-n2-}0xaaaaaaaa## {-d-}0xffff0000##  {-e_q-}0xaaab5556## {-e_r-}0x0000aaaa##
  , T {-n1-}0xaaaaaaaa## {-n2-}0xaaaaaaaa## {-d-}0xffffffff##  {-e_q-}0xaaaaaaab## {-e_r-}0x55555555##
  , T {-n1-}0xaaaaaaaa## {-n2-}0xaaaaaaaa## {-d-}0xfafaafaf##  {-e_q-}0xae14af1b## {-e_r-}0x28758235##
#elif WORD_SIZE_IN_BITS == 64
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000000000000## {-d-}0x0000000080000000##  {-e_q-}0x0000000200000000## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000000000000## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000000000000## {-d-}0x0000000100000000##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000000000000## {-d-}0x0000000100000001##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000000000000## {-d-}0x0000000180000000##  {-e_q-}0x00000000AAAAAAAA## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000000000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x0000000080000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000000000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000000000002## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000000000000## {-d-}0x8000000000000001##  {-e_q-}0x0000000000000001## {-e_r-}0x7FFFFFFFFFFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000000000000## {-d-}0x8000000080000000##  {-e_q-}0x0000000000000001## {-e_r-}0x7FFFFFFF80000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000000000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x0000000000000001## {-e_r-}0x7FFFFFFF00000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000000000001## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000000000001## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000000000001## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000000000001## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000000000001## {-d-}0x0000000080000000##  {-e_q-}0x0000000200000000## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000000000001## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000000000002##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000000000001## {-d-}0x0000000100000000##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000000000001## {-d-}0x0000000100000001##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x0000000000000002##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000000000001## {-d-}0x0000000180000000##  {-e_q-}0x00000000AAAAAAAA## {-e_r-}0x0000000100000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000000000001## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x0000000080000000## {-e_r-}0x0000000080000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000000000001## {-d-}0x8000000000000000##  {-e_q-}0x0000000000000002## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000000000001## {-d-}0x8000000000000001##  {-e_q-}0x0000000000000001## {-e_r-}0x8000000000000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000000000001## {-d-}0x8000000080000000##  {-e_q-}0x0000000000000001## {-e_r-}0x7FFFFFFF80000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000000000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x0000000000000001## {-e_r-}0x7FFFFFFF00000002##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000000000001## {-e_r-}0x0000000100000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000000000001## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000000000001## {-e_r-}0x0000000080000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000000000001## {-e_r-}0x0000000000000002##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000080000000## {-d-}0x0000000080000000##  {-e_q-}0x0000000200000001## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000080000000## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000080000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000080000000## {-d-}0x0000000100000000##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000080000000## {-d-}0x0000000100000001##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x0000000080000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000080000000## {-d-}0x0000000180000000##  {-e_q-}0x00000000AAAAAAAB## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000080000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x0000000080000000## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000080000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000000000002## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000080000000## {-d-}0x8000000000000001##  {-e_q-}0x0000000000000002## {-e_r-}0x000000007FFFFFFE##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000080000000## {-d-}0x8000000080000000##  {-e_q-}0x0000000000000001## {-e_r-}0x8000000000000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000080000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x0000000000000001## {-e_r-}0x7FFFFFFF80000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000000000001## {-e_r-}0x0000000180000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000000000001## {-e_r-}0x000000017FFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000000000001## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000000000001## {-e_r-}0x0000000080000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x00000000FFFFFFFF## {-d-}0x0000000080000000##  {-e_q-}0x0000000200000001## {-e_r-}0x000000007FFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x00000000FFFFFFFF## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x0000000100000002## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x00000000FFFFFFFF## {-d-}0x0000000100000000##  {-e_q-}0x0000000100000000## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x00000000FFFFFFFF## {-d-}0x0000000100000001##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x00000000FFFFFFFF## {-d-}0x0000000180000000##  {-e_q-}0x00000000AAAAAAAB## {-e_r-}0x000000007FFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x00000000FFFFFFFF## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x0000000080000000## {-e_r-}0x000000017FFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x00000000FFFFFFFF## {-d-}0x8000000000000000##  {-e_q-}0x0000000000000002## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x00000000FFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0x0000000000000002## {-e_r-}0x00000000FFFFFFFD##
  , T {-n1-}0x0000000000000001## {-n2-}0x00000000FFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0x0000000000000001## {-e_r-}0x800000007FFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x00000000FFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x0000000000000001## {-e_r-}0x8000000000000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000000000001## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000000000001## {-e_r-}0x00000001FFFFFFFE##
  , T {-n1-}0x0000000000000001## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000000000001## {-e_r-}0x000000017FFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000000000001## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000100000000## {-d-}0x0000000080000000##  {-e_q-}0x0000000200000002## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000100000000## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x0000000100000002## {-e_r-}0x0000000000000002##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000100000000## {-d-}0x0000000100000000##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000100000000## {-d-}0x0000000100000001##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000100000000## {-d-}0x0000000180000000##  {-e_q-}0x00000000AAAAAAAB## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000100000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x0000000080000000## {-e_r-}0x0000000180000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000100000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000000000002## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000100000000## {-d-}0x8000000000000001##  {-e_q-}0x0000000000000002## {-e_r-}0x00000000FFFFFFFE##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000100000000## {-d-}0x8000000080000000##  {-e_q-}0x0000000000000002## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000100000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x0000000000000001## {-e_r-}0x8000000000000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000000000001## {-e_r-}0x0000000200000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000000000001## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000000000001## {-e_r-}0x0000000180000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000000000001## {-e_r-}0x0000000100000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000100000001## {-d-}0x0000000080000000##  {-e_q-}0x0000000200000002## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000100000001## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x0000000100000002## {-e_r-}0x0000000000000003##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000100000001## {-d-}0x0000000100000000##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000100000001## {-d-}0x0000000100000001##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000100000001## {-d-}0x0000000180000000##  {-e_q-}0x00000000AAAAAAAB## {-e_r-}0x0000000080000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000100000001## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x0000000080000000## {-e_r-}0x0000000180000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000100000001## {-d-}0x8000000000000000##  {-e_q-}0x0000000000000002## {-e_r-}0x0000000100000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000100000001## {-d-}0x8000000000000001##  {-e_q-}0x0000000000000002## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000100000001## {-d-}0x8000000080000000##  {-e_q-}0x0000000000000002## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000100000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x0000000000000001## {-e_r-}0x8000000000000002##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000000000001## {-e_r-}0x0000000200000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000000000001## {-e_r-}0x0000000200000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000000000001## {-e_r-}0x0000000180000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000000000001## {-e_r-}0x0000000100000002##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000180000000## {-d-}0x0000000080000000##  {-e_q-}0x0000000200000003## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000180000000## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x0000000100000002## {-e_r-}0x0000000080000002##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000180000000## {-d-}0x0000000100000000##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000180000000## {-d-}0x0000000100000001##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000180000000## {-d-}0x0000000180000000##  {-e_q-}0x00000000AAAAAAAB## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000180000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x0000000080000001## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000180000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000000000002## {-e_r-}0x0000000180000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000180000000## {-d-}0x8000000000000001##  {-e_q-}0x0000000000000002## {-e_r-}0x000000017FFFFFFE##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000180000000## {-d-}0x8000000080000000##  {-e_q-}0x0000000000000002## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000180000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x0000000000000001## {-e_r-}0x8000000080000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000000000001## {-e_r-}0x0000000280000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000000000001## {-e_r-}0x000000027FFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000000000001## {-e_r-}0x0000000200000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000000000001## {-e_r-}0x0000000180000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x00000001FFFFFFFF## {-d-}0x0000000080000000##  {-e_q-}0x0000000200000003## {-e_r-}0x000000007FFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x00000001FFFFFFFF## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x0000000100000003## {-e_r-}0x0000000000000002##
  , T {-n1-}0x0000000000000001## {-n2-}0x00000001FFFFFFFF## {-d-}0x0000000100000000##  {-e_q-}0x0000000100000001## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x00000001FFFFFFFF## {-d-}0x0000000100000001##  {-e_q-}0x0000000100000000## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x00000001FFFFFFFF## {-d-}0x0000000180000000##  {-e_q-}0x00000000AAAAAAAB## {-e_r-}0x000000017FFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x00000001FFFFFFFF## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x0000000080000001## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x00000001FFFFFFFF## {-d-}0x8000000000000000##  {-e_q-}0x0000000000000002## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x00000001FFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0x0000000000000002## {-e_r-}0x00000001FFFFFFFD##
  , T {-n1-}0x0000000000000001## {-n2-}0x00000001FFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0x0000000000000002## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x00000001FFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x0000000000000002## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000000000001## {-e_r-}0x00000002FFFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000000000001## {-e_r-}0x00000002FFFFFFFE##
  , T {-n1-}0x0000000000000001## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000000000001## {-e_r-}0x000000027FFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000000000001## {-e_r-}0x0000000200000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000000000000## {-d-}0x0000000080000000##  {-e_q-}0x0000000300000000## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000000000000## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x0000000180000001## {-e_r-}0x0000000080000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000000000000## {-d-}0x0000000100000000##  {-e_q-}0x0000000180000000## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000000000000## {-d-}0x0000000100000001##  {-e_q-}0x000000017FFFFFFE## {-e_r-}0x0000000080000002##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000000000000## {-d-}0x0000000180000000##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000000000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x00000000C0000000## {-e_r-}0x00000000C0000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000000000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000000000003## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000000000000## {-d-}0x8000000000000001##  {-e_q-}0x0000000000000002## {-e_r-}0x7FFFFFFFFFFFFFFE##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000000000000## {-d-}0x8000000080000000##  {-e_q-}0x0000000000000002## {-e_r-}0x7FFFFFFF00000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000000000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x0000000000000002## {-e_r-}0x7FFFFFFE00000002##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000000000001## {-e_r-}0x8000000100000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000000000001## {-e_r-}0x80000000FFFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000000000001## {-e_r-}0x8000000080000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000000000001## {-e_r-}0x8000000000000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000000000001## {-d-}0x0000000080000000##  {-e_q-}0x0000000300000000## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000000000001## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x0000000180000001## {-e_r-}0x0000000080000002##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000000000001## {-d-}0x0000000100000000##  {-e_q-}0x0000000180000000## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000000000001## {-d-}0x0000000100000001##  {-e_q-}0x000000017FFFFFFE## {-e_r-}0x0000000080000003##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000000000001## {-d-}0x0000000180000000##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000000000001## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x00000000C0000000## {-e_r-}0x00000000C0000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000000000001## {-d-}0x8000000000000000##  {-e_q-}0x0000000000000003## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000000000001## {-d-}0x8000000000000001##  {-e_q-}0x0000000000000002## {-e_r-}0x7FFFFFFFFFFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000000000001## {-d-}0x8000000080000000##  {-e_q-}0x0000000000000002## {-e_r-}0x7FFFFFFF00000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000000000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x0000000000000002## {-e_r-}0x7FFFFFFE00000003##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000000000001## {-e_r-}0x8000000100000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000000000001## {-e_r-}0x8000000100000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000000000001## {-e_r-}0x8000000080000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000000000001## {-e_r-}0x8000000000000002##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000080000000## {-d-}0x0000000080000000##  {-e_q-}0x0000000300000001## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000080000000## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x0000000180000002## {-e_r-}0x0000000000000002##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000080000000## {-d-}0x0000000100000000##  {-e_q-}0x0000000180000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000080000000## {-d-}0x0000000100000001##  {-e_q-}0x000000017FFFFFFF## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000080000000## {-d-}0x0000000180000000##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000080000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x00000000C0000000## {-e_r-}0x0000000140000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000080000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000000000003## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000080000000## {-d-}0x8000000000000001##  {-e_q-}0x0000000000000003## {-e_r-}0x000000007FFFFFFD##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000080000000## {-d-}0x8000000080000000##  {-e_q-}0x0000000000000002## {-e_r-}0x7FFFFFFF80000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000080000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x0000000000000002## {-e_r-}0x7FFFFFFE80000002##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000000000001## {-e_r-}0x8000000180000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000000000001## {-e_r-}0x800000017FFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000000000001## {-e_r-}0x8000000100000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000000000001## {-e_r-}0x8000000080000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x80000000FFFFFFFF## {-d-}0x0000000080000000##  {-e_q-}0x0000000300000001## {-e_r-}0x000000007FFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x80000000FFFFFFFF## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x0000000180000002## {-e_r-}0x0000000080000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x80000000FFFFFFFF## {-d-}0x0000000100000000##  {-e_q-}0x0000000180000000## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x80000000FFFFFFFF## {-d-}0x0000000100000001##  {-e_q-}0x000000017FFFFFFF## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000000000001## {-n2-}0x80000000FFFFFFFF## {-d-}0x0000000180000000##  {-e_q-}0x0000000100000000## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x80000000FFFFFFFF## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x00000000C0000000## {-e_r-}0x00000001BFFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x80000000FFFFFFFF## {-d-}0x8000000000000000##  {-e_q-}0x0000000000000003## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x80000000FFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0x0000000000000003## {-e_r-}0x00000000FFFFFFFC##
  , T {-n1-}0x0000000000000001## {-n2-}0x80000000FFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0x0000000000000002## {-e_r-}0x7FFFFFFFFFFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x80000000FFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x0000000000000002## {-e_r-}0x7FFFFFFF00000001##
  , T {-n1-}0x0000000000000001## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000000000001## {-e_r-}0x80000001FFFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000000000001## {-e_r-}0x80000001FFFFFFFE##
  , T {-n1-}0x0000000000000001## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000000000001## {-e_r-}0x800000017FFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000000000001## {-e_r-}0x8000000100000000##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF00000000## {-d-}0x0000000080000000##  {-e_q-}0x00000003FFFFFFFE## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF00000000## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x0000000200000001## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF00000000## {-d-}0x0000000100000000##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF00000000## {-d-}0x0000000100000001##  {-e_q-}0x00000001FFFFFFFD## {-e_r-}0x0000000000000003##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF00000000## {-d-}0x0000000180000000##  {-e_q-}0x0000000155555554## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF00000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF00000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000000000003## {-e_r-}0x7FFFFFFF00000000##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF00000000## {-d-}0x8000000000000001##  {-e_q-}0x0000000000000003## {-e_r-}0x7FFFFFFEFFFFFFFD##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF00000000## {-d-}0x8000000080000000##  {-e_q-}0x0000000000000003## {-e_r-}0x7FFFFFFD80000000##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF00000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x0000000000000003## {-e_r-}0x7FFFFFFC00000003##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000000000002## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000000000002## {-e_r-}0x00000000FFFFFFFE##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000000000002## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000000000001## {-e_r-}0xFFFFFFFF00000001##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF00000001## {-d-}0x0000000080000000##  {-e_q-}0x00000003FFFFFFFE## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF00000001## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x0000000200000001## {-e_r-}0x0000000000000002##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF00000001## {-d-}0x0000000100000000##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF00000001## {-d-}0x0000000100000001##  {-e_q-}0x00000001FFFFFFFD## {-e_r-}0x0000000000000004##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF00000001## {-d-}0x0000000180000000##  {-e_q-}0x0000000155555554## {-e_r-}0x0000000100000001##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF00000001## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF00000001## {-d-}0x8000000000000000##  {-e_q-}0x0000000000000003## {-e_r-}0x7FFFFFFF00000001##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF00000001## {-d-}0x8000000000000001##  {-e_q-}0x0000000000000003## {-e_r-}0x7FFFFFFEFFFFFFFE##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF00000001## {-d-}0x8000000080000000##  {-e_q-}0x0000000000000003## {-e_r-}0x7FFFFFFD80000001##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF00000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x0000000000000003## {-e_r-}0x7FFFFFFC00000004##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000000000002## {-e_r-}0x0000000100000001##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000000000002## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000000000002## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000000000001## {-e_r-}0xFFFFFFFF00000002##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF80000000## {-d-}0x0000000080000000##  {-e_q-}0x00000003FFFFFFFF## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF80000000## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x0000000200000001## {-e_r-}0x0000000080000001##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF80000000## {-d-}0x0000000100000000##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF80000000## {-d-}0x0000000100000001##  {-e_q-}0x00000001FFFFFFFD## {-e_r-}0x0000000080000003##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF80000000## {-d-}0x0000000180000000##  {-e_q-}0x0000000155555555## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF80000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF80000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000000000003## {-e_r-}0x7FFFFFFF80000000##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF80000000## {-d-}0x8000000000000001##  {-e_q-}0x0000000000000003## {-e_r-}0x7FFFFFFF7FFFFFFD##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF80000000## {-d-}0x8000000080000000##  {-e_q-}0x0000000000000003## {-e_r-}0x7FFFFFFE00000000##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF80000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x0000000000000003## {-e_r-}0x7FFFFFFC80000003##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000000000002## {-e_r-}0x0000000180000000##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000000000002## {-e_r-}0x000000017FFFFFFE##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000000000002## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000000000001## {-e_r-}0xFFFFFFFF80000001##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x0000000080000000##  {-e_q-}0x00000003FFFFFFFF## {-e_r-}0x000000007FFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x0000000200000002## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x0000000100000000##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x0000000100000001##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x0000000180000000##  {-e_q-}0x0000000155555555## {-e_r-}0x000000007FFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x0000000100000000## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x8000000000000000##  {-e_q-}0x0000000000000003## {-e_r-}0x7FFFFFFFFFFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0x0000000000000003## {-e_r-}0x7FFFFFFFFFFFFFFC##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0x0000000000000003## {-e_r-}0x7FFFFFFE7FFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x0000000000000003## {-e_r-}0x7FFFFFFD00000002##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000000000002## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000000000002## {-e_r-}0x00000001FFFFFFFD##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000000000002## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000000000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000000000002## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000000000000## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x8000000080000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000000000000## {-d-}0x0000000100000000##  {-e_q-}0x8000000000000000## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000000000000## {-d-}0x0000000100000001##  {-e_q-}0x7FFFFFFF80000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000000000000## {-d-}0x0000000180000000##  {-e_q-}0x5555555555555555## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000000000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x4000000020000000## {-e_r-}0x0000000020000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000000000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000000000000## {-d-}0x8000000000000001##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x7FFFFFFF00000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000000000000## {-d-}0x8000000080000000##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000000000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000000FFFFFFFE## {-e_r-}0x00000002FFFFFFFE##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000080000000## {-e_r-}0x8000000000000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000080000000## {-e_r-}0x7FFFFFFF80000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000080000000## {-e_r-}0x4000000000000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000080000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000000000001## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x8000000080000000## {-e_r-}0x0000000080000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000000000001## {-d-}0x0000000100000000##  {-e_q-}0x8000000000000000## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000000000001## {-d-}0x0000000100000001##  {-e_q-}0x7FFFFFFF80000000## {-e_r-}0x0000000080000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000000000001## {-d-}0x0000000180000000##  {-e_q-}0x5555555555555555## {-e_r-}0x0000000080000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000000000001## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x4000000020000000## {-e_r-}0x0000000020000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000000000001## {-d-}0x8000000000000000##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000000000001## {-d-}0x8000000000000001##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x7FFFFFFF00000002##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000000000001## {-d-}0x8000000080000000##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x0000000080000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000000000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000000FFFFFFFE## {-e_r-}0x00000002FFFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000080000000## {-e_r-}0x8000000000000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000080000000## {-e_r-}0x7FFFFFFF80000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000080000000## {-e_r-}0x4000000000000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000080000000## {-e_r-}0x0000000080000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000080000000## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x8000000080000001## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000080000000## {-d-}0x0000000100000000##  {-e_q-}0x8000000000000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000080000000## {-d-}0x0000000100000001##  {-e_q-}0x7FFFFFFF80000000## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000080000000## {-d-}0x0000000180000000##  {-e_q-}0x5555555555555555## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000080000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x4000000020000000## {-e_r-}0x00000000A0000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000080000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000080000000## {-d-}0x8000000000000001##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x7FFFFFFF80000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000080000000## {-d-}0x8000000080000000##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000080000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000000FFFFFFFE## {-e_r-}0x000000037FFFFFFE##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000080000000## {-e_r-}0x8000000080000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000080000000## {-e_r-}0x8000000000000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000080000000## {-e_r-}0x4000000080000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000080000000## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x00000000FFFFFFFF## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x8000000080000001## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x00000000FFFFFFFF## {-d-}0x0000000100000000##  {-e_q-}0x8000000000000000## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x00000000FFFFFFFF## {-d-}0x0000000100000001##  {-e_q-}0x7FFFFFFF80000001## {-e_r-}0x000000007FFFFFFE##
  , T {-n1-}0x0000000080000000## {-n2-}0x00000000FFFFFFFF## {-d-}0x0000000180000000##  {-e_q-}0x5555555555555555## {-e_r-}0x000000017FFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x00000000FFFFFFFF## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x4000000020000000## {-e_r-}0x000000011FFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x00000000FFFFFFFF## {-d-}0x8000000000000000##  {-e_q-}0x0000000100000000## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x00000000FFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x8000000000000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x00000000FFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x000000017FFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x00000000FFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000000FFFFFFFE## {-e_r-}0x00000003FFFFFFFD##
  , T {-n1-}0x0000000080000000## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000080000000## {-e_r-}0x80000000FFFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000080000000## {-e_r-}0x800000007FFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000080000000## {-e_r-}0x40000000FFFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000080000000## {-e_r-}0x000000017FFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000100000000## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x8000000080000001## {-e_r-}0x0000000080000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000100000000## {-d-}0x0000000100000000##  {-e_q-}0x8000000000000001## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000100000000## {-d-}0x0000000100000001##  {-e_q-}0x7FFFFFFF80000001## {-e_r-}0x000000007FFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000100000000## {-d-}0x0000000180000000##  {-e_q-}0x5555555555555556## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000100000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x4000000020000000## {-e_r-}0x0000000120000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000100000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000100000000## {-d-}0x8000000000000001##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000100000000## {-d-}0x8000000080000000##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x0000000180000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000100000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000000FFFFFFFE## {-e_r-}0x00000003FFFFFFFE##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000080000000## {-e_r-}0x8000000100000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000080000000## {-e_r-}0x8000000080000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000080000000## {-e_r-}0x4000000100000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000080000000## {-e_r-}0x0000000180000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000100000001## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x8000000080000001## {-e_r-}0x0000000080000002##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000100000001## {-d-}0x0000000100000000##  {-e_q-}0x8000000000000001## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000100000001## {-d-}0x0000000100000001##  {-e_q-}0x7FFFFFFF80000001## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000100000001## {-d-}0x0000000180000000##  {-e_q-}0x5555555555555556## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000100000001## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x4000000020000000## {-e_r-}0x0000000120000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000100000001## {-d-}0x8000000000000000##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000100000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000100000001## {-d-}0x8000000000000001##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000100000001## {-d-}0x8000000080000000##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x0000000180000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000100000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000000FFFFFFFE## {-e_r-}0x00000003FFFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000080000000## {-e_r-}0x8000000100000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000080000000## {-e_r-}0x8000000080000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000080000000## {-e_r-}0x4000000100000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000080000000## {-e_r-}0x0000000180000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000180000000## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x8000000080000002## {-e_r-}0x0000000000000002##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000180000000## {-d-}0x0000000100000000##  {-e_q-}0x8000000000000001## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000180000000## {-d-}0x0000000100000001##  {-e_q-}0x7FFFFFFF80000001## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000180000000## {-d-}0x0000000180000000##  {-e_q-}0x5555555555555556## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000180000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x4000000020000000## {-e_r-}0x00000001A0000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000180000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000180000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000180000000## {-d-}0x8000000000000001##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000180000000## {-d-}0x8000000080000000##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x0000000200000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000180000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000000FFFFFFFE## {-e_r-}0x000000047FFFFFFE##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000080000000## {-e_r-}0x8000000180000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000080000000## {-e_r-}0x8000000100000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000080000000## {-e_r-}0x4000000180000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000080000000## {-e_r-}0x0000000200000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x00000001FFFFFFFF## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x8000000080000002## {-e_r-}0x0000000080000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x00000001FFFFFFFF## {-d-}0x0000000100000000##  {-e_q-}0x8000000000000001## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x00000001FFFFFFFF## {-d-}0x0000000100000001##  {-e_q-}0x7FFFFFFF80000002## {-e_r-}0x000000007FFFFFFD##
  , T {-n1-}0x0000000080000000## {-n2-}0x00000001FFFFFFFF## {-d-}0x0000000180000000##  {-e_q-}0x5555555555555556## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x00000001FFFFFFFF## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x4000000020000001## {-e_r-}0x0000000020000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x00000001FFFFFFFF## {-d-}0x8000000000000000##  {-e_q-}0x0000000100000000## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x00000001FFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0x0000000100000000## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x00000001FFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x000000027FFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x00000001FFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000000FFFFFFFE## {-e_r-}0x00000004FFFFFFFD##
  , T {-n1-}0x0000000080000000## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000080000000## {-e_r-}0x80000001FFFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000080000000## {-e_r-}0x800000017FFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000080000000## {-e_r-}0x40000001FFFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000080000000## {-e_r-}0x000000027FFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000000000000## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x8000000100000001## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000000000000## {-d-}0x0000000100000000##  {-e_q-}0x8000000080000000## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000000000000## {-d-}0x0000000100000001##  {-e_q-}0x8000000000000000## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000000000000## {-d-}0x0000000180000000##  {-e_q-}0x55555555AAAAAAAA## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000000000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x4000000060000000## {-e_r-}0x0000000060000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000000000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000000000000## {-d-}0x8000000000000001##  {-e_q-}0x0000000100000000## {-e_r-}0x7FFFFFFF00000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000000000000## {-d-}0x8000000080000000##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000000000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000080000001## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000080000001## {-e_r-}0x000000007FFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000080000000## {-e_r-}0xC000000000000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000080000000## {-e_r-}0x8000000080000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000000000001## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x8000000100000001## {-e_r-}0x0000000000000002##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000000000001## {-d-}0x0000000100000000##  {-e_q-}0x8000000080000000## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000000000001## {-d-}0x0000000100000001##  {-e_q-}0x8000000000000000## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000000000001## {-d-}0x0000000180000000##  {-e_q-}0x55555555AAAAAAAA## {-e_r-}0x0000000100000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000000000001## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x4000000060000000## {-e_r-}0x0000000060000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000000000001## {-d-}0x8000000000000000##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000000000001## {-d-}0x8000000000000001##  {-e_q-}0x0000000100000000## {-e_r-}0x7FFFFFFF00000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000000000001## {-d-}0x8000000080000000##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000000000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x0000000200000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000080000001## {-e_r-}0x0000000100000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000080000001## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000080000000## {-e_r-}0xC000000000000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000080000000## {-e_r-}0x8000000080000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000080000000## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x8000000100000001## {-e_r-}0x0000000080000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000080000000## {-d-}0x0000000100000000##  {-e_q-}0x8000000080000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000080000000## {-d-}0x0000000100000001##  {-e_q-}0x8000000000000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000080000000## {-d-}0x0000000180000000##  {-e_q-}0x55555555AAAAAAAB## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000080000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x4000000060000000## {-e_r-}0x00000000E0000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000080000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000080000000## {-d-}0x8000000000000001##  {-e_q-}0x0000000100000000## {-e_r-}0x7FFFFFFF80000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000080000000## {-d-}0x8000000080000000##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000080000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x000000027FFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000080000001## {-e_r-}0x0000000180000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000080000001## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000080000000## {-e_r-}0xC000000080000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000080000000## {-e_r-}0x8000000100000000##
  , T {-n1-}0x0000000080000000## {-n2-}0x80000000FFFFFFFF## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x8000000100000002## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000080000000## {-n2-}0x80000000FFFFFFFF## {-d-}0x0000000100000000##  {-e_q-}0x8000000080000000## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x80000000FFFFFFFF## {-d-}0x0000000100000001##  {-e_q-}0x8000000000000000## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x80000000FFFFFFFF## {-d-}0x0000000180000000##  {-e_q-}0x55555555AAAAAAAB## {-e_r-}0x000000007FFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x80000000FFFFFFFF## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x4000000060000000## {-e_r-}0x000000015FFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x80000000FFFFFFFF## {-d-}0x8000000000000000##  {-e_q-}0x0000000100000001## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x80000000FFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0x0000000100000000## {-e_r-}0x7FFFFFFFFFFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x80000000FFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0x0000000100000000## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x80000000FFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x00000002FFFFFFFE##
  , T {-n1-}0x0000000080000000## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000080000001## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000080000001## {-e_r-}0x000000017FFFFFFE##
  , T {-n1-}0x0000000080000000## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000080000000## {-e_r-}0xC0000000FFFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000080000000## {-e_r-}0x800000017FFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF00000000## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x8000000180000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF00000000## {-d-}0x0000000100000000##  {-e_q-}0x80000000FFFFFFFF## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF00000000## {-d-}0x0000000100000001##  {-e_q-}0x800000007FFFFFFE## {-e_r-}0x0000000080000002##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF00000000## {-d-}0x0000000180000000##  {-e_q-}0x55555555FFFFFFFF## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF00000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x400000009FFFFFFF## {-e_r-}0x000000019FFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF00000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000100000001## {-e_r-}0x7FFFFFFF00000000##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF00000000## {-d-}0x8000000000000001##  {-e_q-}0x0000000100000001## {-e_r-}0x7FFFFFFDFFFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF00000000## {-d-}0x8000000080000000##  {-e_q-}0x0000000100000000## {-e_r-}0x7FFFFFFF00000000##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF00000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000080000001## {-e_r-}0x8000000000000000##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000080000001## {-e_r-}0x7FFFFFFF7FFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000080000001## {-e_r-}0x3FFFFFFF80000000##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000080000000## {-e_r-}0xFFFFFFFF80000000##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF00000001## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x8000000180000000## {-e_r-}0x0000000080000001##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF00000001## {-d-}0x0000000100000000##  {-e_q-}0x80000000FFFFFFFF## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF00000001## {-d-}0x0000000100000001##  {-e_q-}0x800000007FFFFFFE## {-e_r-}0x0000000080000003##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF00000001## {-d-}0x0000000180000000##  {-e_q-}0x55555555FFFFFFFF## {-e_r-}0x0000000080000001##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF00000001## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x400000009FFFFFFF## {-e_r-}0x00000001A0000000##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF00000001## {-d-}0x8000000000000000##  {-e_q-}0x0000000100000001## {-e_r-}0x7FFFFFFF00000001##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF00000001## {-d-}0x8000000000000001##  {-e_q-}0x0000000100000001## {-e_r-}0x7FFFFFFE00000000##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF00000001## {-d-}0x8000000080000000##  {-e_q-}0x0000000100000000## {-e_r-}0x7FFFFFFF00000001##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF00000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000080000001## {-e_r-}0x8000000000000001##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000080000001## {-e_r-}0x7FFFFFFF80000000##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000080000001## {-e_r-}0x3FFFFFFF80000001##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000080000000## {-e_r-}0xFFFFFFFF80000001##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF80000000## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x8000000180000001## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF80000000## {-d-}0x0000000100000000##  {-e_q-}0x80000000FFFFFFFF## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF80000000## {-d-}0x0000000100000001##  {-e_q-}0x800000007FFFFFFF## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF80000000## {-d-}0x0000000180000000##  {-e_q-}0x55555555FFFFFFFF## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF80000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x40000000A0000000## {-e_r-}0x0000000020000000##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF80000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000100000001## {-e_r-}0x7FFFFFFF80000000##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF80000000## {-d-}0x8000000000000001##  {-e_q-}0x0000000100000001## {-e_r-}0x7FFFFFFE7FFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF80000000## {-d-}0x8000000080000000##  {-e_q-}0x0000000100000000## {-e_r-}0x7FFFFFFF80000000##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF80000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000080000001## {-e_r-}0x8000000080000000##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000080000001## {-e_r-}0x7FFFFFFFFFFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000080000001## {-e_r-}0x4000000000000000##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000080000001## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x00000000FFFFFFFF##  {-e_q-}0x8000000180000001## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x0000000100000000##  {-e_q-}0x80000000FFFFFFFF## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x0000000100000001##  {-e_q-}0x800000007FFFFFFF## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x0000000180000000##  {-e_q-}0x55555555FFFFFFFF## {-e_r-}0x000000017FFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x40000000A0000000## {-e_r-}0x000000009FFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x8000000000000000##  {-e_q-}0x0000000100000001## {-e_r-}0x7FFFFFFFFFFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0x0000000100000001## {-e_r-}0x7FFFFFFEFFFFFFFE##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0x0000000100000000## {-e_r-}0x7FFFFFFFFFFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x0000000100000000## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000080000001## {-e_r-}0x80000000FFFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000080000001## {-e_r-}0x800000007FFFFFFE##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000080000001## {-e_r-}0x400000007FFFFFFF##
  , T {-n1-}0x0000000080000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000080000001## {-e_r-}0x0000000080000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000000000000## {-d-}0x0000000100000000##  {-e_q-}0xFFFFFFFF00000000## {-e_r-}0x0000000000000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000000000000## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFE00000001## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000000000000## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAA00000000## {-e_r-}0x0000000000000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000000000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x7FFFFFFFBFFFFFFF## {-e_r-}0x00000001BFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000000000000## {-d-}0x8000000000000000##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x0000000000000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000000000000## {-d-}0x8000000000000001##  {-e_q-}0x00000001FFFFFFFD## {-e_r-}0x7FFFFFFE00000003##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000000000000## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFC## {-e_r-}0x0000000200000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000000000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFA## {-e_r-}0x00000007FFFFFFFA##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000000000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0xFFFFFFFE00000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x7FFFFFFF80000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000000000001## {-d-}0x0000000100000000##  {-e_q-}0xFFFFFFFF00000000## {-e_r-}0x0000000000000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000000000001## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFE00000001## {-e_r-}0x0000000100000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000000000001## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAA00000000## {-e_r-}0x0000000000000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000000000001## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x7FFFFFFFBFFFFFFF## {-e_r-}0x00000001C0000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000000000001## {-d-}0x8000000000000000##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x0000000000000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000000000001## {-d-}0x8000000000000001##  {-e_q-}0x00000001FFFFFFFD## {-e_r-}0x7FFFFFFE00000004##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000000000001## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFC## {-e_r-}0x0000000200000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000000000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFA## {-e_r-}0x00000007FFFFFFFB##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000000000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0xFFFFFFFE00000002##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x7FFFFFFF80000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x0000000100000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000080000000## {-d-}0x0000000100000000##  {-e_q-}0xFFFFFFFF00000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000080000000## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFE00000002## {-e_r-}0x000000007FFFFFFE##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000080000000## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAA00000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000080000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x7FFFFFFFC0000000## {-e_r-}0x0000000040000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000080000000## {-d-}0x8000000000000000##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x0000000080000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000080000000## {-d-}0x8000000000000001##  {-e_q-}0x00000001FFFFFFFD## {-e_r-}0x7FFFFFFE80000003##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000080000000## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFC## {-e_r-}0x0000000280000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000080000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFA## {-e_r-}0x000000087FFFFFFA##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0xFFFFFFFE80000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x8000000000000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x000000017FFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x00000000FFFFFFFF## {-d-}0x0000000100000000##  {-e_q-}0xFFFFFFFF00000000## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x00000000FFFFFFFF## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFE00000002## {-e_r-}0x00000000FFFFFFFD##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x00000000FFFFFFFF## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAA00000000## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x00000000FFFFFFFF## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x7FFFFFFFC0000000## {-e_r-}0x00000000BFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x00000000FFFFFFFF## {-d-}0x8000000000000000##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x00000000FFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0x00000001FFFFFFFD## {-e_r-}0x7FFFFFFF00000002##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x00000000FFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFC## {-e_r-}0x00000002FFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x00000000FFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFA## {-e_r-}0x00000008FFFFFFF9##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000000## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0xFFFFFFFF00000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x800000007FFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x00000001FFFFFFFE##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000100000000## {-d-}0x0000000100000000##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0x0000000000000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000100000000## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFE00000002## {-e_r-}0x00000000FFFFFFFE##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000100000000## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAA00000000## {-e_r-}0x0000000100000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000100000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x7FFFFFFFC0000000## {-e_r-}0x00000000C0000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000100000000## {-d-}0x8000000000000000##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x0000000100000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000100000000## {-d-}0x8000000000000001##  {-e_q-}0x00000001FFFFFFFD## {-e_r-}0x7FFFFFFF00000003##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000100000000## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFC## {-e_r-}0x0000000300000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000100000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFA## {-e_r-}0x00000008FFFFFFFA##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000100000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000000000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x8000000080000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000100000001## {-d-}0x0000000100000000##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0x0000000000000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000100000001## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFE00000002## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000100000001## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAA00000000## {-e_r-}0x0000000100000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000100000001## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x7FFFFFFFC0000000## {-e_r-}0x00000000C0000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000100000001## {-d-}0x8000000000000000##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x0000000100000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000100000001## {-d-}0x8000000000000001##  {-e_q-}0x00000001FFFFFFFD## {-e_r-}0x7FFFFFFF00000004##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000100000001## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFC## {-e_r-}0x0000000300000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000100000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFA## {-e_r-}0x00000008FFFFFFFB##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000100000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000000000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x8000000080000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x0000000200000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000180000000## {-d-}0x0000000100000000##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0x0000000080000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000180000000## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFE00000003## {-e_r-}0x000000007FFFFFFD##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000180000000## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAA00000001## {-e_r-}0x0000000000000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000180000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x7FFFFFFFC0000000## {-e_r-}0x0000000140000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000180000000## {-d-}0x8000000000000000##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x0000000180000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000180000000## {-d-}0x8000000000000001##  {-e_q-}0x00000001FFFFFFFD## {-e_r-}0x7FFFFFFF80000003##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000180000000## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFC## {-e_r-}0x0000000380000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000180000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFA## {-e_r-}0x000000097FFFFFFA##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000180000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x8000000100000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x000000027FFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x00000001FFFFFFFF## {-d-}0x0000000100000000##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x00000001FFFFFFFF## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFE00000003## {-e_r-}0x00000000FFFFFFFC##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x00000001FFFFFFFF## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAA00000001## {-e_r-}0x000000007FFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x00000001FFFFFFFF## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x7FFFFFFFC0000000## {-e_r-}0x00000001BFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x00000001FFFFFFFF## {-d-}0x8000000000000000##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x00000001FFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x0000000000000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x00000001FFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFC## {-e_r-}0x00000003FFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x00000001FFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFA## {-e_r-}0x00000009FFFFFFF9##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000000## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000000## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x800000017FFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x00000002FFFFFFFE##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000000000000## {-d-}0x0000000100000000##  {-e_q-}0xFFFFFFFF80000000## {-e_r-}0x0000000000000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000000000000## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFE80000001## {-e_r-}0x000000007FFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000000000000## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAA55555555## {-e_r-}0x0000000080000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000000000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x8000000000000000## {-e_r-}0x0000000000000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000000000000## {-d-}0x8000000000000000##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x0000000000000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000000000000## {-d-}0x8000000000000001##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x7FFFFFFE00000002##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000000000000## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFD## {-e_r-}0x0000000180000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000000000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFB## {-e_r-}0x00000006FFFFFFFB##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000000## {-e_r-}0x8000000000000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000000## {-e_r-}0x7FFFFFFF00000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000000000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x80000000FFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000000000001## {-d-}0x0000000100000000##  {-e_q-}0xFFFFFFFF80000000## {-e_r-}0x0000000000000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000000000001## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFE80000001## {-e_r-}0x0000000080000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000000000001## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAA55555555## {-e_r-}0x0000000080000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000000000001## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x8000000000000000## {-e_r-}0x0000000000000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000000000001## {-d-}0x8000000000000000##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x0000000000000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000000000001## {-d-}0x8000000000000001##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x7FFFFFFE00000003##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000000000001## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFD## {-e_r-}0x0000000180000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000000000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFB## {-e_r-}0x00000006FFFFFFFC##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000000## {-e_r-}0x8000000000000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000000## {-e_r-}0x7FFFFFFF00000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000000000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x8000000100000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000080000000## {-d-}0x0000000100000000##  {-e_q-}0xFFFFFFFF80000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000080000000## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFE80000001## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000080000000## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAA55555555## {-e_r-}0x0000000100000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000080000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x8000000000000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000080000000## {-d-}0x8000000000000000##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x0000000080000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000080000000## {-d-}0x8000000000000001##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x7FFFFFFE80000002##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000080000000## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFD## {-e_r-}0x0000000200000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000080000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFB## {-e_r-}0x000000077FFFFFFB##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000000## {-e_r-}0x8000000080000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000000## {-e_r-}0x7FFFFFFF80000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x800000017FFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x80000000FFFFFFFF## {-d-}0x0000000100000000##  {-e_q-}0xFFFFFFFF80000000## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x80000000FFFFFFFF## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFE80000002## {-e_r-}0x000000007FFFFFFD##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x80000000FFFFFFFF## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAA55555555## {-e_r-}0x000000017FFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x80000000FFFFFFFF## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x8000000000000000## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x80000000FFFFFFFF## {-d-}0x8000000000000000##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x80000000FFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x7FFFFFFF00000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x80000000FFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFD## {-e_r-}0x000000027FFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x80000000FFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFB## {-e_r-}0x00000007FFFFFFFA##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000000## {-e_r-}0x80000000FFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000000## {-e_r-}0x7FFFFFFFFFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000000## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x00000000FFFFFFFF## {-e_r-}0x80000001FFFFFFFE##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF00000000## {-d-}0x0000000100000000##  {-e_q-}0xFFFFFFFFFFFFFFFF## {-e_r-}0x0000000000000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF00000000## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFF00000000## {-e_r-}0x0000000000000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF00000000## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAAAAAAAAAA## {-e_r-}0x0000000000000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF00000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x800000003FFFFFFF## {-e_r-}0x000000013FFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF00000000## {-d-}0x8000000000000000##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x7FFFFFFF00000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF00000000## {-d-}0x8000000000000001##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x7FFFFFFD00000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF00000000## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x0000000000000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF00000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFC## {-e_r-}0x00000004FFFFFFFC##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000000000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000000## {-e_r-}0xFFFFFFFE00000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000000## {-e_r-}0x7FFFFFFF00000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000000000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF00000001## {-d-}0x0000000100000000##  {-e_q-}0xFFFFFFFFFFFFFFFF## {-e_r-}0x0000000000000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF00000001## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFF00000000## {-e_r-}0x0000000000000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF00000001## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAAAAAAAAAA## {-e_r-}0x0000000000000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF00000001## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x800000003FFFFFFF## {-e_r-}0x0000000140000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF00000001## {-d-}0x8000000000000000##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x7FFFFFFF00000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF00000001## {-d-}0x8000000000000001##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x7FFFFFFD00000002##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF00000001## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x0000000000000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF00000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFC## {-e_r-}0x00000004FFFFFFFD##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000000000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000000## {-e_r-}0xFFFFFFFE00000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000000## {-e_r-}0x7FFFFFFF00000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000000000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF80000000## {-d-}0x0000000100000000##  {-e_q-}0xFFFFFFFFFFFFFFFF## {-e_r-}0x0000000080000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF80000000## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFF00000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF80000000## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAAAAAAAAAA## {-e_r-}0x0000000080000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF80000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x800000003FFFFFFF## {-e_r-}0x00000001BFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF80000000## {-d-}0x8000000000000000##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x7FFFFFFF80000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF80000000## {-d-}0x8000000000000001##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x7FFFFFFD80000001##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF80000000## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x0000000080000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF80000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFC## {-e_r-}0x000000057FFFFFFC##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000080000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000000## {-e_r-}0xFFFFFFFE80000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000000## {-e_r-}0x7FFFFFFF80000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x0000000100000000##  {-e_q-}0xFFFFFFFFFFFFFFFF## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFF00000000## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAAAAAAAAAA## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x8000000040000000## {-e_r-}0x000000003FFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x8000000000000000##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x7FFFFFFFFFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x7FFFFFFE00000000##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFC## {-e_r-}0x00000005FFFFFFFB##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000001## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000000## {-e_r-}0xFFFFFFFEFFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000000## {-e_r-}0x7FFFFFFFFFFFFFFF##
  , T {-n1-}0x00000000FFFFFFFF## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000000## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000000000000## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFF00000000## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000000000000## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAAAAAAAAAA## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000000000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x8000000040000000## {-e_r-}0x0000000040000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000000000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000000## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000000000000## {-d-}0x8000000000000001##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x7FFFFFFE00000001##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000000000000## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000000000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFC## {-e_r-}0x00000005FFFFFFFC##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000000## {-e_r-}0xFFFFFFFF00000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000000## {-e_r-}0x8000000000000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000000000001## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000000000001## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAAAAAAAAAA## {-e_r-}0x0000000100000001##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000000000001## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x8000000040000000## {-e_r-}0x0000000040000001##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000000000001## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000000## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000000000001## {-d-}0x8000000000000001##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x7FFFFFFE00000002##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000000000001## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x0000000100000001##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000000000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFC## {-e_r-}0x00000005FFFFFFFD##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000100000001##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000000## {-e_r-}0x8000000000000001##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000100000001##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000080000000## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0x000000007FFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000080000000## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAAAAAAAAAB## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000080000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x8000000040000000## {-e_r-}0x00000000C0000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000080000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000080000000## {-d-}0x8000000000000001##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x7FFFFFFE80000001##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000080000000## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x0000000180000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000080000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFC## {-e_r-}0x000000067FFFFFFC##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000180000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000001## {-e_r-}0x000000007FFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000000## {-e_r-}0x8000000080000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000180000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x00000000FFFFFFFF## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0x00000000FFFFFFFE##
  , T {-n1-}0x0000000100000000## {-n2-}0x00000000FFFFFFFF## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAAAAAAAAAB## {-e_r-}0x000000007FFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0x00000000FFFFFFFF## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x8000000040000000## {-e_r-}0x000000013FFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0x00000000FFFFFFFF## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000000## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0x00000000FFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x7FFFFFFF00000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x00000000FFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0x00000000FFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFC## {-e_r-}0x00000006FFFFFFFB##
  , T {-n1-}0x0000000100000000## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000001## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000001## {-e_r-}0x00000000FFFFFFFE##
  , T {-n1-}0x0000000100000000## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000000## {-e_r-}0x80000000FFFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000000## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000100000000## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000100000000## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAAAAAAAAAB## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000100000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x8000000040000000## {-e_r-}0x0000000140000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000100000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000000## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000100000000## {-d-}0x8000000000000001##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x7FFFFFFF00000001##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000100000000## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x0000000200000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000100000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFC## {-e_r-}0x00000006FFFFFFFC##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000200000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000001## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000000## {-e_r-}0x8000000100000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000200000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000100000001## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000100000001## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAAAAAAAAAB## {-e_r-}0x0000000080000001##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000100000001## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x8000000040000000## {-e_r-}0x0000000140000001##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000100000001## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000000## {-e_r-}0x0000000100000001##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000100000001## {-d-}0x8000000000000001##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x7FFFFFFF00000002##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000100000001## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x0000000200000001##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000100000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFC## {-e_r-}0x00000006FFFFFFFD##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000200000001##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000000## {-e_r-}0x8000000100000001##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000200000001##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000180000000## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFF00000002## {-e_r-}0x000000007FFFFFFE##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000180000000## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAAAAAAAAAB## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000180000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x8000000040000000## {-e_r-}0x00000001C0000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000180000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000000## {-e_r-}0x0000000180000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000180000000## {-d-}0x8000000000000001##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x7FFFFFFF80000001##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000180000000## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x0000000280000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000180000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFC## {-e_r-}0x000000077FFFFFFC##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000280000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000001## {-e_r-}0x000000017FFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000000## {-e_r-}0x8000000180000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000000## {-e_r-}0x0000000280000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x00000001FFFFFFFF## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFF00000002## {-e_r-}0x00000000FFFFFFFD##
  , T {-n1-}0x0000000100000000## {-n2-}0x00000001FFFFFFFF## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAAAAAAAAAB## {-e_r-}0x000000017FFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0x00000001FFFFFFFF## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x8000000040000001## {-e_r-}0x0000000040000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x00000001FFFFFFFF## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000000## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0x00000001FFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x8000000000000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x00000001FFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x00000002FFFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0x00000001FFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFC## {-e_r-}0x00000007FFFFFFFB##
  , T {-n1-}0x0000000100000000## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000001## {-e_r-}0x00000002FFFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000001## {-e_r-}0x00000001FFFFFFFE##
  , T {-n1-}0x0000000100000000## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000000## {-e_r-}0x80000001FFFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000000## {-e_r-}0x00000002FFFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000000000000## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFF80000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000000000000## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAB00000000## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000000000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x8000000080000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000000000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000001## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000000000000## {-d-}0x8000000000000001##  {-e_q-}0x0000000200000000## {-e_r-}0x7FFFFFFE00000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000000000000## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000000000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFD## {-e_r-}0x00000004FFFFFFFD##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000001## {-e_r-}0x8000000100000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000001## {-e_r-}0x7FFFFFFFFFFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000000## {-e_r-}0x8000000100000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000000000001## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFF80000000## {-e_r-}0x0000000080000001##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000000000001## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAB00000000## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000000000001## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x8000000080000000## {-e_r-}0x0000000080000001##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000000000001## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000001## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000000000001## {-d-}0x8000000000000001##  {-e_q-}0x0000000200000000## {-e_r-}0x7FFFFFFE00000001##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000000000001## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x0000000080000001##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000000000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFD## {-e_r-}0x00000004FFFFFFFE##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000001## {-e_r-}0x8000000100000001##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000001## {-e_r-}0x8000000000000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000080000001##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000000## {-e_r-}0x8000000100000001##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000080000000## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFF80000000## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000080000000## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAB00000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000080000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x8000000080000000## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000080000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000001## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000080000000## {-d-}0x8000000000000001##  {-e_q-}0x0000000200000000## {-e_r-}0x7FFFFFFE80000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000080000000## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000080000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFD## {-e_r-}0x000000057FFFFFFD##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000001## {-e_r-}0x8000000180000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000001## {-e_r-}0x800000007FFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000000## {-e_r-}0x8000000180000000##
  , T {-n1-}0x0000000100000000## {-n2-}0x80000000FFFFFFFF## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFF80000001## {-e_r-}0x000000007FFFFFFE##
  , T {-n1-}0x0000000100000000## {-n2-}0x80000000FFFFFFFF## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAB00000000## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0x80000000FFFFFFFF## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x8000000080000000## {-e_r-}0x000000017FFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0x80000000FFFFFFFF## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000001## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0x80000000FFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0x0000000200000000## {-e_r-}0x7FFFFFFEFFFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0x80000000FFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x000000017FFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0x80000000FFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFD## {-e_r-}0x00000005FFFFFFFC##
  , T {-n1-}0x0000000100000000## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000001## {-e_r-}0x80000001FFFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000001## {-e_r-}0x80000000FFFFFFFE##
  , T {-n1-}0x0000000100000000## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000001## {-e_r-}0x000000017FFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000000## {-e_r-}0x80000001FFFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF00000000## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFFFFFFFFFF## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF00000000## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAB55555554## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF00000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x80000000BFFFFFFF## {-e_r-}0x00000001BFFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF00000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000001## {-e_r-}0x7FFFFFFF00000000##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF00000000## {-d-}0x8000000000000001##  {-e_q-}0x0000000200000001## {-e_r-}0x7FFFFFFCFFFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF00000000## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x7FFFFFFF80000000##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF00000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x00000002FFFFFFFE##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000002## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000001## {-e_r-}0xFFFFFFFEFFFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000001## {-e_r-}0x7FFFFFFF80000000##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF00000001## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFFFFFFFFFF## {-e_r-}0x0000000000000002##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF00000001## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAB55555554## {-e_r-}0x0000000100000001##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF00000001## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x80000000BFFFFFFF## {-e_r-}0x00000001C0000000##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF00000001## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000001## {-e_r-}0x7FFFFFFF00000001##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF00000001## {-d-}0x8000000000000001##  {-e_q-}0x0000000200000001## {-e_r-}0x7FFFFFFD00000000##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF00000001## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x7FFFFFFF80000001##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF00000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x00000002FFFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000002## {-e_r-}0x0000000100000001##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000001## {-e_r-}0xFFFFFFFF00000000##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000001## {-e_r-}0x7FFFFFFF80000001##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000000000002##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF80000000## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFFFFFFFFFF## {-e_r-}0x0000000080000001##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF80000000## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAB55555555## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF80000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x80000000C0000000## {-e_r-}0x0000000040000000##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF80000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000001## {-e_r-}0x7FFFFFFF80000000##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF80000000## {-d-}0x8000000000000001##  {-e_q-}0x0000000200000001## {-e_r-}0x7FFFFFFD7FFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF80000000## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x8000000000000000##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF80000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x000000037FFFFFFE##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000002## {-e_r-}0x0000000180000000##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000002## {-e_r-}0x000000007FFFFFFE##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000001## {-e_r-}0x8000000000000000##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000080000001##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x0000000100000001##  {-e_q-}0xFFFFFFFFFFFFFFFF## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAB55555555## {-e_r-}0x000000007FFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x80000000C0000000## {-e_r-}0x00000000BFFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000001## {-e_r-}0x7FFFFFFFFFFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0x0000000200000001## {-e_r-}0x7FFFFFFDFFFFFFFE##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x800000007FFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x00000003FFFFFFFD##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000002## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000002## {-e_r-}0x00000000FFFFFFFD##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000001## {-e_r-}0x800000007FFFFFFF##
  , T {-n1-}0x0000000100000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000000000000## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAB55555555## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000000000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x80000000C0000000## {-e_r-}0x00000000C0000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000000000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000002## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000000000000## {-d-}0x8000000000000001##  {-e_q-}0x0000000200000001## {-e_r-}0x7FFFFFFDFFFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000000000000## {-d-}0x8000000080000000##  {-e_q-}0x0000000200000000## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000000000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x00000003FFFFFFFE##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000002## {-e_r-}0x0000000200000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000002## {-e_r-}0x00000000FFFFFFFE##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000001## {-e_r-}0x8000000080000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000100000001##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000000000001## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAB55555555## {-e_r-}0x0000000080000001##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000000000001## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x80000000C0000000## {-e_r-}0x00000000C0000001##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000000000001## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000002## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000000000001## {-d-}0x8000000000000001##  {-e_q-}0x0000000200000001## {-e_r-}0x7FFFFFFE00000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000000000001## {-d-}0x8000000080000000##  {-e_q-}0x0000000200000000## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000000000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x00000003FFFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000002## {-e_r-}0x0000000200000001##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000002## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000001## {-e_r-}0x8000000080000001##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000100000002##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000080000000## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAB55555555## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000080000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x80000000C0000000## {-e_r-}0x0000000140000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000080000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000002## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000080000000## {-d-}0x8000000000000001##  {-e_q-}0x0000000200000001## {-e_r-}0x7FFFFFFE7FFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000080000000## {-d-}0x8000000080000000##  {-e_q-}0x0000000200000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000080000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x000000047FFFFFFE##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000002## {-e_r-}0x0000000280000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000002## {-e_r-}0x000000017FFFFFFE##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000001## {-e_r-}0x8000000100000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000180000001##
  , T {-n1-}0x0000000100000001## {-n2-}0x00000000FFFFFFFF## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAB55555555## {-e_r-}0x000000017FFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0x00000000FFFFFFFF## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x80000000C0000000## {-e_r-}0x00000001BFFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0x00000000FFFFFFFF## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000002## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0x00000000FFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0x0000000200000001## {-e_r-}0x7FFFFFFEFFFFFFFE##
  , T {-n1-}0x0000000100000001## {-n2-}0x00000000FFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0x0000000200000000## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0x00000000FFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x00000004FFFFFFFD##
  , T {-n1-}0x0000000100000001## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000002## {-e_r-}0x00000002FFFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000002## {-e_r-}0x00000001FFFFFFFD##
  , T {-n1-}0x0000000100000001## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000001## {-e_r-}0x800000017FFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000200000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000100000000## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAB55555556## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000100000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x80000000C0000000## {-e_r-}0x00000001C0000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000100000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000002## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000100000000## {-d-}0x8000000000000001##  {-e_q-}0x0000000200000001## {-e_r-}0x7FFFFFFEFFFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000100000000## {-d-}0x8000000080000000##  {-e_q-}0x0000000200000000## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000100000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x00000004FFFFFFFE##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000002## {-e_r-}0x0000000300000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000002## {-e_r-}0x00000001FFFFFFFE##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000001## {-e_r-}0x8000000180000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000200000001##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000100000001## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAB55555556## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000100000001## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x80000000C0000000## {-e_r-}0x00000001C0000001##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000100000001## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000002## {-e_r-}0x0000000100000001##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000100000001## {-d-}0x8000000000000001##  {-e_q-}0x0000000200000001## {-e_r-}0x7FFFFFFF00000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000100000001## {-d-}0x8000000080000000##  {-e_q-}0x0000000200000000## {-e_r-}0x0000000100000001##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000100000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x00000004FFFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000002## {-e_r-}0x0000000300000001##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000002## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000001## {-e_r-}0x8000000180000001##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000200000002##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000180000000## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAB55555556## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000180000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x80000000C0000001## {-e_r-}0x0000000040000001##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000180000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000002## {-e_r-}0x0000000180000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000180000000## {-d-}0x8000000000000001##  {-e_q-}0x0000000200000001## {-e_r-}0x7FFFFFFF7FFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000180000000## {-d-}0x8000000080000000##  {-e_q-}0x0000000200000000## {-e_r-}0x0000000180000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000180000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x000000057FFFFFFE##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000002## {-e_r-}0x0000000380000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000002## {-e_r-}0x000000027FFFFFFE##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000001## {-e_r-}0x8000000200000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000280000001##
  , T {-n1-}0x0000000100000001## {-n2-}0x00000001FFFFFFFF## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAAB55555556## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0x00000001FFFFFFFF## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x80000000C0000001## {-e_r-}0x00000000C0000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x00000001FFFFFFFF## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000002## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0x00000001FFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0x0000000200000001## {-e_r-}0x7FFFFFFFFFFFFFFE##
  , T {-n1-}0x0000000100000001## {-n2-}0x00000001FFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0x0000000200000000## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0x00000001FFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFE## {-e_r-}0x00000005FFFFFFFD##
  , T {-n1-}0x0000000100000001## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000002## {-e_r-}0x00000003FFFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000002## {-e_r-}0x00000002FFFFFFFD##
  , T {-n1-}0x0000000100000001## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000001## {-e_r-}0x800000027FFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000001## {-e_r-}0x0000000300000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x8000000000000000## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAABAAAAAAAA## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x8000000000000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x8000000100000000## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x8000000000000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000003## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x8000000000000000## {-d-}0x8000000000000001##  {-e_q-}0x0000000200000002## {-e_r-}0x7FFFFFFDFFFFFFFE##
  , T {-n1-}0x0000000100000001## {-n2-}0x8000000000000000## {-d-}0x8000000080000000##  {-e_q-}0x0000000200000000## {-e_r-}0x8000000000000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x8000000000000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x00000002FFFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000002## {-e_r-}0x8000000200000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000002## {-e_r-}0x80000000FFFFFFFE##
  , T {-n1-}0x0000000100000001## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000002## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000001## {-e_r-}0x8000000100000001##
  , T {-n1-}0x0000000100000001## {-n2-}0x8000000000000001## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAABAAAAAAAA## {-e_r-}0x0000000100000001##
  , T {-n1-}0x0000000100000001## {-n2-}0x8000000000000001## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x8000000100000000## {-e_r-}0x0000000100000001##
  , T {-n1-}0x0000000100000001## {-n2-}0x8000000000000001## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000003## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000100000001## {-n2-}0x8000000000000001## {-d-}0x8000000000000001##  {-e_q-}0x0000000200000002## {-e_r-}0x7FFFFFFDFFFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0x8000000000000001## {-d-}0x8000000080000000##  {-e_q-}0x0000000200000000## {-e_r-}0x8000000000000001##
  , T {-n1-}0x0000000100000001## {-n2-}0x8000000000000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x0000000300000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000002## {-e_r-}0x8000000200000001##
  , T {-n1-}0x0000000100000001## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000002## {-e_r-}0x80000000FFFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000002## {-e_r-}0x0000000100000001##
  , T {-n1-}0x0000000100000001## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000001## {-e_r-}0x8000000100000002##
  , T {-n1-}0x0000000100000001## {-n2-}0x8000000080000000## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAABAAAAAAAB## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x8000000080000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x8000000100000000## {-e_r-}0x0000000180000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x8000000080000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000003## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x8000000080000000## {-d-}0x8000000000000001##  {-e_q-}0x0000000200000002## {-e_r-}0x7FFFFFFE7FFFFFFE##
  , T {-n1-}0x0000000100000001## {-n2-}0x8000000080000000## {-d-}0x8000000080000000##  {-e_q-}0x0000000200000001## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x8000000080000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x000000037FFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000002## {-e_r-}0x8000000280000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000002## {-e_r-}0x800000017FFFFFFE##
  , T {-n1-}0x0000000100000001## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000002## {-e_r-}0x0000000180000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000001## {-e_r-}0x8000000180000001##
  , T {-n1-}0x0000000100000001## {-n2-}0x80000000FFFFFFFF## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAABAAAAAAAB## {-e_r-}0x000000007FFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0x80000000FFFFFFFF## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x8000000100000001## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000100000001## {-n2-}0x80000000FFFFFFFF## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000003## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0x80000000FFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0x0000000200000002## {-e_r-}0x7FFFFFFEFFFFFFFD##
  , T {-n1-}0x0000000100000001## {-n2-}0x80000000FFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0x0000000200000001## {-e_r-}0x000000007FFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0x80000000FFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x00000003FFFFFFFE##
  , T {-n1-}0x0000000100000001## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000002## {-e_r-}0x80000002FFFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000002## {-e_r-}0x80000001FFFFFFFD##
  , T {-n1-}0x0000000100000001## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000002## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000001## {-e_r-}0x8000000200000000##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFF00000000## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAABFFFFFFFF## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFF00000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x8000000140000000## {-e_r-}0x0000000040000000##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFF00000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000003## {-e_r-}0x7FFFFFFF00000000##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFF00000000## {-d-}0x8000000000000001##  {-e_q-}0x0000000200000003## {-e_r-}0x7FFFFFFCFFFFFFFD##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFF00000000## {-d-}0x8000000080000000##  {-e_q-}0x0000000200000001## {-e_r-}0x7FFFFFFE80000000##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFF00000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x0000000200000000## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000003## {-e_r-}0x0000000200000000##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000003## {-e_r-}0x00000000FFFFFFFD##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000002## {-e_r-}0x8000000000000000##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000002## {-e_r-}0x0000000000000002##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFF00000001## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAABFFFFFFFF## {-e_r-}0x0000000080000001##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFF00000001## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x8000000140000000## {-e_r-}0x0000000040000001##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFF00000001## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000003## {-e_r-}0x7FFFFFFF00000001##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFF00000001## {-d-}0x8000000000000001##  {-e_q-}0x0000000200000003## {-e_r-}0x7FFFFFFCFFFFFFFE##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFF00000001## {-d-}0x8000000080000000##  {-e_q-}0x0000000200000001## {-e_r-}0x7FFFFFFE80000001##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFF00000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x0000000200000000## {-e_r-}0x0000000100000001##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000003## {-e_r-}0x0000000200000001##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000003## {-e_r-}0x00000000FFFFFFFE##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000002## {-e_r-}0x8000000000000001##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000002## {-e_r-}0x0000000000000003##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFF80000000## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAABFFFFFFFF## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFF80000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x8000000140000000## {-e_r-}0x00000000C0000000##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFF80000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000003## {-e_r-}0x7FFFFFFF80000000##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFF80000000## {-d-}0x8000000000000001##  {-e_q-}0x0000000200000003## {-e_r-}0x7FFFFFFD7FFFFFFD##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFF80000000## {-d-}0x8000000080000000##  {-e_q-}0x0000000200000001## {-e_r-}0x7FFFFFFF00000000##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFF80000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x0000000200000000## {-e_r-}0x0000000180000000##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000003## {-e_r-}0x0000000280000000##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000003## {-e_r-}0x000000017FFFFFFD##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000002## {-e_r-}0x8000000080000000##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000002## {-e_r-}0x0000000080000002##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x0000000180000000##  {-e_q-}0xAAAAAAABFFFFFFFF## {-e_r-}0x000000017FFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x00000001FFFFFFFF##  {-e_q-}0x8000000140000000## {-e_r-}0x000000013FFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x8000000000000000##  {-e_q-}0x0000000200000003## {-e_r-}0x7FFFFFFFFFFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0x0000000200000003## {-e_r-}0x7FFFFFFDFFFFFFFC##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0x0000000200000001## {-e_r-}0x7FFFFFFF7FFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x0000000200000000## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000100000003## {-e_r-}0x00000002FFFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000100000003## {-e_r-}0x00000001FFFFFFFC##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000100000002## {-e_r-}0x80000000FFFFFFFF##
  , T {-n1-}0x0000000100000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000100000002## {-e_r-}0x0000000100000001##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000000000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0xC000000060000000## {-e_r-}0x0000000060000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000000000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000300000000## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000000000000## {-d-}0x8000000000000001##  {-e_q-}0x00000002FFFFFFFF## {-e_r-}0x7FFFFFFD00000001##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000000000000## {-d-}0x8000000080000000##  {-e_q-}0x00000002FFFFFFFD## {-e_r-}0x0000000180000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000000000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000002FFFFFFFA## {-e_r-}0x00000008FFFFFFFA##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000180000001## {-e_r-}0x8000000100000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000180000001## {-e_r-}0x7FFFFFFF7FFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000180000000## {-e_r-}0xC000000000000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000180000000## {-e_r-}0x0000000180000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000000000001## {-d-}0x00000001FFFFFFFF##  {-e_q-}0xC000000060000000## {-e_r-}0x0000000060000001##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000000000001## {-d-}0x8000000000000000##  {-e_q-}0x0000000300000000## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000000000001## {-d-}0x8000000000000001##  {-e_q-}0x00000002FFFFFFFF## {-e_r-}0x7FFFFFFD00000002##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000000000001## {-d-}0x8000000080000000##  {-e_q-}0x00000002FFFFFFFD## {-e_r-}0x0000000180000001##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000000000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000002FFFFFFFA## {-e_r-}0x00000008FFFFFFFB##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000180000001## {-e_r-}0x8000000100000001##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000180000001## {-e_r-}0x7FFFFFFF80000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000180000000## {-e_r-}0xC000000000000001##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000180000000## {-e_r-}0x0000000180000001##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000080000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0xC000000060000000## {-e_r-}0x00000000E0000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000080000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000300000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000080000000## {-d-}0x8000000000000001##  {-e_q-}0x00000002FFFFFFFF## {-e_r-}0x7FFFFFFD80000001##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000080000000## {-d-}0x8000000080000000##  {-e_q-}0x00000002FFFFFFFD## {-e_r-}0x0000000200000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000080000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000002FFFFFFFA## {-e_r-}0x000000097FFFFFFA##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000180000001## {-e_r-}0x8000000180000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000180000001## {-e_r-}0x7FFFFFFFFFFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000180000000## {-e_r-}0xC000000080000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000180000000## {-e_r-}0x0000000200000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x00000000FFFFFFFF## {-d-}0x00000001FFFFFFFF##  {-e_q-}0xC000000060000000## {-e_r-}0x000000015FFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0x00000000FFFFFFFF## {-d-}0x8000000000000000##  {-e_q-}0x0000000300000000## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0x00000000FFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0x00000002FFFFFFFF## {-e_r-}0x7FFFFFFE00000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x00000000FFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0x00000002FFFFFFFD## {-e_r-}0x000000027FFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0x00000000FFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000002FFFFFFFA## {-e_r-}0x00000009FFFFFFF9##
  , T {-n1-}0x0000000180000000## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000180000001## {-e_r-}0x80000001FFFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000180000001## {-e_r-}0x800000007FFFFFFE##
  , T {-n1-}0x0000000180000000## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000180000000## {-e_r-}0xC0000000FFFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000180000000## {-e_r-}0x000000027FFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000100000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0xC000000060000000## {-e_r-}0x0000000160000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000100000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000300000000## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000100000000## {-d-}0x8000000000000001##  {-e_q-}0x00000002FFFFFFFF## {-e_r-}0x7FFFFFFE00000001##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000100000000## {-d-}0x8000000080000000##  {-e_q-}0x00000002FFFFFFFD## {-e_r-}0x0000000280000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000100000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000002FFFFFFFA## {-e_r-}0x00000009FFFFFFFA##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000180000001## {-e_r-}0x8000000200000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000180000001## {-e_r-}0x800000007FFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000180000000## {-e_r-}0xC000000100000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000180000000## {-e_r-}0x0000000280000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000100000001## {-d-}0x00000001FFFFFFFF##  {-e_q-}0xC000000060000000## {-e_r-}0x0000000160000001##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000100000001## {-d-}0x8000000000000000##  {-e_q-}0x0000000300000000## {-e_r-}0x0000000100000001##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000100000001## {-d-}0x8000000000000001##  {-e_q-}0x00000002FFFFFFFF## {-e_r-}0x7FFFFFFE00000002##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000100000001## {-d-}0x8000000080000000##  {-e_q-}0x00000002FFFFFFFD## {-e_r-}0x0000000280000001##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000100000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000002FFFFFFFA## {-e_r-}0x00000009FFFFFFFB##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000180000001## {-e_r-}0x8000000200000001##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000180000001## {-e_r-}0x8000000080000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000180000000## {-e_r-}0xC000000100000001##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000180000000## {-e_r-}0x0000000280000001##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000180000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0xC000000060000000## {-e_r-}0x00000001E0000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000180000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000300000000## {-e_r-}0x0000000180000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000180000000## {-d-}0x8000000000000001##  {-e_q-}0x00000002FFFFFFFF## {-e_r-}0x7FFFFFFE80000001##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000180000000## {-d-}0x8000000080000000##  {-e_q-}0x00000002FFFFFFFD## {-e_r-}0x0000000300000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000180000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000002FFFFFFFA## {-e_r-}0x0000000A7FFFFFFA##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000180000001## {-e_r-}0x8000000280000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000180000001## {-e_r-}0x80000000FFFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000180000000## {-e_r-}0xC000000180000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000180000000## {-e_r-}0x0000000300000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x00000001FFFFFFFF## {-d-}0x00000001FFFFFFFF##  {-e_q-}0xC000000060000001## {-e_r-}0x0000000060000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x00000001FFFFFFFF## {-d-}0x8000000000000000##  {-e_q-}0x0000000300000000## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0x00000001FFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0x00000002FFFFFFFF## {-e_r-}0x7FFFFFFF00000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x00000001FFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0x00000002FFFFFFFD## {-e_r-}0x000000037FFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0x00000001FFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000002FFFFFFFA## {-e_r-}0x0000000AFFFFFFF9##
  , T {-n1-}0x0000000180000000## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000180000001## {-e_r-}0x80000002FFFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000180000001## {-e_r-}0x800000017FFFFFFE##
  , T {-n1-}0x0000000180000000## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000180000000## {-e_r-}0xC0000001FFFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000180000000## {-e_r-}0x000000037FFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0x8000000000000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0xC0000000A0000000## {-e_r-}0x00000000A0000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x8000000000000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000300000001## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x8000000000000000## {-d-}0x8000000000000001##  {-e_q-}0x0000000300000000## {-e_r-}0x7FFFFFFD00000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x8000000000000000## {-d-}0x8000000080000000##  {-e_q-}0x00000002FFFFFFFE## {-e_r-}0x0000000100000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x8000000000000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000002FFFFFFFB## {-e_r-}0x00000007FFFFFFFB##
  , T {-n1-}0x0000000180000000## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000180000002## {-e_r-}0x0000000200000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000180000002## {-e_r-}0x000000007FFFFFFE##
  , T {-n1-}0x0000000180000000## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000180000001## {-e_r-}0x4000000080000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000180000000## {-e_r-}0x8000000180000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x8000000000000001## {-d-}0x00000001FFFFFFFF##  {-e_q-}0xC0000000A0000000## {-e_r-}0x00000000A0000001##
  , T {-n1-}0x0000000180000000## {-n2-}0x8000000000000001## {-d-}0x8000000000000000##  {-e_q-}0x0000000300000001## {-e_r-}0x0000000000000001##
  , T {-n1-}0x0000000180000000## {-n2-}0x8000000000000001## {-d-}0x8000000000000001##  {-e_q-}0x0000000300000000## {-e_r-}0x7FFFFFFD00000001##
  , T {-n1-}0x0000000180000000## {-n2-}0x8000000000000001## {-d-}0x8000000080000000##  {-e_q-}0x00000002FFFFFFFE## {-e_r-}0x0000000100000001##
  , T {-n1-}0x0000000180000000## {-n2-}0x8000000000000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000002FFFFFFFB## {-e_r-}0x00000007FFFFFFFC##
  , T {-n1-}0x0000000180000000## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000180000002## {-e_r-}0x0000000200000001##
  , T {-n1-}0x0000000180000000## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000180000002## {-e_r-}0x000000007FFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000180000001## {-e_r-}0x4000000080000001##
  , T {-n1-}0x0000000180000000## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000180000000## {-e_r-}0x8000000180000001##
  , T {-n1-}0x0000000180000000## {-n2-}0x8000000080000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0xC0000000A0000000## {-e_r-}0x0000000120000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x8000000080000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000300000001## {-e_r-}0x0000000080000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x8000000080000000## {-d-}0x8000000000000001##  {-e_q-}0x0000000300000000## {-e_r-}0x7FFFFFFD80000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x8000000080000000## {-d-}0x8000000080000000##  {-e_q-}0x00000002FFFFFFFE## {-e_r-}0x0000000180000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x8000000080000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000002FFFFFFFB## {-e_r-}0x000000087FFFFFFB##
  , T {-n1-}0x0000000180000000## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000180000002## {-e_r-}0x0000000280000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000180000002## {-e_r-}0x00000000FFFFFFFE##
  , T {-n1-}0x0000000180000000## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000180000001## {-e_r-}0x4000000100000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000180000000## {-e_r-}0x8000000200000000##
  , T {-n1-}0x0000000180000000## {-n2-}0x80000000FFFFFFFF## {-d-}0x00000001FFFFFFFF##  {-e_q-}0xC0000000A0000000## {-e_r-}0x000000019FFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0x80000000FFFFFFFF## {-d-}0x8000000000000000##  {-e_q-}0x0000000300000001## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0x80000000FFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0x0000000300000000## {-e_r-}0x7FFFFFFDFFFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0x80000000FFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0x00000002FFFFFFFE## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0x80000000FFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000002FFFFFFFB## {-e_r-}0x00000008FFFFFFFA##
  , T {-n1-}0x0000000180000000## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000180000002## {-e_r-}0x00000002FFFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000180000002## {-e_r-}0x000000017FFFFFFD##
  , T {-n1-}0x0000000180000000## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000180000001## {-e_r-}0x400000017FFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000180000000## {-e_r-}0x800000027FFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFF00000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0xC0000000DFFFFFFF## {-e_r-}0x00000001DFFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFF00000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000300000001## {-e_r-}0x7FFFFFFF00000000##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFF00000000## {-d-}0x8000000000000001##  {-e_q-}0x0000000300000001## {-e_r-}0x7FFFFFFBFFFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFF00000000## {-d-}0x8000000080000000##  {-e_q-}0x00000002FFFFFFFE## {-e_r-}0x8000000000000000##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFF00000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000002FFFFFFFC## {-e_r-}0x00000005FFFFFFFC##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000180000002## {-e_r-}0x8000000100000000##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000180000002## {-e_r-}0x7FFFFFFF7FFFFFFE##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000180000001## {-e_r-}0xBFFFFFFF80000000##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000180000001## {-e_r-}0x0000000080000001##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFF00000001## {-d-}0x00000001FFFFFFFF##  {-e_q-}0xC0000000DFFFFFFF## {-e_r-}0x00000001E0000000##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFF00000001## {-d-}0x8000000000000000##  {-e_q-}0x0000000300000001## {-e_r-}0x7FFFFFFF00000001##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFF00000001## {-d-}0x8000000000000001##  {-e_q-}0x0000000300000001## {-e_r-}0x7FFFFFFC00000000##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFF00000001## {-d-}0x8000000080000000##  {-e_q-}0x00000002FFFFFFFE## {-e_r-}0x8000000000000001##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFF00000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000002FFFFFFFC## {-e_r-}0x00000005FFFFFFFD##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000180000002## {-e_r-}0x8000000100000001##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000180000002## {-e_r-}0x7FFFFFFF7FFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000180000001## {-e_r-}0xBFFFFFFF80000001##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000180000001## {-e_r-}0x0000000080000002##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFF80000000## {-d-}0x00000001FFFFFFFF##  {-e_q-}0xC0000000E0000000## {-e_r-}0x0000000060000000##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFF80000000## {-d-}0x8000000000000000##  {-e_q-}0x0000000300000001## {-e_r-}0x7FFFFFFF80000000##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFF80000000## {-d-}0x8000000000000001##  {-e_q-}0x0000000300000001## {-e_r-}0x7FFFFFFC7FFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFF80000000## {-d-}0x8000000080000000##  {-e_q-}0x00000002FFFFFFFF## {-e_r-}0x0000000000000000##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFF80000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000002FFFFFFFC## {-e_r-}0x000000067FFFFFFC##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000180000002## {-e_r-}0x8000000180000000##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000180000002## {-e_r-}0x7FFFFFFFFFFFFFFE##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000180000001## {-e_r-}0xC000000000000000##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000180000001## {-e_r-}0x0000000100000001##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x00000001FFFFFFFF##  {-e_q-}0xC0000000E0000000## {-e_r-}0x00000000DFFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x8000000000000000##  {-e_q-}0x0000000300000001## {-e_r-}0x7FFFFFFFFFFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0x0000000300000001## {-e_r-}0x7FFFFFFCFFFFFFFE##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0x00000002FFFFFFFF## {-e_r-}0x000000007FFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000002FFFFFFFC## {-e_r-}0x00000006FFFFFFFB##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000180000002## {-e_r-}0x80000001FFFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000180000002## {-e_r-}0x800000007FFFFFFD##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000180000001## {-e_r-}0xC00000007FFFFFFF##
  , T {-n1-}0x0000000180000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000180000001## {-e_r-}0x0000000180000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000000000000## {-d-}0x8000000000000000##  {-e_q-}0x00000003FFFFFFFE## {-e_r-}0x0000000000000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000000000000## {-d-}0x8000000000000001##  {-e_q-}0x00000003FFFFFFFD## {-e_r-}0x7FFFFFFC00000003##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000000000000## {-d-}0x8000000080000000##  {-e_q-}0x00000003FFFFFFFA## {-e_r-}0x0000000300000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000000000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000003FFFFFFF6## {-e_r-}0x0000000DFFFFFFF6##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000200000001## {-e_r-}0x0000000100000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000200000000## {-e_r-}0xFFFFFFFE00000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000200000000## {-e_r-}0x0000000000000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000000000001## {-d-}0x8000000000000000##  {-e_q-}0x00000003FFFFFFFE## {-e_r-}0x0000000000000001##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000000000001## {-d-}0x8000000000000001##  {-e_q-}0x00000003FFFFFFFD## {-e_r-}0x7FFFFFFC00000004##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000000000001## {-d-}0x8000000080000000##  {-e_q-}0x00000003FFFFFFFA## {-e_r-}0x0000000300000001##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000000000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000003FFFFFFF6## {-e_r-}0x0000000DFFFFFFF7##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000200000001## {-e_r-}0x0000000100000001##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000200000000## {-e_r-}0xFFFFFFFE00000001##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000200000000## {-e_r-}0x0000000000000001##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x0000000200000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000080000000## {-d-}0x8000000000000000##  {-e_q-}0x00000003FFFFFFFE## {-e_r-}0x0000000080000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000080000000## {-d-}0x8000000000000001##  {-e_q-}0x00000003FFFFFFFD## {-e_r-}0x7FFFFFFC80000003##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000080000000## {-d-}0x8000000080000000##  {-e_q-}0x00000003FFFFFFFA## {-e_r-}0x0000000380000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000080000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000003FFFFFFF6## {-e_r-}0x0000000E7FFFFFF6##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000200000001## {-e_r-}0x0000000180000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000200000000## {-e_r-}0xFFFFFFFE80000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000200000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x000000027FFFFFFF##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x00000000FFFFFFFF## {-d-}0x8000000000000000##  {-e_q-}0x00000003FFFFFFFE## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x00000000FFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0x00000003FFFFFFFD## {-e_r-}0x7FFFFFFD00000002##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x00000000FFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0x00000003FFFFFFFA## {-e_r-}0x00000003FFFFFFFF##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x00000000FFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000003FFFFFFF6## {-e_r-}0x0000000EFFFFFFF5##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000200000001## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000200000000## {-e_r-}0xFFFFFFFEFFFFFFFF##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000200000000## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x00000002FFFFFFFE##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000100000000## {-d-}0x8000000000000000##  {-e_q-}0x00000003FFFFFFFE## {-e_r-}0x0000000100000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000100000000## {-d-}0x8000000000000001##  {-e_q-}0x00000003FFFFFFFD## {-e_r-}0x7FFFFFFD00000003##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000100000000## {-d-}0x8000000080000000##  {-e_q-}0x00000003FFFFFFFA## {-e_r-}0x0000000400000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000100000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000003FFFFFFF6## {-e_r-}0x0000000EFFFFFFF6##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000200000001## {-e_r-}0x0000000200000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000200000000## {-e_r-}0xFFFFFFFF00000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000200000000## {-e_r-}0x0000000100000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x00000002FFFFFFFF##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000100000001## {-d-}0x8000000000000000##  {-e_q-}0x00000003FFFFFFFE## {-e_r-}0x0000000100000001##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000100000001## {-d-}0x8000000000000001##  {-e_q-}0x00000003FFFFFFFD## {-e_r-}0x7FFFFFFD00000004##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000100000001## {-d-}0x8000000080000000##  {-e_q-}0x00000003FFFFFFFA## {-e_r-}0x0000000400000001##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000100000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000003FFFFFFF6## {-e_r-}0x0000000EFFFFFFF7##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000200000001## {-e_r-}0x0000000200000001##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000200000001## {-e_r-}0x0000000000000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000200000000## {-e_r-}0x0000000100000001##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x0000000300000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000180000000## {-d-}0x8000000000000000##  {-e_q-}0x00000003FFFFFFFE## {-e_r-}0x0000000180000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000180000000## {-d-}0x8000000000000001##  {-e_q-}0x00000003FFFFFFFD## {-e_r-}0x7FFFFFFD80000003##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000180000000## {-d-}0x8000000080000000##  {-e_q-}0x00000003FFFFFFFA## {-e_r-}0x0000000480000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000180000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000003FFFFFFF6## {-e_r-}0x0000000F7FFFFFF6##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000200000001## {-e_r-}0x0000000280000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000200000001## {-e_r-}0x000000007FFFFFFF##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000200000000## {-e_r-}0x0000000180000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x000000037FFFFFFF##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x00000001FFFFFFFF## {-d-}0x8000000000000000##  {-e_q-}0x00000003FFFFFFFE## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x00000001FFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0x00000003FFFFFFFD## {-e_r-}0x7FFFFFFE00000002##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x00000001FFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0x00000003FFFFFFFA## {-e_r-}0x00000004FFFFFFFF##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x00000001FFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000003FFFFFFF6## {-e_r-}0x0000000FFFFFFFF5##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000200000001## {-e_r-}0x00000002FFFFFFFF##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000200000001## {-e_r-}0x00000000FFFFFFFE##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000200000000## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x00000003FFFFFFFE##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x8000000000000000## {-d-}0x8000000000000000##  {-e_q-}0x00000003FFFFFFFF## {-e_r-}0x0000000000000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x8000000000000000## {-d-}0x8000000000000001##  {-e_q-}0x00000003FFFFFFFE## {-e_r-}0x7FFFFFFC00000002##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x8000000000000000## {-d-}0x8000000080000000##  {-e_q-}0x00000003FFFFFFFB## {-e_r-}0x0000000280000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x8000000000000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000003FFFFFFF7## {-e_r-}0x0000000CFFFFFFF7##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000200000001## {-e_r-}0x8000000100000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000200000001## {-e_r-}0x7FFFFFFEFFFFFFFF##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000200000000## {-e_r-}0x8000000000000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x80000001FFFFFFFF##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x8000000000000001## {-d-}0x8000000000000000##  {-e_q-}0x00000003FFFFFFFF## {-e_r-}0x0000000000000001##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x8000000000000001## {-d-}0x8000000000000001##  {-e_q-}0x00000003FFFFFFFE## {-e_r-}0x7FFFFFFC00000003##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x8000000000000001## {-d-}0x8000000080000000##  {-e_q-}0x00000003FFFFFFFB## {-e_r-}0x0000000280000001##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x8000000000000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000003FFFFFFF7## {-e_r-}0x0000000CFFFFFFF8##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000200000001## {-e_r-}0x8000000100000001##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000200000001## {-e_r-}0x7FFFFFFF00000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000200000000## {-e_r-}0x8000000000000001##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x8000000200000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x8000000080000000## {-d-}0x8000000000000000##  {-e_q-}0x00000003FFFFFFFF## {-e_r-}0x0000000080000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x8000000080000000## {-d-}0x8000000000000001##  {-e_q-}0x00000003FFFFFFFE## {-e_r-}0x7FFFFFFC80000002##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x8000000080000000## {-d-}0x8000000080000000##  {-e_q-}0x00000003FFFFFFFB## {-e_r-}0x0000000300000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x8000000080000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000003FFFFFFF7## {-e_r-}0x0000000D7FFFFFF7##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000200000001## {-e_r-}0x8000000180000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000200000001## {-e_r-}0x7FFFFFFF7FFFFFFF##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000200000000## {-e_r-}0x8000000080000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x800000027FFFFFFF##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x80000000FFFFFFFF## {-d-}0x8000000000000000##  {-e_q-}0x00000003FFFFFFFF## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x80000000FFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0x00000003FFFFFFFE## {-e_r-}0x7FFFFFFD00000001##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x80000000FFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0x00000003FFFFFFFB## {-e_r-}0x000000037FFFFFFF##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x80000000FFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000003FFFFFFF7## {-e_r-}0x0000000DFFFFFFF6##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000200000001## {-e_r-}0x80000001FFFFFFFF##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000200000001## {-e_r-}0x7FFFFFFFFFFFFFFE##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000200000000## {-e_r-}0x80000000FFFFFFFF##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x00000001FFFFFFFF## {-e_r-}0x80000002FFFFFFFE##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFF00000000## {-d-}0x8000000000000000##  {-e_q-}0x00000003FFFFFFFF## {-e_r-}0x7FFFFFFF00000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFF00000000## {-d-}0x8000000000000001##  {-e_q-}0x00000003FFFFFFFF## {-e_r-}0x7FFFFFFB00000001##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFF00000000## {-d-}0x8000000080000000##  {-e_q-}0x00000003FFFFFFFC## {-e_r-}0x0000000100000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFF00000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000003FFFFFFF8## {-e_r-}0x0000000AFFFFFFF8##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000200000002## {-e_r-}0x0000000100000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000200000001## {-e_r-}0xFFFFFFFDFFFFFFFF##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000200000000## {-e_r-}0xFFFFFFFF00000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000200000000## {-e_r-}0x0000000100000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFF00000001## {-d-}0x8000000000000000##  {-e_q-}0x00000003FFFFFFFF## {-e_r-}0x7FFFFFFF00000001##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFF00000001## {-d-}0x8000000000000001##  {-e_q-}0x00000003FFFFFFFF## {-e_r-}0x7FFFFFFB00000002##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFF00000001## {-d-}0x8000000080000000##  {-e_q-}0x00000003FFFFFFFC## {-e_r-}0x0000000100000001##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFF00000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000003FFFFFFF8## {-e_r-}0x0000000AFFFFFFF9##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000200000002## {-e_r-}0x0000000100000001##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000200000001## {-e_r-}0xFFFFFFFE00000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000200000000## {-e_r-}0xFFFFFFFF00000001##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000200000000## {-e_r-}0x0000000100000001##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFF80000000## {-d-}0x8000000000000000##  {-e_q-}0x00000003FFFFFFFF## {-e_r-}0x7FFFFFFF80000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFF80000000## {-d-}0x8000000000000001##  {-e_q-}0x00000003FFFFFFFF## {-e_r-}0x7FFFFFFB80000001##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFF80000000## {-d-}0x8000000080000000##  {-e_q-}0x00000003FFFFFFFC## {-e_r-}0x0000000180000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFF80000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000003FFFFFFF8## {-e_r-}0x0000000B7FFFFFF8##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000200000002## {-e_r-}0x0000000180000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000200000001## {-e_r-}0xFFFFFFFE7FFFFFFF##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000200000001## {-e_r-}0x0000000000000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000200000000## {-e_r-}0x0000000180000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x8000000000000000##  {-e_q-}0x00000003FFFFFFFF## {-e_r-}0x7FFFFFFFFFFFFFFF##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0x00000003FFFFFFFF## {-e_r-}0x7FFFFFFC00000000##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0x00000003FFFFFFFC## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0x00000003FFFFFFF8## {-e_r-}0x0000000BFFFFFFF7##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x0000000200000002## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x0000000200000001## {-e_r-}0xFFFFFFFEFFFFFFFE##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x0000000200000001## {-e_r-}0x000000007FFFFFFF##
  , T {-n1-}0x00000001FFFFFFFF## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x0000000200000000## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000000000000## {-d-}0x8000000000000001##  {-e_q-}0xFFFFFFFFFFFFFFFE## {-e_r-}0x0000000000000002##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000000000000## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000000## {-e_r-}0x8000000000000000##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000000000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000005## {-e_r-}0x7FFFFFF900000005##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000000## {-e_r-}0x8000000000000000##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x800000007FFFFFFF## {-e_r-}0xFFFFFFFE80000001##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000000## {-e_r-}0x2000000000000000##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000000## {-e_r-}0x8000000000000000##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000000000001## {-d-}0x8000000000000001##  {-e_q-}0xFFFFFFFFFFFFFFFE## {-e_r-}0x0000000000000003##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000000000001## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000000## {-e_r-}0x8000000000000001##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000000000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000005## {-e_r-}0x7FFFFFF900000006##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000000## {-e_r-}0x8000000000000001##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x800000007FFFFFFF## {-e_r-}0xFFFFFFFE80000002##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000000## {-e_r-}0x2000000000000001##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000000## {-e_r-}0x8000000000000001##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000080000000## {-d-}0x8000000000000001##  {-e_q-}0xFFFFFFFFFFFFFFFE## {-e_r-}0x0000000080000002##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000080000000## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0x0000000000000000##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000080000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000005## {-e_r-}0x7FFFFFF980000005##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000000## {-e_r-}0x8000000080000000##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000080000000## {-e_r-}0x0000000000000000##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000000## {-e_r-}0x2000000080000000##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000000## {-e_r-}0x8000000080000000##
  , T {-n1-}0x8000000000000000## {-n2-}0x00000000FFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0xFFFFFFFFFFFFFFFE## {-e_r-}0x0000000100000001##
  , T {-n1-}0x8000000000000000## {-n2-}0x00000000FFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0x000000007FFFFFFF##
  , T {-n1-}0x8000000000000000## {-n2-}0x00000000FFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000005## {-e_r-}0x7FFFFFFA00000004##
  , T {-n1-}0x8000000000000000## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000000## {-e_r-}0x80000000FFFFFFFF##
  , T {-n1-}0x8000000000000000## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000080000000## {-e_r-}0x000000007FFFFFFF##
  , T {-n1-}0x8000000000000000## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000000## {-e_r-}0x20000000FFFFFFFF##
  , T {-n1-}0x8000000000000000## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000000## {-e_r-}0x80000000FFFFFFFF##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000100000000## {-d-}0x8000000000000001##  {-e_q-}0xFFFFFFFFFFFFFFFE## {-e_r-}0x0000000100000002##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000100000000## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0x0000000080000000##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000100000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000005## {-e_r-}0x7FFFFFFA00000005##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000000## {-e_r-}0x8000000100000000##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000080000000## {-e_r-}0x0000000080000000##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000000## {-e_r-}0x2000000100000000##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000000## {-e_r-}0x8000000100000000##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000100000001## {-d-}0x8000000000000001##  {-e_q-}0xFFFFFFFFFFFFFFFE## {-e_r-}0x0000000100000003##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000100000001## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0x0000000080000001##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000100000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000005## {-e_r-}0x7FFFFFFA00000006##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000000## {-e_r-}0x8000000100000001##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000080000000## {-e_r-}0x0000000080000001##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000000## {-e_r-}0x2000000100000001##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000000## {-e_r-}0x8000000100000001##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000180000000## {-d-}0x8000000000000001##  {-e_q-}0xFFFFFFFFFFFFFFFE## {-e_r-}0x0000000180000002##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000180000000## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0x0000000100000000##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000180000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000005## {-e_r-}0x7FFFFFFA80000005##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000000## {-e_r-}0x8000000180000000##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000080000000## {-e_r-}0x0000000100000000##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000000## {-e_r-}0x2000000180000000##
  , T {-n1-}0x8000000000000000## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000000## {-e_r-}0x8000000180000000##
  , T {-n1-}0x8000000000000000## {-n2-}0x00000001FFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0xFFFFFFFFFFFFFFFE## {-e_r-}0x0000000200000001##
  , T {-n1-}0x8000000000000000## {-n2-}0x00000001FFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0x000000017FFFFFFF##
  , T {-n1-}0x8000000000000000## {-n2-}0x00000001FFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000005## {-e_r-}0x7FFFFFFB00000004##
  , T {-n1-}0x8000000000000000## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000000## {-e_r-}0x80000001FFFFFFFF##
  , T {-n1-}0x8000000000000000## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000080000000## {-e_r-}0x000000017FFFFFFF##
  , T {-n1-}0x8000000000000000## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000000## {-e_r-}0x20000001FFFFFFFF##
  , T {-n1-}0x8000000000000000## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000000## {-e_r-}0x80000001FFFFFFFF##
  , T {-n1-}0x8000000000000000## {-n2-}0x8000000000000000## {-d-}0x8000000000000001##  {-e_q-}0xFFFFFFFFFFFFFFFF## {-e_r-}0x0000000000000001##
  , T {-n1-}0x8000000000000000## {-n2-}0x8000000000000000## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0x7FFFFFFF80000000##
  , T {-n1-}0x8000000000000000## {-n2-}0x8000000000000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000006## {-e_r-}0x7FFFFFF800000006##
  , T {-n1-}0x8000000000000000## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000001## {-e_r-}0x0000000100000000##
  , T {-n1-}0x8000000000000000## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000080000000## {-e_r-}0x7FFFFFFF80000000##
  , T {-n1-}0x8000000000000000## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000000## {-e_r-}0xA000000000000000##
  , T {-n1-}0x8000000000000000## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000001## {-e_r-}0x0000000000000001##
  , T {-n1-}0x8000000000000000## {-n2-}0x8000000000000001## {-d-}0x8000000000000001##  {-e_q-}0xFFFFFFFFFFFFFFFF## {-e_r-}0x0000000000000002##
  , T {-n1-}0x8000000000000000## {-n2-}0x8000000000000001## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0x7FFFFFFF80000001##
  , T {-n1-}0x8000000000000000## {-n2-}0x8000000000000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000006## {-e_r-}0x7FFFFFF800000007##
  , T {-n1-}0x8000000000000000## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000001## {-e_r-}0x0000000100000001##
  , T {-n1-}0x8000000000000000## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000080000000## {-e_r-}0x7FFFFFFF80000001##
  , T {-n1-}0x8000000000000000## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000000## {-e_r-}0xA000000000000001##
  , T {-n1-}0x8000000000000000## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000001## {-e_r-}0x0000000000000002##
  , T {-n1-}0x8000000000000000## {-n2-}0x8000000080000000## {-d-}0x8000000000000001##  {-e_q-}0xFFFFFFFFFFFFFFFF## {-e_r-}0x0000000080000001##
  , T {-n1-}0x8000000000000000## {-n2-}0x8000000080000000## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0x8000000000000000##
  , T {-n1-}0x8000000000000000## {-n2-}0x8000000080000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000006## {-e_r-}0x7FFFFFF880000006##
  , T {-n1-}0x8000000000000000## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000001## {-e_r-}0x0000000180000000##
  , T {-n1-}0x8000000000000000## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000080000000## {-e_r-}0x8000000000000000##
  , T {-n1-}0x8000000000000000## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000000## {-e_r-}0xA000000080000000##
  , T {-n1-}0x8000000000000000## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000001## {-e_r-}0x0000000080000001##
  , T {-n1-}0x8000000000000000## {-n2-}0x80000000FFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0xFFFFFFFFFFFFFFFF## {-e_r-}0x0000000100000000##
  , T {-n1-}0x8000000000000000## {-n2-}0x80000000FFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0x800000007FFFFFFF##
  , T {-n1-}0x8000000000000000## {-n2-}0x80000000FFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000006## {-e_r-}0x7FFFFFF900000005##
  , T {-n1-}0x8000000000000000## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000001## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x8000000000000000## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000080000000## {-e_r-}0x800000007FFFFFFF##
  , T {-n1-}0x8000000000000000## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000000## {-e_r-}0xA0000000FFFFFFFF##
  , T {-n1-}0x8000000000000000## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000001## {-e_r-}0x0000000100000000##
  , T {-n1-}0x8000000000000000## {-n2-}0xFFFFFFFF00000000## {-d-}0x8000000000000001##  {-e_q-}0xFFFFFFFFFFFFFFFF## {-e_r-}0x7FFFFFFF00000001##
  , T {-n1-}0x8000000000000000## {-n2-}0xFFFFFFFF00000000## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000002## {-e_r-}0x7FFFFFFE00000000##
  , T {-n1-}0x8000000000000000## {-n2-}0xFFFFFFFF00000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000007## {-e_r-}0x7FFFFFF600000007##
  , T {-n1-}0x8000000000000000## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000001## {-e_r-}0x8000000000000000##
  , T {-n1-}0x8000000000000000## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000080000000## {-e_r-}0xFFFFFFFE80000000##
  , T {-n1-}0x8000000000000000## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000001## {-e_r-}0x1FFFFFFF80000000##
  , T {-n1-}0x8000000000000000## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000001## {-e_r-}0x7FFFFFFF00000001##
  , T {-n1-}0x8000000000000000## {-n2-}0xFFFFFFFF00000001## {-d-}0x8000000000000001##  {-e_q-}0xFFFFFFFFFFFFFFFF## {-e_r-}0x7FFFFFFF00000002##
  , T {-n1-}0x8000000000000000## {-n2-}0xFFFFFFFF00000001## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000002## {-e_r-}0x7FFFFFFE00000001##
  , T {-n1-}0x8000000000000000## {-n2-}0xFFFFFFFF00000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000007## {-e_r-}0x7FFFFFF600000008##
  , T {-n1-}0x8000000000000000## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000001## {-e_r-}0x8000000000000001##
  , T {-n1-}0x8000000000000000## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000080000000## {-e_r-}0xFFFFFFFE80000001##
  , T {-n1-}0x8000000000000000## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000001## {-e_r-}0x1FFFFFFF80000001##
  , T {-n1-}0x8000000000000000## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000001## {-e_r-}0x7FFFFFFF00000002##
  , T {-n1-}0x8000000000000000## {-n2-}0xFFFFFFFF80000000## {-d-}0x8000000000000001##  {-e_q-}0xFFFFFFFFFFFFFFFF## {-e_r-}0x7FFFFFFF80000001##
  , T {-n1-}0x8000000000000000## {-n2-}0xFFFFFFFF80000000## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000002## {-e_r-}0x7FFFFFFE80000000##
  , T {-n1-}0x8000000000000000## {-n2-}0xFFFFFFFF80000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000007## {-e_r-}0x7FFFFFF680000007##
  , T {-n1-}0x8000000000000000## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000001## {-e_r-}0x8000000080000000##
  , T {-n1-}0x8000000000000000## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000080000000## {-e_r-}0xFFFFFFFF00000000##
  , T {-n1-}0x8000000000000000## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000001## {-e_r-}0x2000000000000000##
  , T {-n1-}0x8000000000000000## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000001## {-e_r-}0x7FFFFFFF80000001##
  , T {-n1-}0x8000000000000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x8000000000000001##  {-e_q-}0xFFFFFFFFFFFFFFFF## {-e_r-}0x8000000000000000##
  , T {-n1-}0x8000000000000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000002## {-e_r-}0x7FFFFFFEFFFFFFFF##
  , T {-n1-}0x8000000000000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000007## {-e_r-}0x7FFFFFF700000006##
  , T {-n1-}0x8000000000000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000001## {-e_r-}0x80000000FFFFFFFF##
  , T {-n1-}0x8000000000000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000080000001## {-e_r-}0x000000007FFFFFFE##
  , T {-n1-}0x8000000000000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000001## {-e_r-}0x200000007FFFFFFF##
  , T {-n1-}0x8000000000000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000001## {-e_r-}0x8000000000000000##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000000000000## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000002## {-e_r-}0x7FFFFFFF00000000##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000000000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000007## {-e_r-}0x7FFFFFF700000007##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000001## {-e_r-}0x8000000100000000##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000080000001## {-e_r-}0x000000007FFFFFFF##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000001## {-e_r-}0x2000000080000000##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000001## {-e_r-}0x8000000000000001##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000000000001## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000002## {-e_r-}0x7FFFFFFF00000001##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000000000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000007## {-e_r-}0x7FFFFFF700000008##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000001## {-e_r-}0x8000000100000001##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000080000001## {-e_r-}0x0000000080000000##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000001## {-e_r-}0x2000000080000001##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000001## {-e_r-}0x8000000000000002##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000080000000## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000002## {-e_r-}0x7FFFFFFF80000000##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000080000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000007## {-e_r-}0x7FFFFFF780000007##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000001## {-e_r-}0x8000000180000000##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000080000001## {-e_r-}0x00000000FFFFFFFF##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000001## {-e_r-}0x2000000100000000##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000001## {-e_r-}0x8000000080000001##
  , T {-n1-}0x8000000000000001## {-n2-}0x00000000FFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000002## {-e_r-}0x7FFFFFFFFFFFFFFF##
  , T {-n1-}0x8000000000000001## {-n2-}0x00000000FFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000007## {-e_r-}0x7FFFFFF800000006##
  , T {-n1-}0x8000000000000001## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000001## {-e_r-}0x80000001FFFFFFFF##
  , T {-n1-}0x8000000000000001## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000080000001## {-e_r-}0x000000017FFFFFFE##
  , T {-n1-}0x8000000000000001## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000001## {-e_r-}0x200000017FFFFFFF##
  , T {-n1-}0x8000000000000001## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000001## {-e_r-}0x8000000100000000##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000100000000## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000002## {-e_r-}0x8000000000000000##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000100000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000007## {-e_r-}0x7FFFFFF800000007##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000001## {-e_r-}0x8000000200000000##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000080000001## {-e_r-}0x000000017FFFFFFF##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000001## {-e_r-}0x2000000180000000##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000001## {-e_r-}0x8000000100000001##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000100000001## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000002## {-e_r-}0x8000000000000001##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000100000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000007## {-e_r-}0x7FFFFFF800000008##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000001## {-e_r-}0x8000000200000001##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000080000001## {-e_r-}0x0000000180000000##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000001## {-e_r-}0x2000000180000001##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000001## {-e_r-}0x8000000100000002##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000180000000## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000003## {-e_r-}0x0000000000000000##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000180000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000007## {-e_r-}0x7FFFFFF880000007##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000001## {-e_r-}0x8000000280000000##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000080000001## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000001## {-e_r-}0x2000000200000000##
  , T {-n1-}0x8000000000000001## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000001## {-e_r-}0x8000000180000001##
  , T {-n1-}0x8000000000000001## {-n2-}0x00000001FFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000003## {-e_r-}0x000000007FFFFFFF##
  , T {-n1-}0x8000000000000001## {-n2-}0x00000001FFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000007## {-e_r-}0x7FFFFFF900000006##
  , T {-n1-}0x8000000000000001## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000001## {-e_r-}0x80000002FFFFFFFF##
  , T {-n1-}0x8000000000000001## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000080000001## {-e_r-}0x000000027FFFFFFE##
  , T {-n1-}0x8000000000000001## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000001## {-e_r-}0x200000027FFFFFFF##
  , T {-n1-}0x8000000000000001## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000001## {-e_r-}0x8000000200000000##
  , T {-n1-}0x8000000000000001## {-n2-}0x8000000000000000## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000003## {-e_r-}0x7FFFFFFE80000000##
  , T {-n1-}0x8000000000000001## {-n2-}0x8000000000000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000008## {-e_r-}0x7FFFFFF600000008##
  , T {-n1-}0x8000000000000001## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000002## {-e_r-}0x0000000200000000##
  , T {-n1-}0x8000000000000001## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000080000001## {-e_r-}0x800000007FFFFFFF##
  , T {-n1-}0x8000000000000001## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000001## {-e_r-}0xA000000080000000##
  , T {-n1-}0x8000000000000001## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000002## {-e_r-}0x0000000000000002##
  , T {-n1-}0x8000000000000001## {-n2-}0x8000000000000001## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000003## {-e_r-}0x7FFFFFFE80000001##
  , T {-n1-}0x8000000000000001## {-n2-}0x8000000000000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000008## {-e_r-}0x7FFFFFF600000009##
  , T {-n1-}0x8000000000000001## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000002## {-e_r-}0x0000000200000001##
  , T {-n1-}0x8000000000000001## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000080000001## {-e_r-}0x8000000080000000##
  , T {-n1-}0x8000000000000001## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000001## {-e_r-}0xA000000080000001##
  , T {-n1-}0x8000000000000001## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000002## {-e_r-}0x0000000000000003##
  , T {-n1-}0x8000000000000001## {-n2-}0x8000000080000000## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000003## {-e_r-}0x7FFFFFFF00000000##
  , T {-n1-}0x8000000000000001## {-n2-}0x8000000080000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000008## {-e_r-}0x7FFFFFF680000008##
  , T {-n1-}0x8000000000000001## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000002## {-e_r-}0x0000000280000000##
  , T {-n1-}0x8000000000000001## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000080000001## {-e_r-}0x80000000FFFFFFFF##
  , T {-n1-}0x8000000000000001## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000001## {-e_r-}0xA000000100000000##
  , T {-n1-}0x8000000000000001## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000002## {-e_r-}0x0000000080000002##
  , T {-n1-}0x8000000000000001## {-n2-}0x80000000FFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000003## {-e_r-}0x7FFFFFFF7FFFFFFF##
  , T {-n1-}0x8000000000000001## {-n2-}0x80000000FFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000008## {-e_r-}0x7FFFFFF700000007##
  , T {-n1-}0x8000000000000001## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000002## {-e_r-}0x00000002FFFFFFFF##
  , T {-n1-}0x8000000000000001## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000080000001## {-e_r-}0x800000017FFFFFFE##
  , T {-n1-}0x8000000000000001## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000001## {-e_r-}0xA00000017FFFFFFF##
  , T {-n1-}0x8000000000000001## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000002## {-e_r-}0x0000000100000001##
  , T {-n1-}0x8000000000000001## {-n2-}0xFFFFFFFF00000000## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000004## {-e_r-}0x7FFFFFFD00000000##
  , T {-n1-}0x8000000000000001## {-n2-}0xFFFFFFFF00000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000009## {-e_r-}0x7FFFFFF400000009##
  , T {-n1-}0x8000000000000001## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000002## {-e_r-}0x8000000100000000##
  , T {-n1-}0x8000000000000001## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000080000002## {-e_r-}0x000000007FFFFFFE##
  , T {-n1-}0x8000000000000001## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000002## {-e_r-}0x2000000000000000##
  , T {-n1-}0x8000000000000001## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000002## {-e_r-}0x7FFFFFFF00000002##
  , T {-n1-}0x8000000000000001## {-n2-}0xFFFFFFFF00000001## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000004## {-e_r-}0x7FFFFFFD00000001##
  , T {-n1-}0x8000000000000001## {-n2-}0xFFFFFFFF00000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000009## {-e_r-}0x7FFFFFF40000000A##
  , T {-n1-}0x8000000000000001## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000002## {-e_r-}0x8000000100000001##
  , T {-n1-}0x8000000000000001## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000080000002## {-e_r-}0x000000007FFFFFFF##
  , T {-n1-}0x8000000000000001## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000002## {-e_r-}0x2000000000000001##
  , T {-n1-}0x8000000000000001## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000002## {-e_r-}0x7FFFFFFF00000003##
  , T {-n1-}0x8000000000000001## {-n2-}0xFFFFFFFF80000000## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000004## {-e_r-}0x7FFFFFFD80000000##
  , T {-n1-}0x8000000000000001## {-n2-}0xFFFFFFFF80000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000009## {-e_r-}0x7FFFFFF480000009##
  , T {-n1-}0x8000000000000001## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000002## {-e_r-}0x8000000180000000##
  , T {-n1-}0x8000000000000001## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000080000002## {-e_r-}0x00000000FFFFFFFE##
  , T {-n1-}0x8000000000000001## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000002## {-e_r-}0x2000000080000000##
  , T {-n1-}0x8000000000000001## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000002## {-e_r-}0x7FFFFFFF80000002##
  , T {-n1-}0x8000000000000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x8000000080000000##  {-e_q-}0xFFFFFFFF00000004## {-e_r-}0x7FFFFFFDFFFFFFFF##
  , T {-n1-}0x8000000000000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFE00000009## {-e_r-}0x7FFFFFF500000008##
  , T {-n1-}0x8000000000000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000080000002## {-e_r-}0x80000001FFFFFFFF##
  , T {-n1-}0x8000000000000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000080000002## {-e_r-}0x000000017FFFFFFD##
  , T {-n1-}0x8000000000000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000040000002## {-e_r-}0x20000000FFFFFFFF##
  , T {-n1-}0x8000000000000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000000000002## {-e_r-}0x8000000000000001##
  , T {-n1-}0x8000000080000000## {-n2-}0x0000000000000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFF00000003## {-e_r-}0x7FFFFFFC00000003##
  , T {-n1-}0x8000000080000000## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000100000001## {-e_r-}0x0000000100000000##
  , T {-n1-}0x8000000080000000## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000100000000## {-e_r-}0x7FFFFFFF00000000##
  , T {-n1-}0x8000000080000000## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x80000000C0000000## {-e_r-}0x6000000000000000##
  , T {-n1-}0x8000000080000000## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000080000000## {-e_r-}0x8000000080000000##
  , T {-n1-}0x8000000080000000## {-n2-}0x0000000000000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFF00000003## {-e_r-}0x7FFFFFFC00000004##
  , T {-n1-}0x8000000080000000## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000100000001## {-e_r-}0x0000000100000001##
  , T {-n1-}0x8000000080000000## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000100000000## {-e_r-}0x7FFFFFFF00000001##
  , T {-n1-}0x8000000080000000## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x80000000C0000000## {-e_r-}0x6000000000000001##
  , T {-n1-}0x8000000080000000## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000080000000## {-e_r-}0x8000000080000001##
  , T {-n1-}0x8000000080000000## {-n2-}0x0000000080000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFF00000003## {-e_r-}0x7FFFFFFC80000003##
  , T {-n1-}0x8000000080000000## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000100000001## {-e_r-}0x0000000180000000##
  , T {-n1-}0x8000000080000000## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000100000000## {-e_r-}0x7FFFFFFF80000000##
  , T {-n1-}0x8000000080000000## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x80000000C0000000## {-e_r-}0x6000000080000000##
  , T {-n1-}0x8000000080000000## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000080000000## {-e_r-}0x8000000100000000##
  , T {-n1-}0x8000000080000000## {-n2-}0x00000000FFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFF00000003## {-e_r-}0x7FFFFFFD00000002##
  , T {-n1-}0x8000000080000000## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000100000001## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x8000000080000000## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000100000000## {-e_r-}0x7FFFFFFFFFFFFFFF##
  , T {-n1-}0x8000000080000000## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x80000000C0000000## {-e_r-}0x60000000FFFFFFFF##
  , T {-n1-}0x8000000080000000## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000080000000## {-e_r-}0x800000017FFFFFFF##
  , T {-n1-}0x8000000080000000## {-n2-}0x0000000100000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFF00000003## {-e_r-}0x7FFFFFFD00000003##
  , T {-n1-}0x8000000080000000## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000100000001## {-e_r-}0x0000000200000000##
  , T {-n1-}0x8000000080000000## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000100000000## {-e_r-}0x8000000000000000##
  , T {-n1-}0x8000000080000000## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x80000000C0000000## {-e_r-}0x6000000100000000##
  , T {-n1-}0x8000000080000000## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000080000000## {-e_r-}0x8000000180000000##
  , T {-n1-}0x8000000080000000## {-n2-}0x0000000100000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFF00000003## {-e_r-}0x7FFFFFFD00000004##
  , T {-n1-}0x8000000080000000## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000100000001## {-e_r-}0x0000000200000001##
  , T {-n1-}0x8000000080000000## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000100000000## {-e_r-}0x8000000000000001##
  , T {-n1-}0x8000000080000000## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x80000000C0000000## {-e_r-}0x6000000100000001##
  , T {-n1-}0x8000000080000000## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000080000000## {-e_r-}0x8000000180000001##
  , T {-n1-}0x8000000080000000## {-n2-}0x0000000180000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFF00000003## {-e_r-}0x7FFFFFFD80000003##
  , T {-n1-}0x8000000080000000## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000100000001## {-e_r-}0x0000000280000000##
  , T {-n1-}0x8000000080000000## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000100000000## {-e_r-}0x8000000080000000##
  , T {-n1-}0x8000000080000000## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x80000000C0000000## {-e_r-}0x6000000180000000##
  , T {-n1-}0x8000000080000000## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000080000000## {-e_r-}0x8000000200000000##
  , T {-n1-}0x8000000080000000## {-n2-}0x00000001FFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFF00000003## {-e_r-}0x7FFFFFFE00000002##
  , T {-n1-}0x8000000080000000## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000100000001## {-e_r-}0x00000002FFFFFFFF##
  , T {-n1-}0x8000000080000000## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000100000000## {-e_r-}0x80000000FFFFFFFF##
  , T {-n1-}0x8000000080000000## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x80000000C0000000## {-e_r-}0x60000001FFFFFFFF##
  , T {-n1-}0x8000000080000000## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000080000000## {-e_r-}0x800000027FFFFFFF##
  , T {-n1-}0x8000000080000000## {-n2-}0x8000000000000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFF00000004## {-e_r-}0x7FFFFFFB00000004##
  , T {-n1-}0x8000000080000000## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000100000001## {-e_r-}0x8000000100000000##
  , T {-n1-}0x8000000080000000## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000100000000## {-e_r-}0xFFFFFFFF00000000##
  , T {-n1-}0x8000000080000000## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x80000000C0000000## {-e_r-}0xE000000000000000##
  , T {-n1-}0x8000000080000000## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000080000001## {-e_r-}0x0000000080000001##
  , T {-n1-}0x8000000080000000## {-n2-}0x8000000000000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFF00000004## {-e_r-}0x7FFFFFFB00000005##
  , T {-n1-}0x8000000080000000## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000100000001## {-e_r-}0x8000000100000001##
  , T {-n1-}0x8000000080000000## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000100000001## {-e_r-}0x0000000000000000##
  , T {-n1-}0x8000000080000000## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x80000000C0000000## {-e_r-}0xE000000000000001##
  , T {-n1-}0x8000000080000000## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000080000001## {-e_r-}0x0000000080000002##
  , T {-n1-}0x8000000080000000## {-n2-}0x8000000080000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFF00000004## {-e_r-}0x7FFFFFFB80000004##
  , T {-n1-}0x8000000080000000## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000100000001## {-e_r-}0x8000000180000000##
  , T {-n1-}0x8000000080000000## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000100000001## {-e_r-}0x000000007FFFFFFF##
  , T {-n1-}0x8000000080000000## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x80000000C0000000## {-e_r-}0xE000000080000000##
  , T {-n1-}0x8000000080000000## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000080000001## {-e_r-}0x0000000100000001##
  , T {-n1-}0x8000000080000000## {-n2-}0x80000000FFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFF00000004## {-e_r-}0x7FFFFFFC00000003##
  , T {-n1-}0x8000000080000000## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000100000001## {-e_r-}0x80000001FFFFFFFF##
  , T {-n1-}0x8000000080000000## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000100000001## {-e_r-}0x00000000FFFFFFFE##
  , T {-n1-}0x8000000080000000## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x80000000C0000000## {-e_r-}0xE0000000FFFFFFFF##
  , T {-n1-}0x8000000080000000## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000080000001## {-e_r-}0x0000000180000000##
  , T {-n1-}0x8000000080000000## {-n2-}0xFFFFFFFF00000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFF00000005## {-e_r-}0x7FFFFFF900000005##
  , T {-n1-}0x8000000080000000## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000100000002## {-e_r-}0x0000000100000000##
  , T {-n1-}0x8000000080000000## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000100000001## {-e_r-}0x7FFFFFFEFFFFFFFF##
  , T {-n1-}0x8000000080000000## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x80000000C0000001## {-e_r-}0x5FFFFFFF80000000##
  , T {-n1-}0x8000000080000000## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000080000001## {-e_r-}0x7FFFFFFF80000001##
  , T {-n1-}0x8000000080000000## {-n2-}0xFFFFFFFF00000001## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFF00000005## {-e_r-}0x7FFFFFF900000006##
  , T {-n1-}0x8000000080000000## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000100000002## {-e_r-}0x0000000100000001##
  , T {-n1-}0x8000000080000000## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000100000001## {-e_r-}0x7FFFFFFF00000000##
  , T {-n1-}0x8000000080000000## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x80000000C0000001## {-e_r-}0x5FFFFFFF80000001##
  , T {-n1-}0x8000000080000000## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000080000001## {-e_r-}0x7FFFFFFF80000002##
  , T {-n1-}0x8000000080000000## {-n2-}0xFFFFFFFF80000000## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFF00000005## {-e_r-}0x7FFFFFF980000005##
  , T {-n1-}0x8000000080000000## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000100000002## {-e_r-}0x0000000180000000##
  , T {-n1-}0x8000000080000000## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000100000001## {-e_r-}0x7FFFFFFF7FFFFFFF##
  , T {-n1-}0x8000000080000000## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x80000000C0000001## {-e_r-}0x6000000000000000##
  , T {-n1-}0x8000000080000000## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000080000001## {-e_r-}0x8000000000000001##
  , T {-n1-}0x8000000080000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0x80000000FFFFFFFF##  {-e_q-}0xFFFFFFFF00000005## {-e_r-}0x7FFFFFFA00000004##
  , T {-n1-}0x8000000080000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000100000002## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x8000000080000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000100000001## {-e_r-}0x7FFFFFFFFFFFFFFE##
  , T {-n1-}0x8000000080000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x80000000C0000001## {-e_r-}0x600000007FFFFFFF##
  , T {-n1-}0x8000000080000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000080000001## {-e_r-}0x8000000080000000##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000180000000## {-e_r-}0x8000000000000000##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x800000017FFFFFFF## {-e_r-}0xFFFFFFFD80000001##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x800000013FFFFFFF## {-e_r-}0x9FFFFFFF80000000##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x80000000FFFFFFFF## {-e_r-}0x80000000FFFFFFFF##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000180000000## {-e_r-}0x8000000000000001##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x800000017FFFFFFF## {-e_r-}0xFFFFFFFD80000002##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x800000013FFFFFFF## {-e_r-}0x9FFFFFFF80000001##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x80000000FFFFFFFF## {-e_r-}0x8000000100000000##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000180000000## {-e_r-}0x8000000080000000##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x800000017FFFFFFF## {-e_r-}0xFFFFFFFE00000001##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x800000013FFFFFFF## {-e_r-}0xA000000000000000##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x80000000FFFFFFFF## {-e_r-}0x800000017FFFFFFF##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000180000000## {-e_r-}0x80000000FFFFFFFF##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x800000017FFFFFFF## {-e_r-}0xFFFFFFFE80000000##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x800000013FFFFFFF## {-e_r-}0xA00000007FFFFFFF##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x80000000FFFFFFFF## {-e_r-}0x80000001FFFFFFFE##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000180000000## {-e_r-}0x8000000100000000##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x800000017FFFFFFF## {-e_r-}0xFFFFFFFE80000001##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x800000013FFFFFFF## {-e_r-}0xA000000080000000##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x80000000FFFFFFFF## {-e_r-}0x80000001FFFFFFFF##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000180000000## {-e_r-}0x8000000100000001##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x800000017FFFFFFF## {-e_r-}0xFFFFFFFE80000002##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x800000013FFFFFFF## {-e_r-}0xA000000080000001##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x80000000FFFFFFFF## {-e_r-}0x8000000200000000##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000180000000## {-e_r-}0x8000000180000000##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000180000000## {-e_r-}0x0000000000000000##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x800000013FFFFFFF## {-e_r-}0xA000000100000000##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x80000000FFFFFFFF## {-e_r-}0x800000027FFFFFFF##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000180000000## {-e_r-}0x80000001FFFFFFFF##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000180000000## {-e_r-}0x000000007FFFFFFF##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x800000013FFFFFFF## {-e_r-}0xA00000017FFFFFFF##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x80000000FFFFFFFF## {-e_r-}0x80000002FFFFFFFE##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000180000001## {-e_r-}0x0000000100000000##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000180000000## {-e_r-}0x7FFFFFFE80000000##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000140000000## {-e_r-}0x2000000000000000##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000100000000## {-e_r-}0x0000000100000000##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000180000001## {-e_r-}0x0000000100000001##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000180000000## {-e_r-}0x7FFFFFFE80000001##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000140000000## {-e_r-}0x2000000000000001##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000100000000## {-e_r-}0x0000000100000001##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000180000001## {-e_r-}0x0000000180000000##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000180000000## {-e_r-}0x7FFFFFFF00000000##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000140000000## {-e_r-}0x2000000080000000##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000100000000## {-e_r-}0x0000000180000000##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000180000001## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000180000000## {-e_r-}0x7FFFFFFF7FFFFFFF##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000140000000## {-e_r-}0x20000000FFFFFFFF##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000100000000## {-e_r-}0x00000001FFFFFFFF##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000180000001## {-e_r-}0x8000000000000000##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000180000000## {-e_r-}0xFFFFFFFD80000000##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000140000000## {-e_r-}0x9FFFFFFF00000000##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000100000000## {-e_r-}0x8000000000000000##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000180000001## {-e_r-}0x8000000000000001##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000180000000## {-e_r-}0xFFFFFFFD80000001##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000140000000## {-e_r-}0x9FFFFFFF00000001##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000100000000## {-e_r-}0x8000000000000001##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000180000001## {-e_r-}0x8000000080000000##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000180000000## {-e_r-}0xFFFFFFFE00000000##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000140000000## {-e_r-}0x9FFFFFFF80000000##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000100000000## {-e_r-}0x8000000080000000##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF00000000##  {-e_q-}0x8000000180000001## {-e_r-}0x80000000FFFFFFFF##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0x8000000180000000## {-e_r-}0xFFFFFFFE7FFFFFFF##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0x8000000140000000## {-e_r-}0x9FFFFFFFFFFFFFFF##
  , T {-n1-}0x80000000FFFFFFFF## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0x8000000100000000## {-e_r-}0x80000000FFFFFFFF##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0xFFFFFFFFFFFFFFFE## {-e_r-}0xFFFFFFFE00000002##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF7FFFFFFF## {-e_r-}0xBFFFFFFF80000000##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000000## {-e_r-}0xFFFFFFFF00000000##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0xFFFFFFFFFFFFFFFE## {-e_r-}0xFFFFFFFE00000003##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF7FFFFFFF## {-e_r-}0xBFFFFFFF80000001##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000000## {-e_r-}0xFFFFFFFF00000001##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0xFFFFFFFFFFFFFFFE## {-e_r-}0xFFFFFFFE80000002##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF7FFFFFFF## {-e_r-}0xC000000000000000##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000000## {-e_r-}0xFFFFFFFF80000000##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0xFFFFFFFFFFFFFFFF## {-e_r-}0x0000000000000000##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF7FFFFFFF## {-e_r-}0xC00000007FFFFFFF##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0x0000000000000000##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0xFFFFFFFFFFFFFFFF## {-e_r-}0x0000000000000001##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF7FFFFFFF## {-e_r-}0xC000000080000000##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0x0000000000000001##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0xFFFFFFFFFFFFFFFF## {-e_r-}0x0000000000000002##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF7FFFFFFF## {-e_r-}0xC000000080000001##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0x0000000000000002##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0xFFFFFFFFFFFFFFFF## {-e_r-}0x0000000080000001##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF7FFFFFFF## {-e_r-}0xC000000100000000##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0x0000000080000001##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0xFFFFFFFFFFFFFFFF## {-e_r-}0x0000000100000000##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF7FFFFFFF## {-e_r-}0xC00000017FFFFFFF##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0x0000000100000000##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0xFFFFFFFFFFFFFFFF## {-e_r-}0x7FFFFFFF00000001##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF80000000## {-e_r-}0x4000000000000000##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0x7FFFFFFF00000001##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0xFFFFFFFFFFFFFFFF## {-e_r-}0x7FFFFFFF00000002##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF80000000## {-e_r-}0x4000000000000001##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0x7FFFFFFF00000002##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0xFFFFFFFFFFFFFFFF## {-e_r-}0x7FFFFFFF80000001##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF80000000## {-e_r-}0x4000000080000000##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0x7FFFFFFF80000001##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0xFFFFFFFFFFFFFFFF## {-e_r-}0x8000000000000000##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF80000000## {-e_r-}0x40000000FFFFFFFF##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0x8000000000000000##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0xFFFFFFFFFFFFFFFF## {-e_r-}0xFFFFFFFE00000001##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF80000000## {-e_r-}0xBFFFFFFF00000000##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0xFFFFFFFE00000001##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF00000001##  {-e_q-}0xFFFFFFFFFFFFFFFF## {-e_r-}0xFFFFFFFE00000002##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF80000000## {-e_r-}0xBFFFFFFF00000001##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0xFFFFFFFE00000002##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF00000001##  {-e_q-}0xFFFFFFFFFFFFFFFF## {-e_r-}0xFFFFFFFE80000001##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF80000000## {-e_r-}0xBFFFFFFF80000000##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0xFFFFFFFE80000001##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF00000001##  {-e_q-}0xFFFFFFFFFFFFFFFF## {-e_r-}0xFFFFFFFF00000000##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF80000000## {-e_r-}0xBFFFFFFFFFFFFFFF##
  , T {-n1-}0xFFFFFFFF00000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0xFFFFFFFF00000000##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF80000000## {-e_r-}0xC000000000000000##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0xFFFFFFFF00000001##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF80000000## {-e_r-}0xC000000000000001##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0xFFFFFFFF00000002##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF80000000## {-e_r-}0xC000000080000000##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000001## {-e_r-}0xFFFFFFFF80000001##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF80000000## {-e_r-}0xC0000000FFFFFFFF##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000002## {-e_r-}0x0000000000000001##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF80000000## {-e_r-}0xC000000100000000##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000002## {-e_r-}0x0000000000000002##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF80000000## {-e_r-}0xC000000100000001##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000002## {-e_r-}0x0000000000000003##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF80000000## {-e_r-}0xC000000180000000##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000002## {-e_r-}0x0000000080000002##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF80000000## {-e_r-}0xC0000001FFFFFFFF##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000002## {-e_r-}0x0000000100000001##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF80000001## {-e_r-}0x4000000080000000##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000002## {-e_r-}0x7FFFFFFF00000002##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF80000001## {-e_r-}0x4000000080000001##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000002## {-e_r-}0x7FFFFFFF00000003##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF80000001## {-e_r-}0x4000000100000000##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000002## {-e_r-}0x7FFFFFFF80000002##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF80000001## {-e_r-}0x400000017FFFFFFF##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000002## {-e_r-}0x8000000000000001##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF80000001## {-e_r-}0xBFFFFFFF80000000##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000002## {-e_r-}0xFFFFFFFE00000002##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF80000001## {-e_r-}0xBFFFFFFF80000001##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000002## {-e_r-}0xFFFFFFFE00000003##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF80000001## {-e_r-}0xC000000000000000##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000002## {-e_r-}0xFFFFFFFE80000002##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFF80000000##  {-e_q-}0xFFFFFFFF80000001## {-e_r-}0xC00000007FFFFFFF##
  , T {-n1-}0xFFFFFFFF00000001## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF00000002## {-e_r-}0xFFFFFFFF00000001##
  , T {-n1-}0xFFFFFFFF80000000## {-n2-}0x0000000000000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF80000000## {-e_r-}0xFFFFFFFF80000000##
  , T {-n1-}0xFFFFFFFF80000000## {-n2-}0x0000000000000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF80000000## {-e_r-}0xFFFFFFFF80000001##
  , T {-n1-}0xFFFFFFFF80000000## {-n2-}0x0000000080000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF80000001## {-e_r-}0x0000000000000001##
  , T {-n1-}0xFFFFFFFF80000000## {-n2-}0x00000000FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF80000001## {-e_r-}0x0000000080000000##
  , T {-n1-}0xFFFFFFFF80000000## {-n2-}0x0000000100000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF80000001## {-e_r-}0x0000000080000001##
  , T {-n1-}0xFFFFFFFF80000000## {-n2-}0x0000000100000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF80000001## {-e_r-}0x0000000080000002##
  , T {-n1-}0xFFFFFFFF80000000## {-n2-}0x0000000180000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF80000001## {-e_r-}0x0000000100000001##
  , T {-n1-}0xFFFFFFFF80000000## {-n2-}0x00000001FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF80000001## {-e_r-}0x0000000180000000##
  , T {-n1-}0xFFFFFFFF80000000## {-n2-}0x8000000000000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF80000001## {-e_r-}0x7FFFFFFF80000001##
  , T {-n1-}0xFFFFFFFF80000000## {-n2-}0x8000000000000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF80000001## {-e_r-}0x7FFFFFFF80000002##
  , T {-n1-}0xFFFFFFFF80000000## {-n2-}0x8000000080000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF80000001## {-e_r-}0x8000000000000001##
  , T {-n1-}0xFFFFFFFF80000000## {-n2-}0x80000000FFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF80000001## {-e_r-}0x8000000080000000##
  , T {-n1-}0xFFFFFFFF80000000## {-n2-}0xFFFFFFFF00000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF80000001## {-e_r-}0xFFFFFFFE80000001##
  , T {-n1-}0xFFFFFFFF80000000## {-n2-}0xFFFFFFFF00000001## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF80000001## {-e_r-}0xFFFFFFFE80000002##
  , T {-n1-}0xFFFFFFFF80000000## {-n2-}0xFFFFFFFF80000000## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF80000001## {-e_r-}0xFFFFFFFF00000001##
  , T {-n1-}0xFFFFFFFF80000000## {-n2-}0xFFFFFFFFFFFFFFFF## {-d-}0xFFFFFFFFFFFFFFFF##  {-e_q-}0xFFFFFFFF80000001## {-e_r-}0xFFFFFFFF80000000##
#endif
  ]
