{-
HOpenGL - a binding of OpenGL and GLUT for Haskell.
Copyright (C) 2000  Sven Panne <Sven.Panne@BetaResearch.de>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library (COPYING.LIB); if not, write to the Free
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

This module corresponds to chapters 2 (Inititalization) and 3 (Beginning
Event Processing) of the GLUT3 API docs.
-}

module GLUT_Init (
   GLUT_Init.init, initWindowPosition, initWindowSize,
   DisplayMode(..), initDisplayMode,
   marshalDisplayMode,                                           -- internal use only
   BooleanDisplayCapability(..), NumericDisplayCapability(..),   -- @glut_geq_4_9@
   Comparator(..), DisplaySetting(..), initDisplay,              -- @glut_geq_4_9@
   mainLoop,
   createWindow
) where

import Prelude          hiding (init)
import System           (getProgName, getArgs)
import List             (intersperse, genericLength)
import Monad            (unless)
import CForeign         (CInt, CUInt, CString, withCString, peekCString)
import Foreign          (Ptr, nullPtr, Storable(peek), withObject, withMany,
                         withArray0, peekArray)

import GL_BasicTypes    (WindowPosition(..), WindowSize(..), toBitfield)
import GLUT_Constants   (glut_RGB, glut_RGBA, glut_INDEX, glut_SINGLE,
                         glut_DOUBLE, glut_ACCUM, glut_ALPHA, glut_DEPTH,
                         glut_STENCIL, glut_MULTISAMPLE, glut_STEREO,
                         glut_LUMINANCE)
import GLUT_Window      (Window(..))
import GLUT_CBWindow    (DisplayAction, displayFunc)

---------------------------------------------------------------------------

init :: Maybe (String, [String]) -> IO (String, [String])
init Nothing =  do
   prog <- getProgName
   args <- getArgs
   GLUT_Init.init $ Just (prog, args)
init (Just (prog, args)) =
   withObject (1 + genericLength args) $ \argcBuf ->
   withMany withCString (prog : args) $ \argvPtrs ->
   withArray0 nullPtr argvPtrs $ \argvBuf -> do
   glutInit argcBuf argvBuf
   newArgc <- peek argcBuf
   newArgvPtrs <- peekArray (fromIntegral newArgc) argvBuf
   newArgv <- mapM peekCString argvPtrs
   return (head newArgv, tail newArgv)

foreign import "glutInit" unsafe glutInit :: Ptr CInt -> Ptr CString -> IO ()

---------------------------------------------------------------------------

initWindowPosition :: WindowPosition -> IO ()
initWindowPosition (WindowPosition x y) = glutInitWindowPosition (fromIntegral x) (fromIntegral y)

foreign import "glutInitWindowPosition" unsafe glutInitWindowPosition :: CInt -> CInt -> IO ()

initWindowSize :: WindowSize -> IO ()
initWindowSize (WindowSize w h) = glutInitWindowSize (fromIntegral w) (fromIntegral h)

foreign import "glutInitWindowSize" unsafe glutInitWindowSize :: CInt -> CInt -> IO ()

---------------------------------------------------------------------------
-- Display mode bit masks.

data DisplayMode =
     Rgb
   | Rgba
   | Index
   | Single
   | Double
   | Accum
   | Alpha
   | Depth
   | Stencil
   | Multisample
   | Stereo
   | Luminance   -- @glut_api_geq3@
   deriving (Eq,Ord,Enum,Bounded)

marshalDisplayMode :: DisplayMode -> CUInt
marshalDisplayMode Rgb         = glut_RGB
marshalDisplayMode Rgba        = glut_RGBA
marshalDisplayMode Index       = glut_INDEX
marshalDisplayMode Single      = glut_SINGLE
marshalDisplayMode Double      = glut_DOUBLE
marshalDisplayMode Accum       = glut_ACCUM
marshalDisplayMode Alpha       = glut_ALPHA
marshalDisplayMode Depth       = glut_DEPTH
marshalDisplayMode Stencil     = glut_STENCIL
marshalDisplayMode Multisample = glut_MULTISAMPLE
marshalDisplayMode Stereo      = glut_STEREO
marshalDisplayMode Luminance   = glut_LUMINANCE   -- @glut_api_geq3@

initDisplayMode :: [DisplayMode] -> IO ()
initDisplayMode = glutInitDisplayMode . toBitfield marshalDisplayMode

foreign import "glutInitDisplayMode" unsafe glutInitDisplayMode :: CUInt -> IO ()

---------------------------------------------------------------------------

-- @glut_geq_4_9@
data BooleanDisplayCapability =
     Conformant 
   | Double'
   | Index'
   | Single'
   | Slow
   | Stereo'
   | Xdirectcolor
   | Xgrayscale
   | Xpseudocolor
   | Xstaticcolor
   | Xstaticgray
   | Xtruecolor

-- @glut_geq_4_9@
instance Show BooleanDisplayCapability where
   showsPrec _ Conformant   = showString "conformant" 
   showsPrec _ Double'      = showString "double"
   showsPrec _ Index'       = showString "index"
   showsPrec _ Single'      = showString "single"
   showsPrec _ Slow         = showString "slow"
   showsPrec _ Stereo'      = showString "stereo"
   showsPrec _ Xdirectcolor = showString "xdirectcolor"
   showsPrec _ Xgrayscale   = showString "xgrayscale"
   showsPrec _ Xpseudocolor = showString "xpseudocolor"
   showsPrec _ Xstaticcolor = showString "xstaticcolor"
   showsPrec _ Xstaticgray  = showString "xstaticgray"
   showsPrec _ Xtruecolor   = showString "xtruecolor"

-- NOTE: num, win32pfd, and xvisual *must* have a comparator and a value,
-- but we don't care here...
-- @glut_geq_4_9@
data NumericDisplayCapability =
     Acc
   | Acca
   | Alpha'
   | Blue'
   | Buffer
   | Depth'
   | Green'
   | Luminance'
   | Num
   | Red'
   | Rgb'
   | Rgba'
   | Samples
   | Stencil'
   | Win32pfd
   | Xvisual

-- @glut_geq_4_9@
instance Show NumericDisplayCapability where
   showsPrec _ Acc        = showString "acc"
   showsPrec _ Acca       = showString "acca"
   showsPrec _ Alpha'     = showString "alpha"
   showsPrec _ Blue'      = showString "blue"
   showsPrec _ Buffer     = showString "buffer"
   showsPrec _ Depth'     = showString "depth"
   showsPrec _ Green'     = showString "green"
   showsPrec _ Luminance' = showString "luminance"
   showsPrec _ Num        = showString "num"
   showsPrec _ Red'       = showString "red"
   showsPrec _ Rgb'       = showString "rgb"
   showsPrec _ Rgba'      = showString "rgba"
   showsPrec _ Samples    = showString "samples"
   showsPrec _ Stencil'   = showString "stencil"
   showsPrec _ Win32pfd   = showString "win32pfd"
   showsPrec _ Xvisual    = showString "xvisual"

-- @glut_geq_4_9@
data Comparator =
     Equal
   | Notequal
   | Less
   | Greater
   | Lequal
   | Gequal
   | GequalMin

-- @glut_geq_4_9@
instance Show Comparator where
   showsPrec _ Equal     = showString "="
   showsPrec _ Notequal  = showString "!="
   showsPrec _ Less      = showString "<"
   showsPrec _ Greater   = showString ">"
   showsPrec _ Lequal    = showString "<="
   showsPrec _ Gequal    = showString ">="
   showsPrec _ GequalMin = showString "~" 

-- @glut_geq_4_9@
data DisplaySetting =
     ShouldBe    BooleanDisplayCapability
   | ShouldNotBe BooleanDisplayCapability
   | WithDefault NumericDisplayCapability
   | Want        NumericDisplayCapability Comparator Int

-- @glut_geq_4_9@
instance Show DisplaySetting where
   showsPrec _ (ShouldBe    s) = shows s . showString "=1"
   showsPrec _ (ShouldNotBe s) = shows s . showString "=0"
   showsPrec _ (WithDefault s) = shows s
   showsPrec _ (Want s c i)    = shows s . shows c . shows i

-- @glut_geq_4_9@
initDisplay :: [DisplaySetting] -> IO ()
initDisplay settings = withCString (concat . intersperse " " . map show $ settings) glutInitDisplayString

-- @glut_geq_4_9@
foreign import "glutInitDisplayString" unsafe glutInitDisplayString ::CString -> IO ()

---------------------------------------------------------------------------
-- ATTENTION: NO "unsafe" HERE!

foreign import "glutMainLoop" mainLoop :: IO ()

---------------------------------------------------------------------------

createWindow :: String -> DisplayAction -> [DisplayMode]
             -> Maybe WindowPosition -> Maybe WindowSize -> IO Window
createWindow name displayAction modes maybeXY maybeWH = do
   unless (null modes) (initDisplayMode modes)
   maybe (return ()) initWindowPosition maybeXY
   maybe (return ()) initWindowSize     maybeWH
   w <- withCString name glutCreateWindow
   displayFunc displayAction
   return w

foreign import "glutCreateWindow" unsafe glutCreateWindow :: CString -> IO Window
