{-
HOpenGL - a binding of OpenGL and GLUT for Haskell.
Copyright (C) 2001  Sven Panne <Sven.Panne@BetaResearch.de>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library (COPYING.LIB); if not, write to the Free
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-}

module GL_BinaryIO (
   readsBin,    -- :: Storable a =>                ReadSBin a
   showsBin,    -- :: Storable a =>           a -> ShowSBin
   hGetBin,     -- :: Storable a => Handle ->      IO a
   hPutBin      -- :: Storable a => Handle -> a -> IO ()
) where

import IO               ( Handle )
import IOExts           ( unsafePerformIO, hGetBuf, hPutBuf )
import Monad            ( zipWithM )
import Foreign

type ReadSBin a = [Word8] -> Maybe (a, [Word8])
type ShowSBin   = [Word8] -> [Word8]

readsBin :: Storable a => ReadSBin a
readsBin = readsBin_ undefined
   where readsBin_ :: Storable a => a -> ReadSBin a
         readsBin_ undef bytes =
            let s = sizeOf undef
            in if hasNElems s bytes
                  then unsafePerformIO $ alloca $ \buf -> do
                       zipWithM (pokeByteOff buf) [0 .. s-1] bytes
                       val <- peek buf
                       return $ Just (val, drop s bytes)
                  else Nothing

hasNElems :: Int -> [a] -> Bool
hasNElems 0 _      = True
hasNElems _ []     = False
hasNElems n (_:xs) = hasNElems (n-1) xs

showsBin :: Storable a => a -> ShowSBin
showsBin x xs = unsafePerformIO (withObject x (peekNOnto xs (sizeOf x)))
   where peekNOnto bs 0 _   = return bs
         peekNOnto bs i buf = do b <- peekByteOff buf (i-1)
                                 peekNOnto (b:bs) (i-1) buf

hGetBin :: Storable a => Handle -> IO a
hGetBin h = hGetBin_ undefined
   where hGetBin_ :: Storable a => a -> IO a
         hGetBin_ undef = alloca $ \buf -> do
                          let s = sizeOf undef
                          bytesRead <- hGetBuf h buf s
                          if bytesRead < s
                             then ioError (userError "too short")
                             else peek buf

hPutBin :: Storable a => Handle -> a -> IO ()
hPutBin handle val = withObject val $ \buf ->
                     hPutBuf handle buf (sizeOf val)
