{-
HOpenGL - a binding of OpenGL and GLUT for Haskell.
Copyright (C) 2000  Sven Panne <Sven.Panne@BetaResearch.de>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library (COPYING.LIB); if not, write to the Free
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

This module corresponds to section 3.5 (Polygons) of the OpenGL 1.2.1 specs.
-}

module GL_Polygons (
   PolygonSmooth(..),
   cullFace, CullFace(..),
   StipplePattern, stipplePattern, polygonStippleSize,
   polygonStipple, PolygonStipple(..),
   PolygonMode(..),
   marshalPolygonMode, unmarshalPolygonMode,   -- internal use only
   polygonMode,
   polygonOffset, PolygonOffset(..)
) where

import Foreign          (Ptr, withArray)

import GL_Constants     (gl_POLYGON_SMOOTH, gl_CULL_FACE, gl_POLYGON_STIPPLE,
                         gl_POINT, gl_LINE, gl_FILL, gl_POLYGON_OFFSET_POINT,
                         gl_POLYGON_OFFSET_LINE, gl_POLYGON_OFFSET_FILL)
import GL_BasicTypes    (GLenum, GLubyte, GLfloat, Capability(..))
import GL_Colors        (Face, marshalFace)

---------------------------------------------------------------------------

data PolygonSmooth =
     PolygonSmooth
   deriving (Eq,Ord)

instance Capability PolygonSmooth where
   marshalCapability PolygonSmooth = gl_POLYGON_SMOOTH

---------------------------------------------------------------------------
-- Section 3.5.1 (Basic Polygon Rasterization)

cullFace :: Face -> IO ()
cullFace = glCullFace . marshalFace 

foreign import "glCullFace" unsafe glCullFace :: GLenum -> IO ()

data CullFace =
     CullFace
   deriving (Eq,Ord)

instance Capability CullFace where
   marshalCapability CullFace = gl_CULL_FACE

---------------------------------------------------------------------------
-- Section 3.5.2 (Stippling)

-- the list must have length polygonStippleSize
newtype StipplePattern = StipplePattern [GLubyte]

stipplePattern :: [GLubyte] -> StipplePattern
stipplePattern = StipplePattern

polygonStippleSize :: Int
polygonStippleSize = 32*32 `div` 8

polygonStipple :: StipplePattern -> IO ()
polygonStipple (StipplePattern sp) = withArray sp glPolygonStipple

foreign import "glPolygonStipple" unsafe glPolygonStipple :: Ptr GLubyte -> IO ()

data PolygonStipple =
     PolygonStipple
     deriving (Eq,Ord)

instance Capability PolygonStipple where
   marshalCapability PolygonStipple = gl_POLYGON_STIPPLE

---------------------------------------------------------------------------
-- Section 3.5.4 (Options Controlling Polygon Rasterization)

data PolygonMode =
     Point
   | Line
   | Fill
   deriving (Eq,Ord)

marshalPolygonMode :: PolygonMode -> GLenum
marshalPolygonMode Point = gl_POINT
marshalPolygonMode Line  = gl_LINE
marshalPolygonMode Fill  = gl_FILL

unmarshalPolygonMode :: GLenum -> PolygonMode
unmarshalPolygonMode polygonMode
   | polygonMode == gl_POINT = Point
   | polygonMode == gl_LINE  = Line
   | polygonMode == gl_FILL  = Fill
   | otherwise               = error "unmarshalPolygonMode"

polygonMode :: Face -> PolygonMode -> IO ()
polygonMode face = glPolygonMode (marshalFace face) . marshalPolygonMode

foreign import "glPolygonMode" unsafe glPolygonMode :: GLenum -> GLenum -> IO ()

---------------------------------------------------------------------------
-- Section 3.5.5 (Depth Offset)

foreign import "glPolygonOffset" unsafe polygonOffset :: GLfloat -> GLfloat -> IO ()

data PolygonOffset =
     PolygonOffsetPoint
   | PolygonOffsetLine
   | PolygonOffsetFill
   deriving (Eq,Ord)

instance Capability PolygonOffset where
   marshalCapability PolygonOffsetPoint = gl_POLYGON_OFFSET_POINT
   marshalCapability PolygonOffsetLine  = gl_POLYGON_OFFSET_LINE
   marshalCapability PolygonOffsetFill  = gl_POLYGON_OFFSET_FILL
