{-
HOpenGL - a binding of OpenGL and GLUT for Haskell.
Copyright (C) 2001  Sven Panne <Sven.Panne@BetaResearch.de>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library (COPYING.LIB); if not, write to the Free
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

This module corresponds to section 2.8 (Vertex Arrays) of the OpenGL 1.2.1 specs.
-}

module GL_VertexArray(
   Type(..),
   marshalType, unmarshalType,   -- internal use only
   Stride,
   edgeFlagPointer, texCoordPointer, colorPointer,
   indexPointer, normalPointer, vertexPointer,
   ClientCapability(..), enableClientState, disableClientState,
   arrayElement, drawArrays, drawElements,
   drawRangeElements,            -- @GL_1_2@
   InterleavedArraysFormat(..), interleavedArrays
) where

import Foreign          (Ptr)

import GL_Constants     (gl_BYTE, gl_UNSIGNED_BYTE, gl_SHORT, gl_UNSIGNED_SHORT,
                         gl_INT, gl_UNSIGNED_INT, gl_FLOAT, gl_DOUBLE,
                         gl_2_BYTES, gl_3_BYTES, gl_4_BYTES, gl_BITMAP,
                         gl_UNSIGNED_BYTE_3_3_2, gl_UNSIGNED_BYTE_2_3_3_REV,
                         gl_UNSIGNED_SHORT_5_6_5, gl_UNSIGNED_SHORT_5_6_5_REV,
                         gl_UNSIGNED_SHORT_4_4_4_4, gl_UNSIGNED_SHORT_4_4_4_4_REV,
                         gl_UNSIGNED_SHORT_5_5_5_1, gl_UNSIGNED_SHORT_1_5_5_5_REV,
                         gl_UNSIGNED_INT_8_8_8_8, gl_UNSIGNED_INT_8_8_8_8_REV,
                         gl_UNSIGNED_INT_10_10_10_2, gl_UNSIGNED_INT_2_10_10_10_REV,
                         gl_V2F, gl_V3F, gl_C4UB_V2F, gl_C4UB_V3F, gl_C3F_V3F,
                         gl_N3F_V3F, gl_C4F_N3F_V3F, gl_T2F_V3F, gl_T4F_V4F,
                         gl_T2F_C4UB_V3F, gl_T2F_C3F_V3F, gl_T2F_N3F_V3F,
                         gl_T2F_C4F_N3F_V3F, gl_T4F_C4F_N3F_V4F,
                         gl_EDGE_FLAG_ARRAY, gl_TEXTURE_COORD_ARRAY, gl_COLOR_ARRAY,
                         gl_INDEX_ARRAY, gl_NORMAL_ARRAY, gl_VERTEX_ARRAY)
import GL_BasicTypes    (GLenum, GLsizei, GLint, GLuint)
import GL_BeginEnd      (VertexMode, marshalVertexMode)

---------------------------------------------------------------------------

data Type =
     Byte
   | UnsignedByte
   | Short
   | UnsignedShort
   | Int
   | UnsignedInt
   | Float
   | Double
   | TwoBytes
   | ThreeBytes
   | FourBytes
   | Bitmap
   | UnsignedByte332
   | UnsignedByte233Rev
   | UnsignedShort565
   | UnsignedShort565Rev
   | UnsignedShort4444
   | UnsignedShort4444Rev
   | UnsignedShort5551
   | UnsignedShort1555Rev
   | UnsignedInt8888
   | UnsignedInt8888Rev
   | UnsignedInt1010102
   | UnsignedInt2101010Rev
   deriving (Eq,Ord)

marshalType :: Type -> GLenum
marshalType Byte                  = gl_BYTE
marshalType UnsignedByte          = gl_UNSIGNED_BYTE
marshalType Short                 = gl_SHORT
marshalType UnsignedShort         = gl_UNSIGNED_SHORT
marshalType Int                   = gl_INT
marshalType UnsignedInt           = gl_UNSIGNED_INT
marshalType Float                 = gl_FLOAT
marshalType Double                = gl_DOUBLE
marshalType TwoBytes              = gl_2_BYTES
marshalType ThreeBytes            = gl_3_BYTES
marshalType FourBytes             = gl_4_BYTES
marshalType Bitmap                = gl_BITMAP
marshalType UnsignedByte332       = gl_UNSIGNED_BYTE_3_3_2
marshalType UnsignedByte233Rev    = gl_UNSIGNED_BYTE_2_3_3_REV
marshalType UnsignedShort565      = gl_UNSIGNED_SHORT_5_6_5
marshalType UnsignedShort565Rev   = gl_UNSIGNED_SHORT_5_6_5_REV
marshalType UnsignedShort4444     = gl_UNSIGNED_SHORT_4_4_4_4
marshalType UnsignedShort4444Rev  = gl_UNSIGNED_SHORT_4_4_4_4_REV
marshalType UnsignedShort5551     = gl_UNSIGNED_SHORT_5_5_5_1
marshalType UnsignedShort1555Rev  = gl_UNSIGNED_SHORT_1_5_5_5_REV
marshalType UnsignedInt8888       = gl_UNSIGNED_INT_8_8_8_8
marshalType UnsignedInt8888Rev    = gl_UNSIGNED_INT_8_8_8_8_REV
marshalType UnsignedInt1010102    = gl_UNSIGNED_INT_10_10_10_2
marshalType UnsignedInt2101010Rev = gl_UNSIGNED_INT_2_10_10_10_REV

unmarshalType :: GLenum -> Type
unmarshalType t
   | t == gl_BYTE                        = Byte
   | t == gl_UNSIGNED_BYTE               = UnsignedByte
   | t == gl_SHORT                       = Short
   | t == gl_UNSIGNED_SHORT              = UnsignedShort
   | t == gl_INT                         = Int
   | t == gl_UNSIGNED_INT                = UnsignedInt
   | t == gl_FLOAT                       = Float
   | t == gl_DOUBLE                      = Double
   | t == gl_2_BYTES                     = TwoBytes
   | t == gl_3_BYTES                     = ThreeBytes
   | t == gl_4_BYTES                     = FourBytes
   | t == gl_BITMAP                      = Bitmap
   | t == gl_UNSIGNED_BYTE_3_3_2         = UnsignedByte332
   | t == gl_UNSIGNED_BYTE_2_3_3_REV     = UnsignedByte233Rev
   | t == gl_UNSIGNED_SHORT_5_6_5        = UnsignedShort565
   | t == gl_UNSIGNED_SHORT_5_6_5_REV    = UnsignedShort565Rev
   | t == gl_UNSIGNED_SHORT_4_4_4_4      = UnsignedShort4444
   | t == gl_UNSIGNED_SHORT_4_4_4_4_REV  = UnsignedShort4444Rev
   | t == gl_UNSIGNED_SHORT_5_5_5_1      = UnsignedShort5551
   | t == gl_UNSIGNED_SHORT_1_5_5_5_REV  = UnsignedShort1555Rev
   | t == gl_UNSIGNED_INT_8_8_8_8        = UnsignedInt8888
   | t == gl_UNSIGNED_INT_8_8_8_8_REV    = UnsignedInt8888Rev
   | t == gl_UNSIGNED_INT_10_10_10_2     = UnsignedInt1010102
   | t == gl_UNSIGNED_INT_2_10_10_10_REV = UnsignedInt2101010Rev
   | otherwise                           = error "unmarshalType"

type Stride = GLsizei

foreign import "glEdgeFlagPointer" unsafe edgeFlagPointer :: Stride -> Ptr a -> IO ()

texCoordPointer :: GLint -> Type -> Stride -> Ptr a -> IO ()
texCoordPointer size = glTexCoordPointer size . marshalType

foreign import "glTexCoordPointer" unsafe glTexCoordPointer :: GLint -> GLenum -> Stride -> Ptr a -> IO ()

colorPointer :: GLint -> Type -> Stride -> Ptr a -> IO ()
colorPointer size = glColorPointer size . marshalType

foreign import "glColorPointer" unsafe glColorPointer :: GLint -> GLenum -> Stride -> Ptr a -> IO ()

indexPointer :: Type -> Stride -> Ptr a -> IO ()
indexPointer = glIndexPointer . marshalType

foreign import "glIndexPointer" unsafe glIndexPointer :: GLenum -> Stride -> Ptr a -> IO ()

normalPointer :: Type -> Stride -> Ptr a -> IO ()
normalPointer = glNormalPointer . marshalType

foreign import "glNormalPointer" unsafe glNormalPointer :: GLenum -> Stride -> Ptr a -> IO ()

vertexPointer :: GLint -> Type -> Stride -> Ptr a -> IO ()
vertexPointer size = glVertexPointer size . marshalType

foreign import "glVertexPointer" unsafe glVertexPointer :: GLint -> GLenum -> Stride -> Ptr a -> IO ()

data ClientCapability =
     EdgeFlagArray
   | TextureCoordArray
   | ColorArray
   | IndexArray
   | NormalArray
   | VertexArray
   deriving (Eq,Ord)

marshalClientCapability :: ClientCapability -> GLenum
marshalClientCapability EdgeFlagArray     = gl_EDGE_FLAG_ARRAY
marshalClientCapability TextureCoordArray = gl_TEXTURE_COORD_ARRAY
marshalClientCapability ColorArray        = gl_COLOR_ARRAY
marshalClientCapability IndexArray        = gl_INDEX_ARRAY
marshalClientCapability NormalArray       = gl_NORMAL_ARRAY
marshalClientCapability VertexArray       = gl_VERTEX_ARRAY

enableClientState  :: ClientCapability -> IO ()
enableClientState = glEnableClientState . marshalClientCapability

foreign import "glEnableClientState" unsafe glEnableClientState  :: GLenum -> IO ()

disableClientState  :: ClientCapability -> IO ()
disableClientState = glDisableClientState . marshalClientCapability

foreign import "glDisableClientState" unsafe glDisableClientState :: GLenum -> IO ()

foreign import "glArrayElement" unsafe arrayElement :: GLint -> IO ()

drawArrays :: VertexMode -> GLint -> GLsizei -> IO ()
drawArrays = glDrawArrays . marshalVertexMode

foreign import "glDrawArrays" unsafe glDrawArrays :: GLenum -> GLint -> GLsizei -> IO ()

drawElements :: VertexMode -> GLsizei -> Type -> Ptr a -> IO ()
drawElements mode count = glDrawElements (marshalVertexMode mode) count . marshalType

foreign import "glDrawElements" unsafe glDrawElements :: GLenum -> GLsizei -> GLenum -> Ptr a -> IO ()

-- @GL_1_2@
drawRangeElements :: VertexMode -> GLuint -> GLuint -> GLsizei -> Type -> Ptr a -> IO ()
drawRangeElements mode start end count = glDrawRangeElements (marshalVertexMode mode) start end count . marshalType

foreign import "glDrawRangeElements" unsafe glDrawRangeElements :: GLenum -> GLuint -> GLuint -> GLsizei -> GLenum -> Ptr a -> IO ()

data InterleavedArraysFormat =
     V2f
   | V3f
   | C4ubV2f
   | C4ubV3f
   | C3fV3f
   | N3fV3f
   | C4fN3fV3f
   | T2fV3f
   | T4fV4f
   | T2fC4ubV3f
   | T2fC3fV3f
   | T2fN3fV3f
   | T2fC4fN3fV3f
   | T4fC4fN3fV4f
   deriving (Eq,Ord)

marshalInterleavedArraysFormat :: InterleavedArraysFormat -> GLenum
marshalInterleavedArraysFormat V2f          = gl_V2F
marshalInterleavedArraysFormat V3f          = gl_V3F
marshalInterleavedArraysFormat C4ubV2f      = gl_C4UB_V2F
marshalInterleavedArraysFormat C4ubV3f      = gl_C4UB_V3F
marshalInterleavedArraysFormat C3fV3f       = gl_C3F_V3F
marshalInterleavedArraysFormat N3fV3f       = gl_N3F_V3F
marshalInterleavedArraysFormat C4fN3fV3f    = gl_C4F_N3F_V3F
marshalInterleavedArraysFormat T2fV3f       = gl_T2F_V3F
marshalInterleavedArraysFormat T4fV4f       = gl_T4F_V4F
marshalInterleavedArraysFormat T2fC4ubV3f   = gl_T2F_C4UB_V3F
marshalInterleavedArraysFormat T2fC3fV3f    = gl_T2F_C3F_V3F
marshalInterleavedArraysFormat T2fN3fV3f    = gl_T2F_N3F_V3F
marshalInterleavedArraysFormat T2fC4fN3fV3f = gl_T2F_C4F_N3F_V3F
marshalInterleavedArraysFormat T4fC4fN3fV4f = gl_T4F_C4F_N3F_V4F

interleavedArrays :: InterleavedArraysFormat -> Stride -> Ptr a -> IO ()
interleavedArrays = glInterleavedArrays . marshalInterleavedArraysFormat

foreign import "glInterleavedArrays" unsafe glInterleavedArrays :: GLenum -> Stride -> Ptr a -> IO ()
