-- -----------------------------------------------------------------------------
-- $Id: IOExts.hsc,v 1.5 2001/08/23 10:36:50 sewardj Exp $
--
-- (c) The University of Glasgow, 1994-2000
--

-- @IOExts@ provides useful functionality that fall outside the
-- standard Haskell IO interface. Expect the contents of IOExts
-- to be the same for Hugs and GHC (same goes for any other
-- Hugs/GHC extension libraries, unless a function/type is
-- explicitly flagged as being implementation specific
-- extension.)

module IOExts 
        ( fixIO 	      -- :: (a -> IO a) -> IO a
        , unsafePerformIO     -- :: IO a -> a
        , unsafeInterleaveIO  -- :: IO a -> IO a

	, module IORef

	, IOArray	      -- instance of: Eq
	, newIOArray	      -- :: Ix ix => (ix,ix) -> elt -> IO (IOArray ix elt)
	, boundsIOArray       -- :: Ix ix => IOArray ix elt -> (ix, ix)
	, readIOArray         -- :: Ix ix => IOArray ix elt -> ix -> IO elt
	, writeIOArray        -- :: Ix ix => IOArray ix elt -> ix -> elt -> IO ()
	, freezeIOArray       -- :: Ix ix => IOArray ix elt -> IO (Array ix elt)
	, thawIOArray	      -- :: Ix ix => Array ix elt -> IO (IOArray ix elt)

	, unsafeFreezeIOArray -- :: Ix ix => IOArray ix elt -> IO (Array ix elt)
	, unsafeThawIOArray   -- :: Ix ix => Array ix elt -> IO (IOArray ix elt)
	
        , trace		      -- :: String -> a -> a

	, IOModeEx(..)	      	-- instance (Eq, Read, Show)
	, openFileEx	      	-- :: FilePath -> IOModeEx -> IO Handle
	, hSetBinaryMode      	-- :: Handle -> Bool -> IO Bool
	      	
	, hGetBuf            	-- :: Handle -> Ptr a -> Int -> IO Int
	, hGetBufBA  	      	-- :: Handle -> MutableByteArray RealWorld a 
		      	      	--	-> Int -> IO Int

	, hPutBuf     	      	-- :: Handle -> Ptr a -> Int -> IO ()
	, hPutBufBA   	      	-- :: Handle -> MutableByteArray RealWorld a
		      	      	--	-> Int -> IO ()

	, hIsTerminalDevice 	-- :: Handle -> IO Bool
        , hSetEcho		-- :: Handle -> Bool -> IO ()
	, hGetEcho		-- :: Handle -> IO Bool

	, unsafePtrEq		-- :: a -> a -> Bool
	, slurpFile
        , performGC
	, freeHaskellFunctionPtr

	, HandlePosition
	, hTell                	-- :: Handle -> IO Integer

	-- extended IOError predicates
	, isHardwareFault		--  :: IOError -> Bool
	, isInappropriateType		--  :: IOError -> Bool
	, isInterrupted			--  :: IOError -> Bool
	, isInvalidArgument		--  :: IOError -> Bool
	, isOtherError			--  :: IOError -> Bool
	, isProtocolError		--  :: IOError -> Bool
	, isResourceVanished		--  :: IOError -> Bool
	, isSystemError			--  :: IOError -> Bool
	, isTimeExpired			--  :: IOError -> Bool
	, isUnsatisfiedConstraints	--  :: IOError -> Bool
	, isUnsupportedOperation	--  :: IOError -> Bool
#if defined(cygwin32_TARGET_OS) || defined(mingw32_TARGET_OS)
	, isComError			--  :: IOError -> Bool
#endif

	-- DEPRECATED:
	, hGetBufFull, hPutBufFull, hGetBufBAFull, hPutBufBAFull
        ) where

#include "HsStd.h"

import CTypes
import CTypesISO
import MarshalAlloc
import Ptr
import IORef
import ST
import IO
import Array        ( Array )
import MArray
import MutableArray
import Addr         ( Addr )

import IO           ( hPutStr, hPutChar )
import Monad	    ( liftM )

import PrelCError  -- to get our MayBlock versions
import PrelPosix
import PrelBase
import PrelConc
import PrelIOBase
import PrelHandle

import MarshalArray (peekArray)
import PrelCString  (castCCharToChar)

-- ---------------------------------------------------------------------------
-- fixIO

fixIO 		:: (a -> IO a) -> IO a
fixIO m         = stToIO (fixST (ioToST . m))

-- ---------------------------------------------------------------------------
-- unsafePtrEq

unsafePtrEq :: a -> a -> Bool
unsafePtrEq a b =
    case reallyUnsafePtrEquality## a b of
	 0## -> False
	 _  -> True 

-- ---------------------------------------------------------------------------
-- IO Arrays

newIOArray :: Ix i => (i,i) -> e -> IO (IOArray i e)
newIOArray = newArray

boundsIOArray :: Ix i => IOArray i e -> (i,i)
boundsIOArray = bounds

readIOArray :: Ix i => IOArray i e -> i -> IO e
readIOArray = readArray

writeIOArray :: Ix i => IOArray i e -> i -> e -> IO ()
writeIOArray = writeArray

freezeIOArray :: Ix i => IOArray i e -> IO (Array i e)
freezeIOArray = freeze

unsafeFreezeIOArray :: Ix i => IOArray i e -> IO (Array i e)
unsafeFreezeIOArray = unsafeFreeze

thawIOArray :: Ix i => Array i e -> IO (IOArray i e)
thawIOArray = thaw

unsafeThawIOArray :: Ix i => Array i e -> IO (IOArray i e)
unsafeThawIOArray = unsafeThaw

-- ---------------------------------------------------------------------------
-- Trace

{-# NOINLINE trace #-}
trace :: String -> a -> a
trace string expr = unsafePerformIO $ do
    hPutStr stderr string
    hPutChar stderr '\n'
    fd <- withHandle_ "trace" stderr $ (return.haFD)
    postTraceHook fd
    return expr

foreign import "PostTraceHook" postTraceHook :: Int -> IO ()

-- ---------------------------------------------------------------------------
-- performGC

-- Not something you want to call normally, but useful
-- in the cases where you do want to flush stuff out of
-- the heap or make sure you've got room enough

foreign import {-safe-} performGC :: IO ()

-- ---------------------------------------------------------------------------
-- freeHaskellFunctionPtr

-- When using 'foreign export dynamic' to dress up a Haskell
-- IO action to look like a C function pointer, a little bit
-- of memory is allocated (along with a stable pointer to
-- the Haskell IO action). When done with the C function
-- pointer, you'll need to call @freeHaskellFunctionPtr()@ to
-- let go of these resources - here's the Haskell wrapper for
-- that RTS entry point, should you want to free it from
-- within Haskell.

-- SUP: This really belongs into module Foreign, but for legacy reasons
-- we leave it here for now.

foreign import unsafe freeHaskellFunctionPtr :: Addr -> IO ()

-- ---------------------------------------------------------------------------
-- Redirecting handles

-- (Experimental) 

-- Support for redirecting I/O on a handle to another for the
-- duration of an IO action. To re-route a handle, it is first
-- flushed, followed by replacing its innards (i.e., FILE_OBJECT)
-- with that of the other. This happens before and after the
-- action is executed.

-- If the action raises an exception, the handle is replaced back
-- to its old contents, but without flushing it first - as this
-- may provoke exceptions. Notice that the action may perform
-- I/O on either Handle, with the result that the I/O is interleaved.
-- (Why you would want to do this, is a completely different matter.)

-- ToDo: probably want to restrict what kind of handles can be
-- replaced with another - i.e., don't want to be able to replace
-- a writeable handle with a readable one.

#if 0
withHandleFor :: Handle
	      -> Handle
	      -> IO a
	      -> IO a
withHandleFor h1 h2 act = do
   h1_fo <- getFO h1
   plugIn h1_fo
 where
  plugIn h1_fo = do
    hFlush h2
    h2_fo <- withHandle h2 $ \ h2_ -> return (h2_{haFO__=h1_fo}, haFO__ h2_)
    catch (act >>= \ x -> hFlush h2 >> setFO h2 h2_fo >> return x)
    	  (\ err -> setFO h2 h2_fo >> ioError err)

  setFO h fo = 
    withHandle h $ \ h_ -> return (h_{haFO__=fo}, ())

  getFO h = 
    wantRWHandle "withHandleFor" h $ \ h_ ->
    return (haFO__ h_)
#endif        

-- Derived @withHandleFor@ combinators and, at the moment, these
-- are exported from @IOExts@ and not @withHandleFor@ itself.

#if 0
withStdin, withStdout,withStderr :: Handle -> IO a -> IO a
withStdin  h a = withHandleFor h stdin  a
withStdout h a = withHandleFor h stdout a
withStderr h a = withHandleFor h stderr a
#endif

-- ---------------------------------------------------------------------------
-- hTell

-- @hTell@ is the lower-level version of @hGetPosn@ - return the
-- position, without bundling it together with the handle itself:

type HandlePosition = Integer

hTell :: Handle -> IO Integer
hTell h = do
  (HandlePosn _ x) <- hGetPosn h
  return x

-- ---------------------------------------------------------------------------
-- hSetBinaryMode

-- @hSetBinaryMode@ lets you change the translation mode for a handle.
-- On some platforms (e.g., Win32) a distinction is made between being in
-- 'text mode' or 'binary mode', with the former terminating lines
-- by \r\n rather than just \n.

-- Debating the Winnitude or otherwise of such a scheme is less than
-- interesting -- it's there, so we have to cope.

-- A side-effect of calling @hSetBinaryMode@ is that the output buffer
-- (if any) is flushed prior to changing the translation mode.

#if 0
hSetBinaryMode :: Handle -> Bool -> IO Bool
hSetBinaryMode handle is_binary = do 
        -- is_binary = True => set translation mode to binary.
    wantRWHandle "hSetBinaryMode" handle $ \ handle_ -> do
{- Win32 TODO!
    let flg' | is_binary = flg .|. (#const O_BINARY)
	     | otherwise = flg .&. complement (#const O_BINARY)
-}
    toBool `liftM` 
	throwErrnoIfMinus1 
	   (setmode (fromIntegral (haFD handle_)) flg')
#ifdef _WIN32
foreign import "setmode" setmode :: CInt -> CInt -> IO CInt
#else
setmode _ _ = return ()
#endif
#endif

-- ---------------------------------------------------------------------------
-- Reading sequences of bytes.

{-
Semantics of hGetBuf:

   - hGetBuf reads data into the buffer until either

	(a) EOF is reached
	(b) the buffer is full
     
     It returns the amount of data actually read.  This may
     be zero in case (a).  hGetBuf never raises
     an EOF exception, it always returns zero instead.

     If the handle is a pipe or socket, and the writing end
     is closed, hGetBuf will behave as for condition (a).

Semantics of hPutBuf:

    - hPutBuf writes data from the buffer to the handle 
      until the buffer is empty.  It returns ().

      If the handle is a pipe or socket, and the reading end is
      closed, hPutBuf will raise a ResourceVanished exception.
      (If this is a POSIX system, and the program has not 
      asked to ignore SIGPIPE, then a SIGPIPE may be delivered
      instead, whose default action is to terminate the program).
-}

-- ---------------------------------------------------------------------------
-- hPutBuf

hPutBuf :: Handle			-- handle to write to
	-> Ptr a			-- address of buffer
	-> Int				-- number of bytes of data in buffer
	-> IO ()
hPutBuf handle ptr count
  | count <= 0 = illegalBufferSize handle "hPutBuf" count
  | otherwise = 
    wantWritableHandle "hPutBuf" handle $ 
      \ handle_@Handle__{ haFD=fd, haBuffer=ref } -> do

        old_buf@Buffer{ bufBuf=old_raw, bufRPtr=r, bufWPtr=w, bufSize=size }
	  <- readIORef ref

        -- enough room in handle buffer?
        if (size - w > count)
		-- There's enough room in the buffer:
		-- just copy the data in and update bufWPtr.
	    then do memcpy_baoff_ptr old_raw w ptr (fromIntegral count)
		    writeIORef ref old_buf{ bufWPtr = w + count }
		    return ()

		-- else, we have to flush
	    else do flushed_buf <- flushWriteBuffer fd old_buf
		    writeIORef ref flushed_buf
		    -- ToDo: should just memcpy instead of writing if possible
		    writeChunk fd ptr count

writeChunk :: FD -> Ptr a -> Int -> IO ()
writeChunk fd ptr bytes = loop 0 bytes 
 where
  loop :: Int -> Int -> IO ()
  loop _   bytes | bytes <= 0 = return ()
  loop off bytes = do
    r <- fromIntegral `liftM`
	   throwErrnoIfMinus1RetryMayBlock "writeChunk"
	    (c_write (fromIntegral fd) (ptr `plusPtr` off) (fromIntegral bytes))
	    (threadWaitWrite fd)
    -- write can't return 0
    loop (off + r) (bytes - r)

-- ---------------------------------------------------------------------------
-- hGetBuf

hGetBuf :: Handle -> Ptr a -> Int -> IO Int
hGetBuf handle ptr count
  | count <= 0 = illegalBufferSize handle "hGetBuf" count
  | otherwise = 
      wantReadableHandle "hGetBuf" handle $ 
	\ handle_@Handle__{ haFD=fd, haBuffer=ref } -> do
	buf@Buffer{ bufBuf=raw, bufWPtr=w, bufRPtr=r } <- readIORef ref
	if bufferEmpty buf
	   then readChunk fd ptr count
	   else do 
		let avail = w - r
		copied <- if (count >= avail)
		       	    then do 
				memcpy_ptr_baoff ptr raw r (fromIntegral avail)
				writeIORef ref buf{ bufWPtr=0, bufRPtr=0 }
				return avail
		     	    else do
				memcpy_ptr_baoff ptr raw r (fromIntegral count)
				writeIORef ref buf{ bufRPtr = r + count }
				return count

		let remaining = count - copied
		if remaining > 0 
		   then do rest <- readChunk fd (ptr `plusPtr` copied) remaining
			   return (rest + count)
		   else return count
		
readChunk :: FD -> Ptr a -> Int -> IO Int
readChunk fd ptr bytes = loop 0 bytes 
 where
  loop :: Int -> Int -> IO Int
  loop off bytes | bytes <= 0 = return off
  loop off bytes = do
    r <- fromIntegral `liftM`
	   throwErrnoIfMinus1RetryMayBlock "readChunk"
	    (c_read (fromIntegral fd) (ptr `plusPtr` off) (fromIntegral bytes))
	    (threadWaitRead fd)
    if r == 0
	then return off
	else loop (off + r) (bytes - r)

slurpFile :: FilePath -> IO (Ptr (), Int)
slurpFile fname = do
  handle <- openFile fname ReadMode
  sz     <- hFileSize handle
  if sz > fromIntegral (maxBound::Int) then 
    ioError (userError "slurpFile: file too big")
   else do
    let sz_i = fromIntegral sz
    chunk <- mallocBytes sz_i
    r <- hGetBuf handle chunk sz_i
    hClose handle
    return (chunk, r)

-- ---------------------------------------------------------------------------
-- hGetBufBA

hGetBufBA :: Handle -> MutableByteArray RealWorld a -> Int -> IO Int
hGetBufBA handle (MutableByteArray _ _ ptr) count
  | count <= 0 = illegalBufferSize handle "hGetBuf" count
  | otherwise = 
      wantReadableHandle "hGetBuf" handle $ 
	\ handle_@Handle__{ haFD=fd, haBuffer=ref } -> do
	buf@Buffer{ bufBuf=raw, bufWPtr=w, bufRPtr=r } <- readIORef ref
	if bufferEmpty buf
	   then readChunkBA fd ptr 0 count
	   else do 
		let avail = w - r
		copied <- if (count >= avail)
		       	    then do 
				memcpy_ba_baoff ptr raw r (fromIntegral avail)
				writeIORef ref buf{ bufWPtr=0, bufRPtr=0 }
				return avail
		     	    else do 
				memcpy_ba_baoff ptr raw r (fromIntegral count)
				writeIORef ref buf{ bufRPtr = r + count }
				return count

		let remaining = count - copied
		if remaining > 0 
		   then do rest <- readChunkBA fd ptr copied remaining
			   return (rest + count)
		   else return count
		
readChunkBA :: FD -> RawBuffer -> Int -> Int -> IO Int
readChunkBA fd ptr init_off bytes = loop init_off bytes 
 where
  loop :: Int -> Int -> IO Int
  loop off bytes | bytes <= 0 = return (off - init_off)
  loop off bytes = do
    r <- fromIntegral `liftM`
	   throwErrnoIfMinus1RetryMayBlock "readChunk"
	    (readBA (fromIntegral fd) ptr 
		(fromIntegral off) (fromIntegral bytes))
	    (threadWaitRead fd)
    if r == 0
	then return (off - init_off)
	else loop (off + r) (bytes - r)

foreign import "read_IOExts_wrap" unsafe
   readBA :: FD -> RawBuffer -> Int -> CInt -> IO CInt
#def inline \
int read_IOExts_wrap(int fd, void *ptr, HsInt off, int size) \
{ return read(fd, ptr + off, size); }

-- -----------------------------------------------------------------------------
-- hPutBufBA

hPutBufBA
	:: Handle			-- handle to write to
	-> MutableByteArray RealWorld a -- buffer
	-> Int				-- number of bytes of data in buffer
	-> IO ()

hPutBufBA handle (MutableByteArray _ _ raw) count
  | count <= 0 = illegalBufferSize handle "hPutBufBA" count
  | otherwise = do
    wantWritableHandle "hPutBufBA" handle $ 
      \ handle_@Handle__{ haFD=fd, haBuffer=ref } -> do

        old_buf@Buffer{ bufBuf=old_raw, bufRPtr=r, bufWPtr=w, bufSize=size }
	  <- readIORef ref

        -- enough room in handle buffer?
        if (size - w > count)
		-- There's enough room in the buffer:
		-- just copy the data in and update bufWPtr.
	    then do memcpy_baoff_ba old_raw w raw (fromIntegral count)
		    writeIORef ref old_buf{ bufWPtr = w + count }
		    return ()

		-- else, we have to flush
	    else do flushed_buf <- flushWriteBuffer fd old_buf
		    writeIORef ref flushed_buf
		    let this_buf = 
			    Buffer{ bufBuf=raw, bufState=WriteBuffer, 
				    bufRPtr=0, bufWPtr=count, bufSize=count }
		    flushWriteBuffer fd this_buf
		    return ()

-- ---------------------------------------------------------------------------
-- memcpy wrappers

foreign import "memcpy_IOExts_wrap_src_off" unsafe 
   memcpy_ba_baoff :: RawBuffer -> RawBuffer -> Int -> CSize -> IO (Ptr ())
foreign import "memcpy_IOExts_wrap_src_off" unsafe 
   memcpy_ptr_baoff :: Ptr a -> RawBuffer -> Int -> CSize -> IO (Ptr ())
foreign import "memcpy_IOExts_wrap_dst_off" unsafe 
   memcpy_baoff_ba :: RawBuffer -> Int -> RawBuffer -> CSize -> IO (Ptr ())
foreign import "memcpy_IOExts_wrap_dst_off" unsafe 
   memcpy_baoff_ptr :: RawBuffer -> Int -> Ptr a -> CSize -> IO (Ptr ())

#def inline \
void *memcpy_IOExts_wrap_dst_off(char *dst, int dst_off, char *src, size_t sz) \
{ return memcpy(dst+dst_off, src, sz); }

#def inline \
void *memcpy_IOExts_wrap_src_off(char *dst, char *src, int src_off, size_t sz) \
{ return memcpy(dst, src+src_off, sz); }

-- ---------------------------------------------------------------------------
-- deprecated I/O operations

{-# DEPRECATED hGetBufFull "renamed to hGetBuf" #-}
hGetBufFull   = hGetBuf

{-# DEPRECATED hPutBufFull "renamed to hPutBuf" #-}
hPutBufFull   = hPutBuf

{-# DEPRECATED hGetBufBAFull "renamed to hGetBufBA" #-}
hGetBufBAFull = hGetBufBA

{-# DEPRECATED hPutBufBAFull "renamed to hPutBufBA" #-}
hPutBufBAFull = hPutBufBA

-- ---------------------------------------------------------------------------
-- IOError predicates

isHardwareFault :: IOError -> Bool
isHardwareFault (IOException (IOError _ HardwareFault _ _ _)) = True
isHardwareFault _                                             = False

isInappropriateType :: IOError -> Bool
isInappropriateType (IOException (IOError _ InappropriateType _ _ _)) = True
isInappropriateType _                                                 = False

isInterrupted :: IOError -> Bool
isInterrupted (IOException (IOError _ Interrupted _ _ _)) = True
isInterrupted _                                           = False

isInvalidArgument :: IOError -> Bool
isInvalidArgument (IOException (IOError _ InvalidArgument _ _ _)) = True
isInvalidArgument _                                               = False

isOtherError :: IOError -> Bool
isOtherError (IOException (IOError _ OtherError _ _ _)) = True
isOtherError _                                          = False

isProtocolError :: IOError -> Bool
isProtocolError (IOException (IOError _ ProtocolError _ _ _)) = True
isProtocolError _                                             = False

isResourceVanished :: IOError -> Bool
isResourceVanished (IOException (IOError _ ResourceVanished _ _ _)) = True
isResourceVanished _                                                = False

isSystemError :: IOError -> Bool
isSystemError (IOException (IOError _ SystemError _ _ _)) = True
isSystemError _                                           = False

isTimeExpired :: IOError -> Bool
isTimeExpired (IOException (IOError _ TimeExpired _ _ _)) = True
isTimeExpired _                                           = False

isUnsatisfiedConstraints :: IOError -> Bool
isUnsatisfiedConstraints (IOException (IOError _ UnsatisfiedConstraints _ _ _)) = True
isUnsatisfiedConstraints _                                                      = False

isUnsupportedOperation  :: IOError -> Bool
isUnsupportedOperation  (IOException (IOError _ UnsupportedOperation  _ _ _)) = True
isUnsupportedOperation  _                                                     = False

#if defined(cygwin32_TARGET_OS) || defined(mingw32_TARGET_OS)
isComError :: IOError -> Bool
isComError (IOException (IOError _ (ComError _) _ _ _)) = True
isComError _                                            = False
#endif

-----------------------------------------------------------------------------
-- Internal Utils

illegalBufferSize :: Handle -> String -> Int -> IO a
illegalBufferSize handle fn (sz :: Int) = 
	ioException (IOError (Just handle)
			    InvalidArgument  fn
			    ("illegal buffer size " ++ showsPrec 9 sz [])
			    Nothing)

