% -----------------------------------------------------------------------------
% $Id: SystemExts.lhs,v 1.3 2001/08/14 18:09:12 sof Exp $
%
% (c) The GHC Team, 2001
%

Systemy extensions.

\begin{code}
module SystemExts
	( rawSystem,     -- :: String -> IO ExitCode

	, withArgs       -- :: [String] -> IO a -> IO a
	, withProgName   -- :: String -> IO a -> IO a
	) where

import System
import CError     ( throwErrnoIfMinus1 )
import MarshalAlloc
import MarshalArray
import CString hiding ( vectorize )
import Storable

import PrelIOBase
import Exception
import Ptr
import Monad
\end{code}

rawSystem

The same as system, but bypasses the shell.

\begin{code}
rawSystem :: String -> IO ExitCode
rawSystem "" = ioException (IOError Nothing InvalidArgument "rawSystem" "null command" Nothing)
rawSystem cmd =
  withCString cmd $ \s -> do
    status <- throwErrnoIfMinus1 "rawSystem" (primRawSystem s)
    case status of
        0  -> return ExitSuccess
        n  -> return (ExitFailure n)

foreign import ccall "rawSystemCmd" unsafe primRawSystem :: CString -> IO Int
\end{code}

@withArgs args act@ - while executing action @act@, have @System.getArgs@
return @args@.

@withProgName name act@ - while executing action @act@, have @System.getProgName@
return @name@.

When either of these actions return, the values of @getArgs@ / @getProgName@
are restored.

\begin{code}
withArgs xs act = do
   p <- System.getProgName
   withArgv (p:xs) act

withProgName nm act = do
   xs <- System.getArgs
   withArgv (nm:xs) act

\end{code}

Worker routine which marshals and replaces an argv vector for
the duration of an action.

\begin{code}
withArgv :: [String] -> IO a -> IO a
withArgv new_args act = do
  pName <- System.getProgName
  existing_args <- System.getArgs
  px <- setArgs new_args
  Exception.catch (do
	v <- act
	setArgs (pName:existing_args)
	free px
	return v)
	(\ ex -> setArgs (pName:existing_args) >> free px >> ioError ex)

setArgs :: [String] -> IO (Ptr CString)
setArgs argv = do
  vs <- vectorize argv
  setArgsPrim (length argv) vs
  return vs

vectorize :: [String] -> IO (Ptr CString)
vectorize args = do
   vec <- mallocBytes ((length args + 1) * sizeOf nullPtr)
   zipWithM_ (fillIn vec) args [0..]
   pokeElemOff vec (length args) nullPtr
   return vec
 where
  fillIn vec str idx = do
     x <- newCString str
     pokeElemOff vec idx x

foreign import "setProgArgv" setArgsPrim :: Int -> Ptr CString -> IO ()
\end{code}
