module Ospicture ( Picture, Origin, OSPictContext, Pen(..), Osfont.Font
                 , packPicture, unpackPicture, peekPicture, unpeekPicture, peekOSPictContext, peekScreen
                 , defaultPen, dialogPen, setPenAttribute
                 , apppicttoolbox, accpicttoolbox
                 , module StdPictureDef
                 ) where


--	********************************************************************************
--	Clean to Haskell Standard Object I/O library, version 1.2
--	
--	Ospicture contains drawing functions and other operations on Pictures. 
--	********************************************************************************


import IOExts
import Commondef
import Osrgn
import Osfont
import Ostoolbox
import PictCCall_12
import StdIOBasic
import StdPictureDef


data	Picture
	= Picture
		{ pictContext   :: !OSPictContext	-- The context for drawing operations
		, pictOrigin    :: !Origin		-- The current origin of the picture
		, pictPen       :: !Pen			-- The current state of the pen
		, pictToScreen  :: !Bool		-- Flag: the output goes to screen (True) or printer (False)
		}
type	Origin
	= Point2
type	OSPictContext
	= HDC
data  Pen
	= Pen
		{ penSize       :: !Int			-- The width and height of the pen
  		, penForeColour :: !Colour		-- The drawing colour of the pen
		, penBackColour :: !Colour		-- The background colour of the pen
		, penPos        :: !Point2		-- The pen position in local coordinates
		, penFont       :: !Font		-- The font information to draw text and characters
		}


--	Conversion operations to and from Picture

packPicture :: Origin -> Pen -> Bool -> OSPictContext -> IO Picture
packPicture origin pen@(Pen {penSize=penSize,penForeColour=penForeColour,penBackColour=penBackColour,penPos=penPos,penFont=penFont}) isScreenOutput hdc
	= do {
		hdc1 <- winInitPicture
				penSize
				iModeCopy
				initforecolour
				initbackcolour
				initpen
				(osfontname fontimp,osfontstyles fontimp,osfontsize fontimp)
				(0,0)
				hdc;
		return (Picture
			{ pictContext = hdc1
			, pictOrigin  = origin
			, pictPen     = pen
			, pictToScreen= isScreenOutput
			}
		       )
	  }
	where
		fontimp        = osFontgetimp penFont
		initforecolour = toRGBtriple penForeColour
		initbackcolour = toRGBtriple penBackColour
		initpen        = toTuple (penPos-origin)

unpackPicture :: Picture -> IO (Origin,Pen,Bool,OSPictContext)
unpackPicture (Picture {pictOrigin=pictOrigin,pictPen=pictPen,pictToScreen=pictToScreen,pictContext=pictContext})
	= do {
		(_,_,_,_,_,_,hdc) <- winDonePicture pictContext;
		return (pictOrigin,pictPen,pictToScreen,hdc)
	  }

peekPicture :: Picture -> IO (Origin,Pen,Bool,OSPictContext)
peekPicture (Picture {pictOrigin=pictOrigin,pictPen=pictPen,pictToScreen=pictToScreen,pictContext=pictContext})
	= return (pictOrigin,pictPen,pictToScreen,pictContext)

unpeekPicture :: Origin -> Pen -> Bool -> OSPictContext -> IO Picture
unpeekPicture origin pen isScreenOutput hdc
	= return (Picture {pictOrigin=origin,pictPen=pen,pictToScreen=isScreenOutput,pictContext=hdc})

peekOSPictContext :: Picture -> (OSPictContext,Picture)
peekOSPictContext picture
	= (pictContext picture,picture)

peekScreen :: (Picture -> IO (x,Picture)) -> IO x
peekScreen f
	= do {
		hdc          <- winCreateScreenHDC;
		picture      <- packPicture zero defaultPen True hdc;
		(x,picture1) <- f picture;
		(_,_,_,hdc1) <- unpackPicture picture1;
		winDestroyScreenHDC hdc1;
		return x
	  }

defaultPen :: Pen
defaultPen
	= Pen
		{ penSize       = 1
		, penForeColour = Black
		, penBackColour = White
		, penPos        = Point2 {x=0,y=0}
		, penFont       = defaultFont
		}
	where
		defaultFont     = unsafePerformIO osDefaultfont

dialogPen :: Pen
dialogPen
	= Pen
		{ penSize       = 1
		, penForeColour = Black
		, penBackColour = White
		, penPos        = Point2 {x=0,y=0}
		, penFont       = dialogFont
		}
	where
		dialogFont      = unsafePerformIO osDialogfont

setPenAttribute :: PenAttribute -> Pen -> Pen
setPenAttribute (PenSize   size)   pen = pen {penSize      =max 1 size}
setPenAttribute (PenPos    pos)    pen = pen {penPos       =pos       }
setPenAttribute (PenColour colour) pen = pen {penForeColour=colour    }
setPenAttribute (PenBack   colour) pen = pen {penBackColour=colour    }
setPenAttribute (PenFont   font)   pen = pen {penFont      =font      }


{-	Picture interface functions.
-}
apppicttoolbox :: IO () -> Picture -> IO Picture
apppicttoolbox f picture = f >> return picture

accpicttoolbox :: IO x -> Picture -> IO (x,Picture)
accpicttoolbox f picture = f >>= (\x->return (x,picture))


{-
{-	Attribute functions.
-}
--	Access to Origin and Pen:
getpictorigin :: !*Picture -> (!Origin,!*Picture)
getpictorigin picture=:{pictOrigin}
	= (pictOrigin,picture)

setpictorigin :: !Origin !*Picture -> *Picture
setpictorigin origin picture
	= {picture & pictOrigin=origin}

getpictpen :: !*Picture -> (!Pen,!*Picture)
getpictpen picture=:{pictPen}
	# (sPen,uPen)	= sharePen pictPen
	= (sPen,{picture & pictPen=uPen})

setpictpen :: !Pen !*Picture -> *Picture
setpictpen {penSize,penForeColour,penBackColour,penPos,penFont} picture
	# picture	= setpictpensize    penSize       picture
	# picture	= setpictpencolour  penForeColour picture
	# picture	= setpictbackcolour penBackColour picture
	# picture	= setpictpenpos     penPos        picture
	# picture	= setpictpenfont    penFont       picture
	= picture


--	Change the pen position:
setpictpenpos :: !Point2 !*Picture -> *Picture
setpictpenpos newpos=:{x=x`,y=y`} picture=:{pictToolbox,pictOrigin,pictPen=pen=:{penPos={x,y}},pictContext}
	| x==x` && y==y`
		= picture
	| otherwise
		# (context,tb)	= winMovePenTo (toTuple (newpos-pictOrigin)) (pictContext,pictToolbox)
		  pen			= {pen & penPos={x=x`,y=y`}}
		= {picture & pictToolbox=tb,pictContext=context,pictPen=pen}

getpictpenpos :: !*Picture -> (!Point2,!*Picture)
getpictpenpos picture=:{pictPen={penPos={x,y}}}
	= ({x=x,y=y},picture)

movepictpenpos :: !Vector2 !*Picture -> *Picture
movepictpenpos v=:{vx,vy} picture=:{pictToolbox,pictPen=pen=:{penPos={x,y}},pictContext}
	# (context,tb)	= winMovePen (toTuple v) (pictContext,pictToolbox)
	  pen			= {pen & penPos={x=x+vx,y=y+vy}}
	= {picture & pictToolbox=tb,pictContext=context,pictPen=pen}

--	Change the pen size:
setpictpensize :: !Int !*Picture -> *Picture
setpictpensize w picture=:{pictToolbox,pictContext,pictPen}
	| w`==pictPen.penSize
		= picture
	| otherwise
		# (context,tb)	= winSetPenSize w` (pictContext,pictToolbox)
		  pen			= {pictPen & penSize=w`}
		= {picture & pictToolbox=tb,pictContext=context,pictPen=pen}
where
	w` = max 1 w

getpictpensize :: !*Picture -> (!Int,!*Picture)
getpictpensize picture=:{pictPen={penSize}}
	= (penSize,picture)


--	Change the PenColour:
setpictpencolour :: !Colour !*Picture -> *Picture
setpictpencolour colour picture=:{pictToolbox,pictPen,pictContext}
	| reqRGB==curRGB
		= picture
	| otherwise
		# (context,tb)	= winSetPenColor reqRGB (pictContext,pictToolbox)
		  pen			= {pictPen & penForeColour=colour}
		= {picture & pictPen=pen,pictToolbox=tb,pictContext=context}
where
	reqRGB				= toRGBtriple colour
	curRGB				= toRGBtriple pictPen.penForeColour

setpictbackcolour :: !Colour !*Picture -> *Picture
setpictbackcolour colour picture=:{pictToolbox,pictPen,pictContext}
	| reqRGB==curRGB
		= picture
	| otherwise
		# (context,tb)	= winSetBackColor (toRGBtriple colour) (pictContext,pictToolbox)
		  pen			= {pictPen & penBackColour=colour}
		= {picture & pictPen=pen,pictToolbox=tb,pictContext=context}
where
	reqRGB				= toRGBtriple colour
	curRGB				= toRGBtriple pictPen.penBackColour
-}

toRGBtriple :: Colour -> (Int,Int,Int)
toRGBtriple (RGB rgb) = (setBetween (r rgb) minRGB maxRGB,setBetween (g rgb) minRGB maxRGB,setBetween (b rgb) minRGB maxRGB)
toRGBtriple Black     = (minRGB,minRGB,minRGB)
toRGBtriple DarkGrey  = (round((fromIntegral maxRGB)/4.0), round((fromIntegral maxRGB)/4.0), round((fromIntegral maxRGB)/4.0))
toRGBtriple Grey      = (round((fromIntegral maxRGB)/2.0), round((fromIntegral maxRGB)/2.0), round((fromIntegral maxRGB)/2.0))
toRGBtriple LightGrey = (round((fromIntegral maxRGB)*0.75),round((fromIntegral maxRGB)*0.75),round((fromIntegral maxRGB)*0.75))
toRGBtriple White     = (maxRGB,maxRGB,maxRGB)
toRGBtriple Red       = (maxRGB,minRGB,minRGB)
toRGBtriple Green     = (minRGB,maxRGB,minRGB)
toRGBtriple Blue      = (minRGB,minRGB,maxRGB)
toRGBtriple Cyan      = (minRGB,maxRGB,maxRGB)
toRGBtriple Magenta   = (maxRGB,minRGB,maxRGB)
toRGBtriple Yellow    = (maxRGB,maxRGB,minRGB)

{-
getpictpencolour :: !*Picture -> (!Colour,!*Picture)
getpictpencolour picture=:{pictPen={penForeColour}}
	= (penForeColour,picture)

getpictbackcolour :: !*Picture -> (!Colour,!*Picture)
getpictbackcolour picture=:{pictPen={penBackColour}}
	= (penBackColour,picture)


--	Change the font attributes:
setpictpenfont :: !Font !*Picture -> *Picture
setpictpenfont font picture=:{pictToolbox,pictContext,pictPen=pen}
	| imp==osFontgetimp pen.penFont
		= picture
	| otherwise
		# (context,tb)	= winSetFont (osfontname,osfontstyles,osfontsize) (pictContext,pictToolbox)
		  pen			= {pen & penFont=font}
		= {picture & pictToolbox=tb,pictContext=context,pictPen=pen}
where
	imp										= osFontgetimp font
	{osfontname,osfontstyles,osfontsize}	= imp

getpictpenfont :: !*Picture -> (!Font,!*Picture)
getpictpenfont picture=:{pictPen={penFont}}
	= (penFont,picture)

setpictpendefaultfont :: !*Picture -> *Picture
setpictpendefaultfont picture=:{pictToolbox,pictContext,pictPen}
	# (font,tb)		= osDefaultfont pictToolbox
	  {osfontname,osfontstyles,osfontsize}
	  				= osFontgetimp font
	# (context,tb)	= winSetFont (osfontname,osfontstyles,osfontsize) (pictContext,tb)
	  pen			= {pictPen & penFont=font}
	= {picture & pictToolbox=tb,pictContext=context,pictPen=pen}


{-	Drawing mode setting functions.
-}
setpictxormode :: !*Picture -> *Picture
setpictxormode picture=:{pictToolbox,pictContext}
	# (context,tb)	= winSetMode iModeXor (pictContext,pictToolbox)
	= {picture & pictToolbox=tb,pictContext=context}

setpicthilitemode :: !*Picture -> *Picture
setpicthilitemode picture=:{pictToolbox,pictContext}
	# (context,tb)	= winSetMode iModeXor (pictContext,pictToolbox)
	= {picture & pictToolbox=tb,pictContext=context}

setpictnormalmode :: !*Picture -> *Picture
setpictnormalmode picture=:{pictToolbox,pictContext}
	# (context,tb)	= winSetMode iModeCopy (pictContext,pictToolbox)
	= {picture & pictToolbox=tb,pictContext=context}


{-	Point2 drawing operations.
	pictdrawpoint
		only draws a point at that position. The pen position is not changed.
-}
pictdrawpoint :: !Point2 !*Picture -> *Picture
pictdrawpoint pos=:{x,y} picture=:{pictPen={penSize},pictOrigin={x=ox,y=oy},pictToolbox,pictContext}
	| penSize==1
		# (context,tb)	= winDrawPoint (x`,y`) (pictContext,pictToolbox)
		= {picture & pictToolbox=tb,pictContext=context}
	| otherwise
		# (context,tb)	= winFillRectangle {rleft=x`,rtop=y`,rright=x`+penSize,rbottom=y`+penSize} (pictContext,pictToolbox)
		= {picture & pictToolbox=tb,pictContext=context}
where
	(x`,y`)	= (x-ox,y-oy)


{-	Line drawing operations.
	pictdrawlineto
		draws a line from the current pen position to the given pen position. 
		The new pen position is the endpoint of the line.	
	pictdrawline
		draws a line from the first point to the second point. The pen position
		is not changed.
-}
pictdrawlineto :: !Point2 !*Picture -> *Picture
pictdrawlineto pos=:{x,y} picture=:{pictOrigin,pictToolbox,pictContext,pictPen}
	# (context,tb)	= winLinePenTo (toTuple (pos-pictOrigin)) (pictContext,pictToolbox)
	  pen			= {pictPen & penPos={x=x,y=y}}
	= {picture & pictToolbox=tb,pictContext=context,pictPen=pen}

pictundrawlineto :: !Point2 !*Picture -> *Picture
pictundrawlineto pos=:{x,y} picture=:{pictOrigin,pictToolbox,pictContext,pictPen=pen=:{penForeColour,penBackColour}}
	# (context,tb)	= winSetPenColor (toRGBtriple penBackColour) (pictContext,pictToolbox)
	# (context,tb)	= winLinePenTo (toTuple (pos-pictOrigin)) (context,tb)
	# (context,tb)	= winSetPenColor (toRGBtriple penForeColour) (context,tb)
	= {picture & pictToolbox=tb,pictContext=context,pictPen={pen & penPos={x=x,y=y}}}

pictdrawline :: !Point2 !Point2 !*Picture -> *Picture
pictdrawline a b picture=:{pictOrigin,pictToolbox,pictContext}
	# (context,tb)	= winDrawLine (toTuple (a-pictOrigin)) (toTuple (b-pictOrigin)) (pictContext,pictToolbox)
	= {picture & pictToolbox=tb,pictContext=context}

pictundrawline :: !Point2 !Point2 !*Picture -> *Picture
pictundrawline a b picture=:{pictOrigin,pictToolbox,pictContext,pictPen={penForeColour,penBackColour}}
	# (context,tb)	= winSetPenColor (toRGBtriple penBackColour) (pictContext,pictToolbox)
	# (context,tb)	= winDrawLine (toTuple (a-pictOrigin)) (toTuple (b-pictOrigin)) (context,tb)
	# (context,tb)	= winSetPenColor (toRGBtriple penForeColour) (context,tb)
	= {picture & pictToolbox=tb,pictContext=context}


{-	Text drawing operations.
	pictdraw(char/string) draws a char/string at the current pen position. The new
		pen position is immediately after the drawn char/string.
-}
pictdrawchar :: !Char !*Picture -> *Picture
pictdrawchar char picture=:{pictContext,pictToolbox,pictPen,pictOrigin}
	# (context,tb)		= winDrawChar (toInt char) (pictContext,pictToolbox)
	# (x`,y`,context,tb)= winGetPenPos (context,tb)
	#! {x,y}			= pictOrigin
	#! pen				= {pictPen & penPos={x=x+x`,y=y+y`}}
	= {picture & pictContext=context,pictToolbox=tb,pictPen=pen}

pictundrawchar :: !Char !*Picture -> *Picture
pictundrawchar char picture=:{pictContext,pictToolbox,pictPen=pen=:{penForeColour,penBackColour},pictOrigin={x=ox,y=oy}}
	# (context,tb)		= winSetPenColor (toRGBtriple penBackColour) (pictContext,pictToolbox)
	# (context,tb)		= winDrawChar (toInt char) (context,tb)
	# (context,tb)		= winSetPenColor (toRGBtriple penForeColour) (context,tb)
	# (x,y,context,tb)	= winGetPenPos (context,tb)
	= {picture & pictContext=context,pictToolbox=tb,pictPen={pen & penPos={x=x+ox,y=y+oy}}}

pictdrawstring :: !String !*Picture -> *Picture
pictdrawstring string picture=:{pictContext,pictToolbox,pictPen,pictOrigin={x=ox,y=oy}}
	# (context,tb)		= winDrawString string (pictContext,pictToolbox)
	# (x,y,context,tb)	= winGetPenPos (context,tb)
	  pen				= {pictPen & penPos={x=x+ox,y=y+oy}}
	= {picture & pictContext=context,pictToolbox=tb,pictPen=pen}

pictundrawstring :: !String !*Picture -> *Picture
pictundrawstring string picture=:{pictContext,pictToolbox,pictPen=pen=:{penForeColour,penBackColour},pictOrigin={x=ox,y=oy}}
	# (context,tb)		= winSetPenColor (toRGBtriple penBackColour) (pictContext,pictToolbox)
	# (context,tb)		= winDrawString string (context,tb)
	# (context,tb)		= winSetPenColor (toRGBtriple penForeColour) (context,tb)
	# (x,y,context,tb)	= winGetPenPos (context,tb)
	= {picture & pictContext=context,pictToolbox=tb,pictPen={pen & penPos={x=x+ox,y=y+oy}}}


{-	Oval drawing operations.
	pict(draw/fill)oval center oval 
		draws/fills an oval at center with horizontal and vertical radius. The new
		pen position is not changed.
-}
pictdrawoval :: !Point2 !Oval !*Picture -> *Picture
pictdrawoval center oval picture=:{pictContext,pictToolbox,pictOrigin}
	# (context,tb)	= winDrawOval rect (pictContext,pictToolbox)
	= {picture & pictContext=context,pictToolbox=tb}
where
	rect	= ovalToRect (center-pictOrigin) oval

pictundrawoval :: !Point2 !Oval !*Picture -> *Picture
pictundrawoval center oval picture=:{pictContext,pictToolbox,pictOrigin,pictPen={penBackColour,penForeColour}}
	# (context,tb)	= winSetPenColor (toRGBtriple penBackColour) (pictContext,pictToolbox)
	# (context,tb)	= winDrawOval rect (context,tb)
	# (context,tb)	= winSetPenColor (toRGBtriple penForeColour) (context,tb)
	= {picture & pictContext=context,pictToolbox=tb}
where
	rect	= ovalToRect (center-pictOrigin) oval

pictfilloval :: !Point2 !Oval !*Picture -> *Picture
pictfilloval center oval picture=:{pictContext,pictToolbox,pictOrigin}
	# (context,tb)	= winFillOval rect (pictContext,pictToolbox)
	= {picture & pictContext=context,pictToolbox=tb}
where
	rect	= ovalToRect (center-pictOrigin) oval

pictunfilloval :: !Point2 !Oval !*Picture -> *Picture
pictunfilloval center oval picture=:{pictContext,pictToolbox,pictOrigin,pictPen}
	# (context,tb)	= winEraseOval rect (pictContext,pictToolbox)
	= {picture & pictContext=context,pictToolbox=tb}
where
	rect	= ovalToRect (center-pictOrigin) oval

ovalToRect :: !Point2 !Oval -> Rect
ovalToRect {x,y} {oval_rx,oval_ry}
	= {rleft=x-rx,rtop=y-ry,rright=x+rx,rbottom=y+ry}
where
	rx	= abs oval_rx
	ry	= abs oval_ry


{-	Curve drawing operations.
	pict(draw/fill)curve movePen point curve
		draws/fills a curve starting at point with a shape defined by curve. If movePen
		is True, then the new pen position is at the end of the curve, otherwise it does
		not change.
-}
pictdrawcurve :: !Bool !Point2 !Curve !*Picture -> *Picture
pictdrawcurve movePen start=:{x,y} curve picture=:{pictContext,pictToolbox,pictOrigin}
	# (context,tb)		= winDrawCurve wrect (toTuple wstart) (toTuple wend) (pictContext,pictToolbox)
	# picture			= {picture & pictContext=context,pictToolbox=tb}
	| not movePen		= picture
	| otherwise			= setpictpenpos end picture
where
	start`				= start-pictOrigin
	(wrect,wstart,wend)	= getcurve_rect_begin_end start` curve
	end					= wend+pictOrigin

pictundrawcurve :: !Bool !Point2 !Curve !*Picture -> *Picture
pictundrawcurve movePen start=:{x,y} curve picture=:{pictContext,pictToolbox,pictOrigin,pictPen={penForeColour,penBackColour}}
	# (context,tb)		= winSetPenColor (toRGBtriple penBackColour) (pictContext,pictToolbox)
	# (context,tb)		= winDrawCurve wrect (toTuple wstart) (toTuple wend) (context,tb)
	# (context,tb)		= winSetPenColor (toRGBtriple penForeColour) (context,tb)
	# picture			= {picture & pictContext=context,pictToolbox=tb}
	| not movePen		= picture
	| otherwise			= setpictpenpos end picture
where
	start`				= start-pictOrigin
	(wrect,wstart,wend)	= getcurve_rect_begin_end start` curve
	end					= wend+pictOrigin

pictfillcurve :: !Bool !Point2 !Curve !*Picture -> *Picture
pictfillcurve movePen start curve picture=:{pictContext,pictToolbox,pictOrigin}
	# (context,tb)		= winFillWedge wrect (toTuple wstart) (toTuple wend) (pictContext,pictToolbox)
	# picture			= {picture & pictContext=context,pictToolbox=tb}
	| not movePen		= picture
	| otherwise			= setpictpenpos end picture
where
	start`				= start-pictOrigin
	(wrect,wstart,wend)	= getcurve_rect_begin_end start` curve
	end					= wend+pictOrigin

pictunfillcurve :: !Bool !Point2 !Curve !*Picture -> *Picture
pictunfillcurve movePen start curve picture=:{pictContext,pictToolbox,pictOrigin,pictPen={penForeColour,penBackColour}}
	# (context,tb)		= winSetPenColor (toRGBtriple penBackColour) (pictContext,pictToolbox)
	# (context,tb)		= winFillWedge wrect (toTuple wstart) (toTuple wend) (context,tb)
	# (context,tb)		= winSetPenColor (toRGBtriple penForeColour) (context,tb)
	# picture			= {picture & pictContext=context,pictToolbox=tb}
	| not movePen		= picture
	| otherwise			= setpictpenpos end picture
where
	start`				= start-pictOrigin
	(wrect,wstart,wend)	= getcurve_rect_begin_end start` curve
	end					= wend+pictOrigin

getcurve_rect_begin_end :: !Point2 !Curve -> (!Rect,!Point2,!Point2)
getcurve_rect_begin_end start=:{x,y} {curve_oval={oval_rx,oval_ry},curve_from,curve_to,curve_clockwise}
	| curve_clockwise	= (rect,end,start)
	| otherwise			= (rect,start,end)
where
	rx`					= toReal (abs oval_rx)
	ry`					= toReal (abs oval_ry)
	cx					= x -(toInt ((cos curve_from)*rx`))
	cy					= y +(toInt ((sin curve_from)*ry`))
	ex					= cx+(toInt ((cos curve_to  )*rx`))
	ey					= cy-(toInt ((sin curve_to  )*ry`))
	end					= {x=ex,y=ey}
	rect				= {rleft=cx-oval_rx,rtop=cy-oval_ry,rright=cx+oval_rx,rbottom=cy+oval_ry}


{-	Rect drawing operations.
	pict(draw/fill)rect rect
		draws/fills a rect. The pen position is not changed.
-}
pictdrawrect :: !Rect !*Picture -> *Picture
pictdrawrect r picture=:{pictContext,pictToolbox,pictOrigin}
	# (context,tb)	= winDrawRectangle (subVector (toVector pictOrigin) r) (pictContext,pictToolbox)
	= {picture & pictContext=context,pictToolbox=tb}

pictundrawrect :: !Rect !*Picture -> *Picture
pictundrawrect r picture=:{pictContext,pictToolbox,pictOrigin,pictPen={penForeColour,penBackColour}}
	# (context,tb)	= winSetPenColor (toRGBtriple penBackColour) (pictContext,pictToolbox)
	# (context,tb)	= winDrawRectangle (subVector (toVector pictOrigin) r) (context,tb)
	# (context,tb)	= winSetPenColor (toRGBtriple penForeColour) (context,tb)
	= {picture & pictContext=context,pictToolbox=tb}

pictfillrect :: !Rect !*Picture -> *Picture
pictfillrect r picture=:{pictContext,pictToolbox,pictOrigin}
	# (context,tb)	= winFillRectangle (subVector (toVector pictOrigin) r) (pictContext,pictToolbox)
	= {picture & pictContext=context,pictToolbox=tb}

pictunfillrect :: !Rect !*Picture -> *Picture
pictunfillrect r picture=:{pictContext,pictToolbox,pictOrigin}
	# (context,tb)	= winEraseRectangle (subVector (toVector pictOrigin) r) (pictContext,pictToolbox)
	= {picture & pictContext=context,pictToolbox=tb}


{-	Scrolling operation (handle with care).
-}
pictscroll :: !Rect !Vector2 !*Picture -> (!Rect,!*Picture)
pictscroll r v picture=:{pictContext,pictToolbox,pictOrigin}
	# (updRect,(context,tb))	= winScrollRectangle (subVector (toVector pictOrigin) r) (toTuple v) (pictContext,pictToolbox)
	= (updRect,{picture & pictContext=context,pictToolbox=tb})

{-	Polygon drawing operations.
	pict(draw/fill)polygon point polygon
		draws/fills a polygon starting at point. The pen position is not changed.
-}
pictdrawpolygon :: !Point2 !Polygon !*Picture -> *Picture
pictdrawpolygon start {polygon_shape} picture=:{pictContext,pictToolbox,pictOrigin}
	# tb			= transferPolygon (start-pictOrigin) polygon_shape pictToolbox
	# (context,tb)	= winDrawPolygon (pictContext,tb)
	# tb			= winEndPolygon tb
	= {picture & pictContext=context,pictToolbox=tb}

pictundrawpolygon :: !Point2 !Polygon !*Picture -> *Picture
pictundrawpolygon start {polygon_shape} picture=:{pictContext,pictToolbox,pictOrigin,pictPen={penForeColour,penBackColour}}
	# tb			= transferPolygon (start-pictOrigin) polygon_shape pictToolbox
	# (context,tb)	= winSetPenColor (toRGBtriple penBackColour) (pictContext,tb)
	# (context,tb)	= winDrawPolygon (context,tb)
	# tb			= winEndPolygon tb
	# (context,tb)	= winSetPenColor (toRGBtriple penForeColour) (context,tb)
	= {picture & pictContext=context,pictToolbox=tb}

pictfillpolygon :: !Point2 !Polygon !*Picture -> *Picture
pictfillpolygon start {polygon_shape} picture=:{pictPen={penSize},pictContext,pictToolbox,pictOrigin}
	# tb			= transferPolygon (start-pictOrigin) polygon_shape pictToolbox
	# (context,tb)	= winSetPenSize 1 (pictContext,tb)
	# (context,tb)	= winFillPolygon (context,tb)
	# (context,tb)	= winDrawPolygon (context,tb)
	# (context,tb)	= winSetPenSize penSize (context,tb)
	# tb			= winEndPolygon tb
	= {picture & pictContext=context,pictToolbox=tb}

pictunfillpolygon :: !Point2 !Polygon !*Picture -> *Picture
pictunfillpolygon start {polygon_shape} picture=:{pictPen={penSize,penForeColour,penBackColour},pictContext,pictToolbox,pictOrigin}
	# tb			= transferPolygon (start-pictOrigin) polygon_shape pictToolbox
	# (context,tb)	= winSetPenColor (toRGBtriple penBackColour) (pictContext,tb)
	# (context,tb)	= winSetPenSize 1 (context,tb)
	# (context,tb)	= winFillPolygon  (context,tb)
	# (context,tb)	= winDrawPolygon  (context,tb)
	# (context,tb)	= winSetPenSize penSize (context,tb)
	# tb			= winEndPolygon tb
	# (context,tb)	= winSetPenColor (toRGBtriple penForeColour) (context,tb)
	= {picture & pictContext=context,pictToolbox=tb}

transferPolygon :: !Point2 ![Vector2] !*OSToolbox -> *OSToolbox
transferPolygon start vs tb
	# tb	= winStartPolygon (1 + length vs) tb
	# tb	= winAddPolygonPoint wstart tb
	# tb	= transferShape wstart vs tb
	= tb
where
	wstart	= toTuple start
	
	transferShape :: !(!Int,!Int) ![Vector2] !*OSToolbox -> *OSToolbox
	transferShape (x,y) [{vx,vy}:vs] tb
   		= transferShape newpos vs (winAddPolygonPoint newpos tb)
	where
		newpos = (x+vx,y+vy)
	transferShape _ _ tb
		= tb

{-	Clipping operations.
	pictgetcliprgn gets the current clipping region.
	pictsetcliprgn sets the given clipping region.
	pictandcliprgn takes the intersection of the current clipping region and the argument region.
-}
pictgetcliprgn :: !*Picture -> (!OSRgnHandle,!*Picture)
pictgetcliprgn picture=:{pictContext,pictToolbox}
	# (cliprgn,(context,tb)) = winGetClipRgnPicture (pictContext,pictToolbox)
	= (cliprgn,{picture & pictContext=context,pictToolbox=tb})

pictsetcliprgn :: !OSRgnHandle !*Picture -> *Picture
pictsetcliprgn cliprgn picture=:{pictContext,pictToolbox}
	# (context,tb)	= winSetClipRgnPicture cliprgn (pictContext,pictToolbox)
	= {picture & pictContext=context,pictToolbox=tb}

pictandcliprgn :: !OSRgnHandle !*Picture -> *Picture
pictandcliprgn cliprgn picture=:{pictContext,pictToolbox}
	# (context,tb)	= winClipRgnPicture cliprgn (pictContext,pictToolbox)
	= {picture & pictContext=context,pictToolbox=tb}

{-	Resolution access function (added by MW):
-}
getResolutionC :: !OSPictContext !*OSToolbox -> (!(!Int,!Int),!*OSToolbox)
getResolutionC _ _
	= code 	{
 				ccall getResolutionC "I:VII:I"
			}

-- MW: scaling of screen coordinates to printer coordinates.
getPictureScalingFactors :: !OSPictContext !*OSToolbox -> (!(!Int,!Int),!(!Int,!Int),!OSPictContext,!*OSToolbox)
getPictureScalingFactors _ _
	= code
	{
		ccall WinGetPictureScaleFactor "II-IIIIII"
	}
-}