module Oswindow ( osControlTitleSpecialChars
                , osMinWindowSize, osMinCompoundSize
                , osInitialiseWindows
                , osGetButtonControlSize, osGetButtonControlHeight
                , osGetTextControlSize,   osGetTextControlHeight
                , osGetEditControlSize,   osGetEditControlHeight
                , osGetButtonControlMinWidth, osGetTextControlMinWidth, osGetEditControlMinWidth
                , osCreateDialog
                , OKorCANCEL(..)
                , osCreateTextControl, osCreateEditControl, osCreateButtonControl
                , Ostypes.DelayActivationInfo(..)
                , osDestroyWindow
                , osDestroyTextControl, osDestroyEditControl, osDestroyButtonControl
                , osUpdateTextControl, osUpdateEditControl, osUpdateButtonControl
                , osClipTextControl, osClipEditControl, osClipButtonControl
                , osGrabWindowPictContext, osReleaseWindowPictContext
                , toOSscrollbarRange, fromOSscrollbarRange, osScrollbarIsVisible, osScrollbarsAreVisible
                , osInvalidateWindow, osInvalidateWindowRect, osValidateWindowRect, osValidateWindowRgn, 
                , osDisableWindow, osEnableWindow, 
                , osActivateWindow, osActivateControl
                , osStackWindow
                , osHideWindow, osShowWindow, 
                , osSetWindowCursor
                , osGetWindowPos, osGetWindowViewFrameSize, osGetWindowSize
                , osSetWindowPos, osSetWindowViewFrameSize, osSetWindowSize
                , osSetWindowTitle
                , osSetEditControlText, osGetEditControlText, osSetEditControlCursor, osSetEditControlSelect
                , osSetEditControlShow, osSetEditControlPos, osSetEditControlSize
                , osSetTextControlText, osSetTextControlSelect, osSetTextControlShow, osSetTextControlPos, osSetTextControlSize
                , osSetButtonControlText, osSetButtonControlSelect, osSetButtonControlShow, osSetButtonControlPos, osSetButtonControlSize
                , Ossystem.OSWindowMetrics(..)
                ) where


--	********************************************************************************
--	Clean to Haskell Standard Object I/O library, version 1.2
--	
--	Oswindow contains OS operations to manage windows and controls.
--	********************************************************************************


import ClCCall_12
import ClCrossCall_12
import CleanStdList
import Cutil_12 (btoi)
import Commondef
import Osdocumentinterface
import Osevent
import Osfont
import Ospicture
import Osrgn
import Ossystem
import Ostoolbox
import Ostypes
import RgnCCall_12
import WindowCCall_12
import WindowCrossCall_12
import Windowhandle


oswindowFatalError :: String -> String -> x
oswindowFatalError function error
	= dumpFatalError function "Oswindow" error


{-	System dependent constants:
-}
osControlTitleSpecialChars :: [Char]
osControlTitleSpecialChars = []		-- Special prefix characters that should be removed


{-	System dependent metrics:
-}

osMinWindowSize :: (Int,Int)
osMinWindowSize = winMinimumWinSize

osMinCompoundSize :: (Int,Int)
osMinCompoundSize = (0,0)		-- PA: (0,0)<--WinMinimumWinSize (Check if this safe)


{-	Initialisation:
-}
osInitialiseWindows :: IO ()
osInitialiseWindows = winInitialiseWindows


{-	Determine the size of controls.
-}
osGetButtonControlSize :: OSWindowMetrics -> String -> IO (Int,Int)
osGetButtonControlSize wMetrics text
	= do {
		widths <- osGetfontstringwidths False 0 [text] (osmFont wMetrics);
		return (2*osmHeight wMetrics+hd widths,osGetButtonControlHeight wMetrics)
	  }

osGetButtonControlHeight :: OSWindowMetrics -> Int
osGetButtonControlHeight wMetrics = 2*osmHeight wMetrics

osGetTextControlSize :: OSWindowMetrics -> String -> IO (Int,Int)
osGetTextControlSize wMetrics text
	= do {
		widths <- osGetfontstringwidths False 0 [text] (osmFont wMetrics);
		return (hd widths+round ((fromIntegral (osmHeight wMetrics))/4.0),osGetTextControlHeight wMetrics)
	  }

osGetTextControlHeight :: OSWindowMetrics -> Int
osGetTextControlHeight (OSWindowMetrics {osmHeight=osmHeight}) = osmHeight+round ((fromIntegral osmHeight)/2)

osGetEditControlSize :: OSWindowMetrics -> Int -> Int -> IO (Int,Int)
osGetEditControlSize wMetrics width nrlines
	= return (width,osGetEditControlHeight wMetrics nrlines)

osGetEditControlHeight :: OSWindowMetrics -> Int -> Int
osGetEditControlHeight (OSWindowMetrics {osmHeight=osmHeight}) nrlines = round ((fromIntegral osmHeight)/2.0)+osmHeight*nrlines


{-	Determine the minimum width of controls.
-}
osGetButtonControlMinWidth :: OSWindowMetrics -> Int
osGetButtonControlMinWidth (OSWindowMetrics {osmHeight=osmHeight}) = 2*osmHeight

osGetTextControlMinWidth :: OSWindowMetrics -> Int
osGetTextControlMinWidth (OSWindowMetrics {osmHeight=osmHeight}) = round ((fromIntegral osmHeight)/4.0)

osGetEditControlMinWidth :: OSWindowMetrics -> Int
osGetEditControlMinWidth _ = 0


{-	Window creation functions.
-}
osCreateDialog :: Bool -> Bool -> String -> (Int,Int) -> (Int,Int) -> OSWindowPtr
               -> (s -> (OSWindowPtr,s))
               -> (OSWindowPtr -> s -> IO s)
               -> (OSWindowPtr -> OSWindowPtr -> OSPictContext -> s -> IO s)
               -> OSDInfo -> s
               -> IO ([DelayActivationInfo],OSWindowPtr,s)
osCreateDialog isModal isClosable title pos size behindptr get_focus create_controls update_controls osdinfo control_info
	= do {
		textptr      <- winMakeCString title;
		let createcci = rq4Cci ccRqCREATEDIALOG textptr parentptr (if behindptr==osNoWindowPtr then 0 else behindptr) (btoi isModal)
		in
		do {
			(returncci,(control_info1,delay_info))
			     <- issueCleanRequest (osCreateDialogCallback get_focus create_controls update_controls)
			                          createcci
			                          (control_info,[]);
			winReleaseCString textptr;
			let msg  = ccMsg returncci
			    wptr = if      msg==ccRETURN1 then p1 returncci
			           else if msg==ccWASQUIT then osNoWindowPtr
			           else                        oswindowCreateError 1 "osCreateDialog"
			in  return (reverse delay_info,wptr,control_info1)
		}
	  }
	where
		parentptr = case getOSDInfoOSInfo osdinfo of
		                Nothing   -> 0
		                Just info -> osFrame info
		
		osCreateDialogCallback :: (s -> (OSWindowPtr,s))
		                       -> (OSWindowPtr -> s -> IO s)
		                       -> (OSWindowPtr -> OSWindowPtr -> OSPictContext -> s -> IO s)
		                       -> CrossCallInfo
		                       -> (s,[DelayActivationInfo])
		                       -> IO (CrossCallInfo,(s,[DelayActivationInfo]))
		osCreateDialogCallback get_focus create_controls update_controls cci s@(control_info,delay_info)
			| msg==ccWmPAINT
				= winFakePaint (p1 cci) >> return (return0Cci, s)
			| msg==ccWmACTIVATE
				= return (return0Cci, (control_info,(DelayActivatedWindow (p1 cci)):delay_info))
			| msg==ccWmDEACTIVATE
				= return (return0Cci, (control_info,(DelayDeactivatedWindow (p1 cci)):delay_info))
			| msg==ccWmINITDIALOG
				= do {
					control_info1 <- create_controls (p1 cci) control_info;
					let (defhandle,control_info2) = get_focus control_info1
					    (x,y)                     = pos
					    (w,h)                     = size
					    r5cci                     = return5Cci x y w h (if defhandle==osNoWindowPtr then 0 else defhandle)
					in  return (r5cci, (control_info2,delay_info))
				  }
			| msg==ccWmDRAWCONTROL
				= do {
					control_info1 <- update_controls (p1 cci) (p2 cci) (p3 cci) control_info;
					return (return0Cci, (control_info1,delay_info))
				  }
			| msg==ccWmKEYBOARD || msg==ccWmSETFOCUS || msg==ccWmKILLFOCUS
				= return (return0Cci, s)
			| otherwise
				= oswindowFatalError "osCreateDialogCallback" ("unknown message type ("++show msg++")")
			where
				msg = ccMsg cci


{-	Control creation functions.
-}
oswindowCreateError :: Int -> String -> x
oswindowCreateError arity function
	= oswindowFatalError function ("Expected ccRETURN"++show arity++" value")

osIgnoreCallback :: CrossCallInfo -> IO CrossCallInfo
osIgnoreCallback ccinfo
	| ccMsg ccinfo==ccWmPAINT
		= winFakePaint (p1 ccinfo) >> return return0Cci
	| otherwise
		= return return0Cci

osIgnoreCallback' :: CrossCallInfo -> [DelayActivationInfo] -> IO (CrossCallInfo,[DelayActivationInfo])
osIgnoreCallback' ccinfo delayinfo
	| msg==ccWmPAINT
		= winFakePaint (p1 ccinfo) >> return (return0Cci,delayinfo)
	| msg==ccWmACTIVATE
		= return (return0Cci,(DelayActivatedWindow (p1 ccinfo)):delayinfo)
	| msg==ccWmDEACTIVATE
		= return (return0Cci,(DelayDeactivatedWindow (p1 ccinfo)):delayinfo)
	| otherwise
		= return (return0Cci,delayinfo)
	where
		msg = ccMsg ccinfo


{-	OKorCANCEL type is used to tell Windows that a (Custom)ButtonControl is 
	the OK, CANCEL, or normal button.
-}
data	OKorCANCEL
	= OK | CANCEL | NORMAL
	deriving (Show)

ok_toInt OK     = isOKBUTTON
ok_toInt CANCEL = isCANCELBUTTON
ok_toInt NORMAL = isNORMALBUTTON

osCreateTextControl :: OSWindowPtr -> (Int,Int) -> String -> Bool -> (Int,Int) -> (Int,Int) -> IO OSWindowPtr
osCreateTextControl parentWindow parentPos text show (x',y') (w,h)
	= do {
		returncci <- issueCleanRequest2 osIgnoreCallback createcci;
		let msg     = ccMsg returncci
		    textPtr = if      msg==ccRETURN1  then p1 returncci
		              else if msg==ccWASQUIT then osNoWindowPtr
		              else                        oswindowCreateError 1 "osCreateTextControl"
		in
		do {
			winSetWindowTitle textPtr text;
			winShowControl    textPtr show;
			return textPtr
		}
	  }
	where
		(x,y)     = (x'-fst parentPos,y'-snd parentPos)
		createcci = rq5Cci ccRqCREATESTATICTXT parentWindow x y w h

osCreateEditControl :: OSWindowPtr -> (Int,Int) -> String -> Bool -> Bool -> Bool -> (Int,Int) -> (Int,Int) -> IO OSWindowPtr
osCreateEditControl parentWindow parentPos text visible able isKeySensitive (x',y') (w,h)
	= do {
		wMetrics           <- osDefaultWindowMetrics;
		let nrLines         = round ((fromIntegral (h-osmHeight wMetrics))/(fromIntegral (osmHeight wMetrics)))
		    isMultiLine     = nrLines>1
		    editflags       = (if isMultiLine then editISMULTILINE else 0) + (if isKeySensitive then editISKEYSENSITIVE else 0)
		    createcci       = rq6Cci ccRqCREATEEDITTXT parentWindow x y w h editflags
		in
		do {
			returncci  <- issueCleanRequest2 osIgnoreCallback createcci;
			let msg     = ccMsg returncci
			    editPtr = if      msg==ccRETURN1 then p1 returncci
			              else if msg==ccWASQUIT then osNoWindowPtr
			              else                        oswindowCreateError 1 "osCreateEditControl"
			in
			do {
				winSetWindowTitle editPtr text;
				winEnableControl  editPtr able;
				winShowControl    editPtr visible;
				return editPtr
			}
		}
	  }
	where
		(x,y) = (x'-fst parentPos,y'-snd parentPos)

osCreateButtonControl :: OSWindowPtr -> (Int,Int) -> String -> Bool -> Bool -> (Int,Int) -> (Int,Int) -> OKorCANCEL -> IO OSWindowPtr
osCreateButtonControl parentWindow parentPos title show able (x',y') (w,h) okOrCancel
	= do {
		returncci    <- issueCleanRequest2 osIgnoreCallback createcci;
		let msg       = ccMsg returncci
		    buttonPtr = if      msg==ccRETURN1 then p1 returncci
		                else if msg==ccWASQUIT then osNoWindowPtr
		                else                   oswindowCreateError 1 "osCreateButtonControl"
		in
		do {
			winSetWindowTitle buttonPtr title;
			winEnableControl  buttonPtr able;
			winShowControl    buttonPtr show;
			return buttonPtr
		}
	  }
	where
		(x,y)     = (x'-fst parentPos,y'-snd parentPos)
		createcci = rq6Cci ccRqCREATEBUTTON parentWindow x y w h (ok_toInt okOrCancel)


{-	Window destruction operations.
	osDestroyWindow checks the process document interface and applies the appropriate destruction operation.
-}
osDestroyWindow :: OSDInfo -> Bool -> Bool -> OSWindowPtr -> (SchedulerEvent -> s -> IO ([Int],s)) -> s -> IO ([DelayActivationInfo],s)
osDestroyWindow osdInfo isModal isWindow wPtr handleOSEvent state
	| di==MDI
		= let osinfo     = fromJust (getOSDInfoOSInfo  osdInfo)
		      destroycci = if   isWindow   then rq3Cci ccRqDESTROYMDIDOCWINDOW (osFrame osinfo) (osClient osinfo) wPtr
		                   else if isModal then rq1Cci ccRqDESTROYMODALDIALOG wPtr
		                   else                 rq1Cci ccRqDESTROYWINDOW wPtr
		  in do {
		  	(_,(delayInfo,state1)) <- issueCleanRequest (osDelayCallback handleOSEvent) destroycci ([],state);
		  	return (reverse delayInfo,state1)
		     }
	| di==SDI
		= let destroycci = if isModal then rq1Cci ccRqDESTROYMODALDIALOG wPtr
		                              else rq1Cci ccRqDESTROYWINDOW wPtr
		  in do {
		  	(_,(delayInfo,state1)) <- issueCleanRequest (osDelayCallback handleOSEvent) destroycci ([],state);
		  	return (reverse delayInfo,state1)
		     }
	-- It's a NDI process
	| isWindow		{- This condition should never occur (NDI processes have only dialogues). -}
		= oswindowFatalError "osDestroyWindow" "trying to destroy window of NDI process"
	| otherwise
		= let destroycci = if isModal then rq1Cci ccRqDESTROYMODALDIALOG wPtr
		                              else rq1Cci ccRqDESTROYWINDOW wPtr
		  in do {
		  	(_,(delayInfo,state1)) <- issueCleanRequest (osDelayCallback handleOSEvent) destroycci ([],state);
		  	return (reverse delayInfo,state1)
		     }
	where
		di     = getOSDInfoDocumentInterface osdInfo

osDelayCallback :: (SchedulerEvent -> s -> IO ([Int],s)) -> CrossCallInfo -> ([DelayActivationInfo],s) -> IO (CrossCallInfo,([DelayActivationInfo],s))
osDelayCallback handleOSEvent osEvent (delayinfo,s)
	| toBeHandled
		= do {	(replyToOS,s1) <- handleOSEvent (ScheduleOSEvent osEvent []) s;
			return (setReplyInOSEvent replyToOS,(delayinfo,s1))
		  }
	| msg==ccWmACTIVATE
		= return (return0Cci,((DelayActivatedWindow (p1 osEvent)):delayinfo,s))
	| msg==ccWmDEACTIVATE
		= return (return0Cci,((DelayDeactivatedWindow (p1 osEvent)):delayinfo,s))
	| toBeSkipped
		= return (return0Cci,(delayinfo,s))
	| otherwise
		= oswindowFatalError "osDelayCallback" ("unexpected delay message "++show msg)
	where
		msg         = ccMsg osEvent
		toBeHandled = isMember msg [ccWmPAINT,ccWmDRAWCONTROL,ccWmKEYBOARD,ccWmKILLFOCUS,ccWmMOUSE,ccWmSETFOCUS]
		toBeSkipped = isMember msg [ccWmCLOSE,ccWmIDLETIMER,ccWmSIZE]


{-	Control destruction operations.
-}
destroycontrol :: OSWindowPtr -> IO ()
destroycontrol wPtr
	= issueCleanRequest2 osDestroyControlCallback (rq1Cci ccRqDESTROYWINDOW wPtr) >> return ()
	where
		osDestroyControlCallback :: CrossCallInfo -> IO CrossCallInfo
		osDestroyControlCallback info@(CrossCallInfo {ccMsg=ccMsg})
			| ccMsg==ccWmPAINT
				= winFakePaint (p1 info) >> return return0Cci
			| expected
				= return return0Cci
			| otherwise
				= oswindowFatalError "osDestroyControlCallback" ("unexpected message "++show ccMsg)
			where
				expected = isMember ccMsg [ccWmACTIVATE,ccWmBUTTONCLICKED,ccWmCOMBOSELECT,ccWmCOMMAND,ccWmDEACTIVATE
				                          ,ccWmDRAWCONTROL,ccWmIDLETIMER,ccWmKEYBOARD,ccWmKILLFOCUS,ccWmSETFOCUS
				                          ]

osDestroyTextControl :: OSWindowPtr -> IO ()
osDestroyTextControl wPtr = destroycontrol wPtr

osDestroyEditControl :: OSWindowPtr -> IO ()
osDestroyEditControl wPtr = destroycontrol wPtr

osDestroyButtonControl :: OSWindowPtr -> IO ()
osDestroyButtonControl wPtr = destroycontrol wPtr


{-	Control update operations.
-}
osUpdateTextControl :: Rect -> OSWindowPtr -> OSWindowPtr -> IO ()
osUpdateTextControl area parentWindow theControl = updatecontrol theControl area

osUpdateEditControl :: Rect -> OSWindowPtr -> OSWindowPtr -> IO ()
osUpdateEditControl area parentWindow theControl = updatecontrol theControl area

osUpdateButtonControl :: Rect -> OSWindowPtr -> OSWindowPtr -> IO ()
osUpdateButtonControl area parentWindow theControl = updatecontrol theControl area

updatecontrol :: OSWindowPtr -> Rect -> IO ()
updatecontrol theControl rect = winUpdateWindowRect theControl (toTuple4 rect)


{-	Control clipping operations.
-}
oscliprectrgn :: (Int,Int) -> Rect -> (Int,Int) -> (Int,Int) -> IO OSRgnHandle
oscliprectrgn parent_pos@(parent_x,parent_y) rect (x,y) (w,h)
	= osnewrectrgn (intersectRects area item)
	where
		area = subVector (fromTuple parent_pos) rect
		x'   = x-parent_x
		y'   = y-parent_y
		item = Rect {rleft=x',rtop=y',rright=x'+w,rbottom=y'+h}

osClipTextControl :: OSWindowPtr -> (Int,Int) -> Rect -> (Int,Int) -> (Int,Int) -> IO OSRgnHandle
osClipTextControl _ parentPos area itemPos itemSize = oscliprectrgn parentPos area itemPos itemSize

osClipEditControl :: OSWindowPtr -> (Int,Int) -> Rect -> (Int,Int) -> (Int,Int) -> IO OSRgnHandle
osClipEditControl _ parentPos area itemPos itemSize = oscliprectrgn parentPos area itemPos itemSize

osClipButtonControl :: OSWindowPtr -> (Int,Int) -> Rect -> (Int,Int) -> (Int,Int) -> IO OSRgnHandle
osClipButtonControl _ parentPos area itemPos itemSize = oscliprectrgn parentPos area itemPos itemSize


{-	Window graphics context access operations.
-}
osGrabWindowPictContext :: OSWindowPtr -> IO OSPictContext
osGrabWindowPictContext wPtr
	= winGetDC wPtr

osReleaseWindowPictContext :: OSWindowPtr -> OSPictContext -> IO ()
osReleaseWindowPictContext wPtr hdc
	= winReleaseDC wPtr hdc


{-	Window access operations.
-}
toOSscrollbarRange :: (Int,Int,Int) -> Int -> (Int,Int,Int,Int)
toOSscrollbarRange (domainMin,viewMin,domainMax) viewSize
	= (osRangeMin,osThumb,osRangeMax,osThumbSize+1)
	where
		(osRangeMin,osRangeMax) = toOSRange (domainMin,domainMax)
		range                   =  domainMax- domainMin
		osRange                 = osRangeMax-osRangeMin
		osThumb                 = inRange osRangeMin osRange (viewMin-domainMin) range
		osThumbSize             = if viewSize>=range then osRange else round (((fromIntegral viewSize)/(fromIntegral range))*(fromIntegral osRange))

fromOSscrollbarRange :: (Int,Int) -> Int -> Int
fromOSscrollbarRange (domainMin,domainMax) osThumb
	= inRange domainMin range (osThumb-osRangeMin) osRange
	where
		(osRangeMin,osRangeMax) = toOSRange (domainMin,domainMax)
		range                   =  domainMax- domainMin
		osRange                 = osRangeMax-osRangeMin

osScrollbarIsVisible :: (Int,Int) -> Int -> Bool
osScrollbarIsVisible (domainMin,domainMax) viewSize
	= viewSize<domainMax-domainMin

osScrollbarsAreVisible :: OSWindowMetrics -> Rect -> (Int,Int) -> (Bool,Bool) -> (Bool,Bool)
osScrollbarsAreVisible (OSWindowMetrics {osmHSliderHeight=osmHSliderHeight,osmVSliderWidth=osmVSliderWidth})
                       (Rect {rleft=xMin,rtop=yMin,rright=xMax,rbottom=yMax})
                       (width,height)
                       (hasHScroll,hasVScroll)
	= visScrollbars (False,False) (hasHScroll && (osScrollbarIsVisible hRange width),hasVScroll && (osScrollbarIsVisible vRange height))
	where
		hRange = (xMin,xMax)
		vRange = (yMin,yMax)
		
		visScrollbars :: (Bool,Bool) -> (Bool,Bool) -> (Bool,Bool)
		visScrollbars (showH1,showV1) (showH2,showV2)
			| showH1==showH2 && showV1==showV2
				= (showH1,showV1)
			| otherwise
				= visScrollbars (showH2,showV2) (showH,showV)
			where
				showH = if showV2 then hasHScroll && osScrollbarIsVisible hRange (width -osmVSliderWidth ) else showH2
				showV = if showH2 then hasVScroll && osScrollbarIsVisible vRange (height-osmHSliderHeight) else showV2

toOSRange :: (Int,Int) -> (Int,Int)
toOSRange (min,max)
	= (osSliderMin,if range<=osSliderRange then osSliderMin+range else osSliderMax)
	where
		range = max-min

inRange :: Int -> Int -> Int -> Int -> Int
inRange destMin destRange sourceValue sourceRange
	= destMin + round (((fromIntegral sourceValue) / (fromIntegral sourceRange)) * (fromIntegral destRange))

osSliderMin   = 0		-- 0
osSliderMax   = 32767		-- maxSigned2ByteInt
osSliderRange = 32767		-- osSliderMax-osSliderMin


{-	Window access operations.
-}
osInvalidateWindow :: OSWindowPtr -> IO ()
osInvalidateWindow theWindow
	= winInvalidateWindow theWindow

osInvalidateWindowRect :: OSWindowPtr -> Rect -> IO ()
osInvalidateWindowRect theWindow rect
	= winInvalidateRect theWindow (toTuple4 rect)

osValidateWindowRect :: OSWindowPtr -> Rect -> IO ()
osValidateWindowRect theWindow rect
	= winValidateRect theWindow (toTuple4 rect)

osValidateWindowRgn :: OSWindowPtr -> OSRgnHandle -> IO ()
osValidateWindowRgn theWindow rgn
	= winValidateRgn theWindow rgn

osDisableWindow :: OSWindowPtr -> (Bool,Bool) -> Bool -> IO ()
osDisableWindow theWindow scrollInfo modalContext
	= winSetSelectStateWindow theWindow scrollInfo False modalContext

osEnableWindow :: OSWindowPtr -> (Bool,Bool) -> Bool -> IO ()
osEnableWindow theWindow scrollInfo modalContext
	= winSetSelectStateWindow theWindow scrollInfo True modalContext

osActivateWindow :: OSDInfo -> OSWindowPtr -> (OSEvent -> s -> IO s) -> s -> IO ([DelayActivationInfo],s)
osActivateWindow osdInfo thisWindow handleOSEvent state
	= do {
		(_,(delayinfo,state1)) <- issueCleanRequest (osCallback handleOSEvent)
		                                            (rq3Cci ccRqACTIVATEWINDOW (btoi isMDI) clientPtr thisWindow)
		                                            ([],state);
		return (reverse delayinfo,state1)
	  }
	where
		isMDI     = getOSDInfoDocumentInterface osdInfo==MDI
		clientPtr = case getOSDInfoOSInfo osdInfo of
		                Just osinfo -> osClient osinfo
		                nothing     -> oswindowFatalError "osActivateWindow" "illegal DocumentInterface context"
		
	{-	osCallback delays activate and deactivate events.
		All other events are passed to the callback function.
	-}	osCallback :: (OSEvent -> s -> IO s) -> CrossCallInfo -> ([DelayActivationInfo],s) -> IO (CrossCallInfo,([DelayActivationInfo],s))
		osCallback handleOSEvent osEvent@(CrossCallInfo {ccMsg=ccMsg,p1=p1,p2=p2}) (delayinfo,s)
			| isJust maybeEvent
				= return (return0Cci,((fromJust maybeEvent):delayinfo,s))
			| otherwise
				= do { s1 <- handleOSEvent osEvent s;
				       return (return0Cci,(delayinfo,s1))
				  }
			where
				maybeEvent   = lookup ccMsg [(ccWmACTIVATE , DelayActivatedWindow    p1)
				                            ,(ccWmDEACTIVATE,DelayDeactivatedWindow  p1)
				                            ,(ccWmKILLFOCUS, DelayDeactivatedControl p1 p2)
				                            ,(ccWmSETFOCUS,  DelayActivatedControl   p1 p2)
				                            ]


osActivateControl :: OSWindowPtr -> OSWindowPtr -> IO [DelayActivationInfo]
osActivateControl parentWindow controlPtr
	= do {
		(_,delayinfo) <- issueCleanRequest osIgnoreCallback' (rq1Cci ccRqACTIVATECONTROL controlPtr) [];
		return (reverse delayinfo)
	  }
	where
		osIgnoreCallback' :: CrossCallInfo -> [DelayActivationInfo] -> IO (CrossCallInfo,[DelayActivationInfo])
		osIgnoreCallback' cci@(CrossCallInfo {ccMsg=msg,p1=p1, p2=p2}) delayinfo
			| msg==ccWmPAINT
				= winFakePaint p1 >> return (return0Cci,delayinfo)
			| msg==ccWmKILLFOCUS
				= return (return0Cci,(DelayDeactivatedControl p1 p2):delayinfo)
			| msg==ccWmSETFOCUS
				= return (return0Cci,(DelayActivatedControl p1 p2):delayinfo)
			| otherwise
				= return (return0Cci,delayinfo)

osStackWindow :: OSWindowPtr -> OSWindowPtr -> (OSEvent -> s -> IO s) -> s -> IO ([DelayActivationInfo],s)
osStackWindow thisWindow behindWindow handleOSEvent state
	= do {
		(_,(delayinfo,state1)) <- issueCleanRequest (osCallback handleOSEvent)
		                                            (rq2Cci ccRqRESTACKWINDOW thisWindow behindWindow)
		                                            ([],state);
		return (reverse delayinfo,state1)
	  }
	where
	{-	osCallback delays activate and deactivate events.
		All other events are passed to the callback function. 
		PA: is now identical to osActivateWindow!!
	-}	osCallback :: (OSEvent -> s -> IO s) -> CrossCallInfo -> ([DelayActivationInfo],s) -> IO (CrossCallInfo,([DelayActivationInfo],s))
		osCallback handleOSEvent osEvent@(CrossCallInfo {ccMsg=msg,p1=p1}) (delayinfo,s)
			| msg==ccWmACTIVATE
				= return (return0Cci,((DelayActivatedWindow p1):delayinfo,s))
			| msg==ccWmDEACTIVATE
				= return (return0Cci,((DelayDeactivatedWindow p1):delayinfo,s))
			| otherwise
				= do {
					s1 <- handleOSEvent osEvent s;
					return (return0Cci,(delayinfo,s1))
				  }

osHideWindow :: OSWindowPtr -> Bool -> IO [DelayActivationInfo]
osHideWindow wPtr activate
	= do {
		(_,delayinfo) <- issueCleanRequest osIgnoreCallback' (rq3Cci ccRqSHOWWINDOW wPtr (btoi False) (btoi activate)) [];
		return (reverse delayinfo)
	  }

osShowWindow :: OSWindowPtr -> Bool -> IO [DelayActivationInfo]
osShowWindow wPtr activate
	= do {
		(_,delayinfo) <- issueCleanRequest osIgnoreCallback' (rq3Cci ccRqSHOWWINDOW wPtr (btoi True) (btoi activate)) [];
		return (reverse delayinfo)
	  }

osSetWindowCursor :: OSWindowPtr -> Int -> IO ()
osSetWindowCursor wPtr cursorCode
	= winSetWindowCursor wPtr cursorCode

osGetWindowPos :: OSWindowPtr -> IO (Int,Int)
osGetWindowPos wPtr
	= winGetWindowPos wPtr

osGetWindowViewFrameSize :: OSWindowPtr -> IO (Int,Int)
osGetWindowViewFrameSize wPtr
	= winGetClientSize wPtr

osGetWindowSize :: OSWindowPtr -> IO (Int,Int)
osGetWindowSize wPtr
	= winGetWindowSize wPtr

osSetWindowPos :: OSWindowPtr -> (Int,Int) -> Bool -> Bool -> IO ()
osSetWindowPos wPtr pos update inclScrollbars
	= winSetWindowPos wPtr pos update inclScrollbars

osSetWindowViewFrameSize :: OSWindowPtr -> (Int,Int) -> IO ()
osSetWindowViewFrameSize wPtr size
	= winSetClientSize wPtr size

osSetWindowSize :: OSWindowPtr -> (Int,Int) -> Bool -> IO ()
osSetWindowSize wPtr size update
	= winSetWindowSize wPtr size update

osSetWindowTitle :: OSWindowPtr -> String -> IO ()
osSetWindowTitle wPtr title
	= winSetWindowTitle wPtr title


{-	Control access operations.
-}

--	On edit controls:

osSetEditControlText :: OSWindowPtr -> OSWindowPtr -> Rect -> Rect -> Bool -> String -> IO ()
osSetEditControlText _ ePtr _ _ _ text
	= winSetWindowTitle ePtr text

osGetEditControlText :: OSWindowPtr -> OSWindowPtr -> IO String
osGetEditControlText _ ePtr
	= winGetWindowText ePtr

osSetEditControlCursor :: OSWindowPtr -> OSWindowPtr -> Rect -> Rect -> Int -> IO ()
osSetEditControlCursor _ ePtr _ _ pos
	= winSetEditSelection ePtr pos (pos+1)

osSetEditControlSelect :: OSWindowPtr -> OSWindowPtr -> Rect -> Bool -> IO ()
osSetEditControlSelect _ ePtr _ select
	= winEnableControl ePtr select

osSetEditControlShow :: OSWindowPtr -> OSWindowPtr -> Rect -> Bool -> IO ()
osSetEditControlShow _ ePtr _ show
	= winShowControl ePtr show

osSetEditControlPos :: OSWindowPtr -> (Int,Int) -> OSWindowPtr -> (Int,Int) -> (Int,Int) -> Bool -> IO ()
osSetEditControlPos _ (parent_x,parent_y) editPtr (x,y) _ update
	= winSetWindowPos editPtr (x-parent_x,y-parent_y) update False

osSetEditControlSize :: OSWindowPtr -> (Int,Int) -> OSWindowPtr -> (Int,Int) -> (Int,Int) -> Bool -> IO ()
osSetEditControlSize _ _ editPtr _ size update
	= winSetWindowSize editPtr size update


--	On text controls:

osSetTextControlText :: OSWindowPtr -> OSWindowPtr -> Rect -> Rect -> Bool -> String -> IO ()
osSetTextControlText _ tPtr _ _ _ text
	= winSetWindowTitle tPtr text

osSetTextControlSelect :: OSWindowPtr -> OSWindowPtr -> Rect -> Bool -> IO ()
osSetTextControlSelect _ tPtr _ select
	= winEnableControl tPtr select

osSetTextControlShow :: OSWindowPtr -> OSWindowPtr -> Rect -> Bool -> IO ()
osSetTextControlShow _ tPtr _ show
	= winShowControl tPtr show

osSetTextControlPos :: OSWindowPtr -> (Int,Int) -> OSWindowPtr -> (Int,Int) -> (Int,Int) -> Bool -> IO ()
osSetTextControlPos _ (parent_x,parent_y) textPtr (x,y) _ update
	= winSetWindowPos textPtr (x-parent_x,y-parent_y) update False

osSetTextControlSize :: OSWindowPtr -> (Int,Int) -> OSWindowPtr -> (Int,Int) -> (Int,Int) -> Bool -> IO ()
osSetTextControlSize _ _ textPtr _ size update
	= winSetWindowSize textPtr size update


--	On button controls:

osSetButtonControlText :: OSWindowPtr -> OSWindowPtr -> Rect -> String -> IO ()
osSetButtonControlText _ bPtr _ text
	= winSetWindowTitle bPtr text

osSetButtonControlSelect :: OSWindowPtr -> OSWindowPtr -> Rect -> Bool -> IO ()
osSetButtonControlSelect _ bPtr _ select
	= winEnableControl bPtr select

osSetButtonControlShow :: OSWindowPtr -> OSWindowPtr -> Rect -> Bool -> IO ()
osSetButtonControlShow _ bPtr _ show
	= winShowControl bPtr show

osSetButtonControlPos :: OSWindowPtr -> (Int,Int) -> OSWindowPtr -> (Int,Int) -> (Int,Int) -> Bool -> IO ()
osSetButtonControlPos _ (parent_x,parent_y) buttonPtr (x,y) _ update
	= winSetWindowPos buttonPtr (x-parent_x,y-parent_y) update False

osSetButtonControlSize :: OSWindowPtr -> (Int,Int) -> OSWindowPtr -> (Int,Int) -> (Int,Int) -> Bool -> IO ()
osSetButtonControlSize _ _ buttonPtr _ size update
	= winSetWindowSize buttonPtr size update
