module Windowevent (windowEvent, module IOstate) where


--	********************************************************************************
--	Clean to Haskell Standard Object I/O library, version 1.2
--	
--	Windowevent defines the DeviceEventFunction for the window device.
--	This function is placed in a separate module because it is platform dependent.
--	********************************************************************************


import Bits		-- Defines .&. for bitand;
import Int		-- Defines Int32 instance for Bits;
import ClCCall_12
import ClCrossCall_12
import CleanStdMisc
import Commondef
import Controlcreate
import Cutil_12 (i32)
import Deviceevents
import IOstate
import Osevent
import StdControlAttribute
import Windowaccess
#include "preprocess.h"
         -- Compile with -cpp; set MVAR=1 for MVAR version


windoweventFatalError :: String -> String -> x
windoweventFatalError function error
	= dumpFatalError function "Windowevent" error


{-	windowEvent filters the scheduler events that can be handled by this window device.
	For the time being no timer controls are added, so these events are ignored.
	windowEvent assumes that it is not applied to an empty IOSt.
-}
windowEvent :: IOSt IF_MVAR(,ps) -> SchedulerEvent -> IO (Bool,Maybe DeviceEvent,SchedulerEvent)
windowEvent ioState schedulerEvent
	| not hasDevice		-- This condition should never occur: WindowDevice must have been 'installed'
		= windoweventFatalError "dEvent windowFunctions" "could not retrieve WindowSystemState from IOSt"
	| otherwise
		= windowEvent2 ioState1 schedulerEvent
	where
		(hasDevice,ioState1) = ioStHasDevice WindowDevice ioState
		
		windowEvent2 :: IOSt IF_MVAR(,ps) -> SchedulerEvent -> IO (Bool,Maybe DeviceEvent,SchedulerEvent)
		
		windowEvent2 ioState schedulerEvent@(ScheduleOSEvent osEvent _)
			| isNothing maybe_filterFun
				= return (False,Nothing,schedulerEvent)
			| otherwise
				= do {
					(myEvent,replyToOS,deviceEvent)
					     <- fromJust maybe_filterFun wMetrics osEvent windows;
					let schedulerEvent1 = if   isJust replyToOS
					                      then ScheduleOSEvent osEvent (fromJust replyToOS)
					                      else schedulerEvent
					in  return (myEvent,deviceEvent,schedulerEvent1)
				  }
			where
				((_,wDevice),ioState1) = ioStGetDevice WindowDevice ioState
				(wMetrics,   ioState2) = ioStGetOSWindowMetrics ioState1
				windows                = windowSystemStateGetWindowHandles wDevice
				maybe_filterFun
					= lookup (ccMsg osEvent)
					         [
					           (ccWmACTIVATE,        filterACTIVATE)
					         , (ccWmBUTTONCLICKED,   filterBUTTONCLICKED)
					         , (ccWmCLOSE,           filterCLOSE)
					--       , (ccWmCOMBOSELECT,     filterCOMBOSELECT)
					         , (ccWmDEACTIVATE,      filterDEACTIVATE)
					--       , (ccWmDRAWCONTROL,     filterDRAWCONTROL)
					         , (ccWmIDLEDIALOG,      filterIDLEDIALOG)
					         , (ccWmINITDIALOG,      filterINITDIALOG)
					         , (ccWmKEYBOARD,        filterKEYBOARD)
					--       , (ccWmKILLFOCUS,       filterKILLFOCUS)
					--       , (ccWmLOSTKEY,         filterLOSTKEY)
					--       , (ccWmLOSTMOUSE,       filterLOSTMOUSE)
					--       , (ccWmMOUSE,           filterMOUSE)
					--       , (ccWmPAINT,           filterPAINT)
					--       , (ccWmSCROLLBARACTION, filterSCROLLBARACTION)
					--       , (ccWmSETFOCUS,        filterSETFOCUS)
					--       , (ccWmSIZE,            filterSIZE)
					--       , (ccWmSPECIALBUTTON,   filterSPECIALBUTTON)
					         ]
		
		windowEvent2 ioState schedulerEvent@(ScheduleMsgEvent _)
			= return (False,Nothing,schedulerEvent)


{-	Because Haskell has no support for macros the function filterOSEvent that was defined by 
	pattern matching on the ccWmXXX messages has now been split into distinct functions named
	filterXXX. 
	No check is performed on the correct ccMsg field of the OSEvent argument.
	In this implementation not all events are handled. These have been placed in comments.
	They have however already been given the proper name and type.
-}

filterACTIVATE :: OSWindowMetrics -> OSEvent -> WindowHandles IF_MVAR(,ps) -> IO (Bool,Maybe [Int],Maybe DeviceEvent)
filterACTIVATE _ (CrossCallInfo {p1=wptr}) windows
	| not found
		= return (False,Nothing,Nothing)
	| active                             -- The window is already active, skip
		= return (True,Nothing,Nothing)
	| otherwise
		= let windows2        = setWindowHandlesWindow wsH2 windows1
		      (activeModal,_) = getWindowHandlesActiveModalDialog windows2
		  in  return (True,Nothing,if isJust activeModal then Just (WindowInitialise wids) else Just (WindowActivation wids))
	where
		(found, wsH,windows1) = getWindowHandlesWindow (toWID wptr) windows
		(active,wsH1)         = getWindowStateHandleActive wsH
		(wids,  wsH2)         = getWindowStateHandleWIDS   wsH1

filterBUTTONCLICKED :: OSWindowMetrics -> OSEvent -> WindowHandles IF_MVAR(,ps) -> IO (Bool,Maybe [Int],Maybe DeviceEvent)
filterBUTTONCLICKED _ (CrossCallInfo {p1=wptr,p2=cptr,p3=mods,p4=toolbarIndex}) windows
	| not found
		= return (False,Nothing,Nothing)
	| not able
		= return (True,Nothing,Nothing)
	| otherwise
		= let (wids,       wsH2) = getWindowStateHandleWIDS wsH1
		      (maybeItemNr,_)    = getControlsItemNr cptr   wsH2
		      controlSelectInfo  = case maybeItemNr of
		                                Nothing     -> Nothing
		                                Just itemNr -> Just (ControlSelection 
		                                                        (ControlSelectInfo
		                                                            { csWIDS      = wids
		                                                            , csItemNr    = itemNr
		                                                            , csItemPtr   = cptr
		                                                            , csMoreData  = 0
		                                                            , csModifiers = toModifiers mods
		                                                            }
		                                                    )   )
		  in  return (True,Nothing,controlSelectInfo)
	where
		(found,wsH,_)            = getWindowHandlesWindow (toWID wptr) windows
		(able,wsH1)              = getWindowStateHandleSelect wsH
		
		getControlsItemNr :: OSWindowPtr -> WindowStateHandle IF_MVAR(,ps) -> (Maybe Int,WindowStateHandle IF_MVAR(,ps))
		getControlsItemNr cptr wsH
			= accessWItemHandle "getControlsItemNr" condition (toSt wItemNr) wsH
			where
				condition itemH@(WItemHandle {wItemPtr=wItemPtr,wItemKind=wItemKind,wItemInfo=wItemInfo})
					| cptr==wItemPtr
						= True
				{-	| wItemKind==IsRadioControl
						= contains (\item->radioItemPtr item==cptr) (radioItems (getWItemRadioInfo wItemInfo))
					| wItemKind==IsCheckControl
						= contains (\item->checkItemPtr item==cptr) (checkItems (getWItemCheckInfo wItemInfo))
				-}	| otherwise
						= False

filterCLOSE :: OSWindowMetrics -> OSEvent -> WindowHandles IF_MVAR(,ps) -> IO (Bool,Maybe [Int],Maybe DeviceEvent)
filterCLOSE _ (CrossCallInfo {p1=wPtr}) windows
	| not found
		= return (False,Nothing,Nothing)
	| otherwise
		= let (wids,_) = getWindowStateHandleWIDS wsH
		  in  return (True,Nothing,Just (WindowRequestClose wids))
	where
		(found,wsH,_)  = getWindowHandlesWindow (toWID wPtr) windows
{-
filterCOMBOSELECT :: OSWindowMetrics -> OSEvent -> WindowHandles IF_MVAR(,ps) -> IO (Bool,Maybe [Int],Maybe DeviceEvent)
filterCOMBOSELECT _ (CrossCallInfo {ccMsg=ccWm,p1=wPtr,p2=cPtr,p3=index}) windows ioState
	# (found,wsH,windows)	= getWindowHandlesWindow (toWID wPtr) windows
	| not found
		= (False,Nothing,Nothing,windows,ioState)
	# (able,wsH)			= getWindowStateHandleSelect wsH
	| not able
		= (True,Nothing,Nothing,setWindowHandlesWindow wsH windows,ioState)
	| otherwise
		# (wids,  wsH)		= getWindowStateHandleWIDS wsH
		  (itemNr,wsH)		= getPopUpControlItemNr cPtr wsH
		  controlSelectInfo	= if (itemNr==0)	-- itemNrs are always > 0
								Nothing
								(Just (ControlSelection {csWIDS		= wids
														,csItemNr	= itemNr
														,csItemPtr	= cPtr
														,csMoreData	= index+1
														,csModifiers= NoModifiers
														})
								)
		= (True,Nothing,controlSelectInfo,setWindowHandlesWindow wsH windows,ioState)
where
	getPopUpControlItemNr :: !OSWindowPtr !(WindowStateHandle .pst) -> (!Int,!WindowStateHandle .pst)
	getPopUpControlItemNr cPtr wsH=:{wshHandle=Just wlsH=:{wlsHandle=wH=:{whItems}}}
		# (_,itemNr,itemHs)				= getPopUpControlsItemNr cPtr whItems
		= (itemNr,{wsH & wshHandle=Just {wlsH & wlsHandle={wH & whItems=itemHs}}})
	where
		getPopUpControlsItemNr :: !OSWindowPtr ![WElementHandle .ls .pst] -> (!Bool,!Int,![WElementHandle .ls .pst])
		getPopUpControlsItemNr cPtr [itemH:itemHs]
			# (found,itemNr,itemH)		= getPopUpControlItemNr cPtr itemH
			| found
				= (found,itemNr,[itemH:itemHs])
			| otherwise
				# (found,itemNr,itemHs)	= getPopUpControlsItemNr cPtr itemHs
				= (found,itemNr,[itemH:itemHs])
		where
			getPopUpControlItemNr :: !OSWindowPtr !(WElementHandle .ls .pst) -> (!Bool,!Int,!WElementHandle .ls .pst)
			getPopUpControlItemNr cPtr (WItemHandle itemH=:{wItemPtr,wItemNr,wItemKind,wItemSelect,wItemShow,wItems})
				| cPtr==wItemPtr
					= (True,if (wItemKind==IsPopUpControl && wItemSelect && wItemShow) wItemNr 0,WItemHandle itemH)
				| wItemShow
					# (found,itemNr,itemHs)	= getPopUpControlsItemNr cPtr wItems
					= (found,itemNr,WItemHandle {itemH & wItems=itemHs})
				| otherwise
					= (False,0,WItemHandle itemH)
			
			getPopUpControlItemNr cPtr (WListLSHandle itemHs)
				# (found,itemNr,itemHs)	= getPopUpControlsItemNr cPtr itemHs
				= (found,itemNr,WListLSHandle itemHs)
			
			getPopUpControlItemNr cPtr (WExtendLSHandle wExH=:{wExtendItems=itemHs})
				# (found,itemNr,itemHs)	= getPopUpControlsItemNr cPtr itemHs
				= (found,itemNr,WExtendLSHandle {wExH & wExtendItems=itemHs})
			
			getPopUpControlItemNr cPtr (WChangeLSHandle wChH=:{wChangeItems=itemHs})
				# (found,itemNr,itemHs)	= getPopUpControlsItemNr cPtr itemHs
				= (found,itemNr,WChangeLSHandle {wChH & wChangeItems=itemHs})
		
		getPopUpControlsItemNr _ []
			= (False,0,[])
	
	getPopUpControlItemNr _ _
		= windoweventFatalError "getPopUpControlItemNr" "window placeholder not expected"
-}
filterDEACTIVATE :: OSWindowMetrics -> OSEvent -> WindowHandles IF_MVAR(,ps) -> IO (Bool,Maybe [Int],Maybe DeviceEvent)
filterDEACTIVATE _ (CrossCallInfo {p1=wptr}) windows
	| not found
		= return (False,Nothing,Nothing)
	| not active                         -- The window is already inactive, skip
	        = return (True,Nothing,Nothing)
	| otherwise
		= let windows2        = setWindowHandlesWindow wsH2 windows1
		      (activeModal,_) = getWindowHandlesActiveModalDialog windows2
		  in  return (True,Nothing,if isJust activeModal then Nothing else Just (WindowDeactivation wids))
	where
		(found,wsH,windows1)  = getWindowHandlesWindow (toWID wptr) windows
		(active,wsH1)         = getWindowStateHandleActive wsH
		(wids,  wsH2)         = getWindowStateHandleWIDS wsH1
{-
filterDRAWCONTROL :: OSWindowMetrics -> OSEvent -> WindowHandles IF_MVAR(,ps) -> IO (Bool,Maybe [Int],Maybe DeviceEvent)
filterDRAWCONTROL _ (CrossCallInfo {p1=wPtr,p2=cPtr,p3=gc}) windows ioState
	# (found,wsH,windows)	= getWindowHandlesWindow (toWID wPtr) windows
	| not found
		= (False,Nothing,Nothing,windows,ioState)
	| otherwise
		# (wids,wsH)		= getWindowStateHandleWIDS wsH
		# (controls,wsH)	= getUpdateControls cPtr wsH
		  updateInfo		= if (isEmpty controls)
								Nothing
								(Just (WindowUpdate {updWIDS=wids,updWindowArea=zero,updControls=controls,updGContext=Just gc}))
		= (True,Nothing,updateInfo,setWindowHandlesWindow wsH windows,ioState)
where
	getUpdateControls :: !OSWindowPtr !(WindowStateHandle .pst) -> (![ControlUpdateInfo],!WindowStateHandle .pst)
	getUpdateControls cPtr wsH=:{wshHandle=Just wlsH=:{wlsHandle=wH=:{whItems,whSize}}}
		# (_,controls,itemHs)				= getUpdateControls cPtr (sizeToRect whSize) whItems
		= (controls,{wsH & wshHandle=Just {wlsH & wlsHandle={wH & whItems=itemHs}}})
	where
		getUpdateControls :: !OSWindowPtr !Rect ![WElementHandle .ls .pst] -> (!Bool,![ControlUpdateInfo],![WElementHandle .ls .pst])
		getUpdateControls cPtr clipRect [itemH:itemHs]
			# (found,controls,itemH)		= getUpdateControl cPtr clipRect itemH
			| found
				= (found,controls,[itemH:itemHs])
			| otherwise
				# (found,controls,itemHs)	= getUpdateControls cPtr clipRect itemHs
				= (found,controls,[itemH:itemHs])
		where
			getUpdateControl :: !OSWindowPtr !Rect !(WElementHandle .ls .pst) -> (!Bool,![ControlUpdateInfo],!WElementHandle .ls .pst)
			getUpdateControl cPtr clipRect (WItemHandle itemH=:{wItemPtr,wItemNr,wItemShow,wItemPos,wItemSize,wItems})
				| cPtr==wItemPtr
					= (True,[{cuItemNr=wItemNr,cuItemPtr=wItemPtr,cuArea=clipRect1}],WItemHandle itemH)
				| wItemShow
					# (found,controls,itemHs)	= getUpdateControls cPtr clipRect1 wItems
					= (found,controls,WItemHandle {itemH & wItems=itemHs})
				| otherwise
					= (False,[],WItemHandle itemH)
			where
				clipRect1						= intersectRects clipRect (posSizeToRect wItemPos wItemSize)
			
			getUpdateControl cPtr clipRect (WListLSHandle itemHs)
				# (found,controls,itemHs)		= getUpdateControls cPtr clipRect itemHs
				= (found,controls,WListLSHandle itemHs)
			
			getUpdateControl cPtr clipRect (WExtendLSHandle wExH=:{wExtendItems=itemHs})
				# (found,controls,itemHs)		= getUpdateControls cPtr clipRect itemHs
				= (found,controls,WExtendLSHandle {wExH & wExtendItems=itemHs})
			
			getUpdateControl cPtr clipRect (WChangeLSHandle wChH=:{wChangeItems=itemHs})
				# (found,controls,itemHs)		= getUpdateControls cPtr clipRect itemHs
				= (found,controls,WChangeLSHandle {wChH & wChangeItems=itemHs})
		
		getUpdateControls _ _ []
			= (False,[],[])
	
	getUpdateControls _ _
		= windoweventFatalError "getUpdateControls" "placeholder not expected"
-}
{-	ccWmIDLEDIALOG is sent after a modal dialogue and its controls have been created.
		At that moment the initialisation action can be evaluated. This is done by the 
		WindowInitialise device event. 
-}
filterIDLEDIALOG :: OSWindowMetrics -> OSEvent -> WindowHandles IF_MVAR(,ps) -> IO (Bool,Maybe [Int],Maybe DeviceEvent)
{-	No modal dialogues nor WindowInitialise attribute incorporated yet. If so, then this code: -}
filterIDLEDIALOG _ _ windows
	= return (False,Nothing,Nothing)
{-	...can be replaced by this code:
filterIDLEDIALOG _ (CrossCallInfo {p1=wptr}) windows
	| isNothing maybeWIDS
		= return (False,Nothing,Nothing)
	| wptr/=wPtr wids
		= return (False,Nothing,Nothing)
	| otherwise
		= return (True,Nothing,Just (WindowInitialise (fromJust maybeWIDS)))
	where
		(maybeWIDS,_) = getWindowHandlesActiveModalDialog windows
		wids          = fromJust maybeWIDS
-}

{-	ccWmINITDIALOG is generated for modal dialogs. It contains the ptr to the modal dialog.
	It is assumed that the current active window is the modal dialogue, with a zero ptr.
	filterINITDIALOG updates the ptr field and returns the WindowCreateControls DeviceEvent.
	This causes the Windowdevice dDoIO function to create all the controls of the
	dialog. 
	Note that CcWmINITDIALOG now replies to the OS only the desired position, and size.
	The focus control set to 0 as it will be set by windowCreateControls.
-}
filterINITDIALOG :: OSWindowMetrics -> OSEvent -> WindowHandles IF_MVAR(,ps) -> IO (Bool,Maybe [Int],Maybe DeviceEvent)
filterINITDIALOG wMetrics (CrossCallInfo {p1=wptr}) windows
	| isNothing maybeWIDS
		= return (False,Nothing,Nothing)
	| wPtr wids/=0
		= return (False,Nothing,Nothing)
	| otherwise
		= return (True,Just (getDialogMetrics wsH),Just (WindowCreateControls (wids {wPtr=wptr})))
	where
		(maybeWIDS,windows1) = getWindowHandlesActiveWindow windows
		wids                 = fromJust maybeWIDS
		(_,wsH,_)            = getWindowHandlesWindow (toWID (0::OSWindowPtr)) windows1
		
		getDialogMetrics :: WindowStateHandle IF_MVAR(,ps) -> [Int]
#if MVAR
		getDialogMetrics (WindowStateHandle {wshHandle=Just wH})
#else
		getDialogMetrics (WindowStateHandle _ (Just (WindowLSHandle {wlsHandle=wH})))
#endif
			= [-1,-1,w size,h size]
			where
				size  = whSize wH
		getDialogMetrics _
			= windoweventFatalError "getDialogMetrics" "placeholder not expected"

{-	ccWmKEYBOARD events are generated for windows/dialogues and keyboard sensitive
	controls.
	In this implementation windows/dialogues are ignored (p1==p2 in the OSEvent argument).
	The system also does not keep track of what GUI element has the keyboard input focus.
-} 
filterKEYBOARD :: OSWindowMetrics -> OSEvent -> WindowHandles IF_MVAR(,ps) -> IO (Bool,Maybe [Int],Maybe DeviceEvent)
filterKEYBOARD _ (CrossCallInfo {p1=wptr,p2=cptr,p3=keycode,p4=state,p5=mods}) windows
	= let (found,wsH,_)              = getWindowHandlesWindow (toWID wptr) windows
	  in
	  if   not found
	  then return (False,Nothing,Nothing)
	  else let (wids,wsH1)           = getWindowStateHandleWIDS wsH
	       in
	       if   wptr==cptr			-- The keyboard action takes place in the window
	       then windoweventFatalError "filterKEYBOARD" "window keyboard input currently not implemented."
	       else				-- The keyboard action takes place in a control 
		    let (maybeItemKey,_) = accessWItemHandle "okControlItemsNrKeyboardState"
			                                     (condition cptr)
			                                     (okControlItemNrKeyboardState keycode state mods)
			                                     wsH1
			deviceEvent      = case maybeItemKey of
			                        Just (ok,itemNr,key) -> if   ok
			                                                then Just (ControlKeyboardAction
			                                                              (ControlKeyboardActionInfo
			                                                                  { ckWIDS          = wids
			                                                                  , ckItemNr        = itemNr
			                                                                  , ckItemPtr       = cptr
			                                                                  , ckKeyboardState = key
			                                                                  }
			                                                          )   )
			                                                else Nothing
			                        _                    -> Nothing
		    in  return (True,Nothing,deviceEvent)
	where
		condition :: OSWindowPtr -> Cond (WElementHandle IF_MVAR(,ls ps))
		condition cptr itemH = cptr==wItemPtr itemH
		
		okControlItemNrKeyboardState :: Int -> Int -> Int -> St (WElementHandle IF_MVAR(,ls ps)) (Bool,Int,KeyboardState)
		okControlItemNrKeyboardState keycode state mods itemH@(WItemHandle {wItemNr=wItemNr,wItemAtts=wItemAtts})
			| okKeyboardAtt
				= ((okKeyboardAtt,wItemNr,keystate),itemH)
			| otherwise
				= ((False,0,undef),itemH)
			where
				noKeyboardAtt = ControlKeyboard (const False) Unable undef
				(filter,_,_)  = getControlKeyboardAtt (snd (cselect isControlKeyboard noKeyboardAtt wItemAtts))
				okKeyboardAtt = filter keystate
				keystate      = keyState keycode state mods
				pressState    = getKeyboardStateKeyState keystate
				isDownKey     = pressState==KeyDown False
		
		keyState :: Int -> Int -> Int -> KeyboardState
		keyState keycode state mods
			| isJust maybeSpecial = SpecialKey (fromJust maybeSpecial) ks modifiers
			| otherwise           = CharKey (toEnum keycode) ks	-- toChar --> toEnum
			where
				modifiers       = toModifiers mods
				ks              = fromJust $ lookup state
							[(keyDOWN,  KeyDown False)
							,(keyREPEAT,KeyDown True)
							,(keyUP,    KeyUp)
							]
				maybeSpecial    = lookup keycode
							[(winBackSpKey,backSpaceKey)
							,(winBeginKey, beginKey)
							,(winDelKey,   deleteKey)
							,(winDownKey,  downKey)
							,(winEndKey,   endKey)
							,(winEscapeKey,escapeKey)
							,(winHelpKey,  helpKey)
							,(winLeftKey,  leftKey)
							,(winPgDownKey,pgDownKey)
							,(winPgUpKey,  pgUpKey)
							,(winReturnKey,enterKey)
							,(winRightKey, rightKey)
							,(winUpKey,    upKey)
							,(winF1Key,    f1Key)
							,(winF2Key,    f2Key)
							,(winF3Key,    f3Key)
							,(winF4Key,    f4Key)
							,(winF5Key,    f5Key)
							,(winF6Key,    f6Key)
							,(winF7Key,    f7Key)
							,(winF8Key,    f8Key)
							,(winF9Key,    f9Key)
							,(winF10Key,   f10Key)
							,(winF11Key,   f11Key)
							,(winF12Key,   f12Key)
							]
{-
filterKILLFOCUS :: OSWindowMetrics -> OSEvent -> WindowHandles IF_MVAR(,ps) -> IO (Bool,Maybe [Int],Maybe DeviceEvent)
filterKILLFOCUS _ (CrossCallInfo {p1=wPtr,p2=cPtr}) windows ioState
	# (found,wsH,windows)	= getWindowHandlesWindow (toWID wPtr) windows
	| not found
		= (False,Nothing,Nothing,windows,ioState)
	# (wids,wsH)			= getWindowStateHandleWIDS wsH
	  (found,itemNr,wsH)	= getControlKeyFocusItemNr False cPtr wsH
	  windows				= setWindowHandlesWindow wsH windows
	| not found
		= (True,Nothing,Nothing,windows,ioState)
	| otherwise
		= (True,Nothing,Just (ControlLooseKeyFocus {ckfWIDS=wids,ckfItemNr=itemNr,ckfItemPtr=cPtr}),windows,ioState)
-}
{-
filterLOSTKEY :: OSWindowMetrics -> OSEvent -> WindowHandles IF_MVAR(,ps) -> IO (Bool,Maybe [Int],Maybe DeviceEvent)
filterLOSTKEY _ (CrossCallInfo {p1=wPtr,p2=cPtr}) windows ioState
	# (found,wsH,windows)			= getWindowHandlesWindow (toWID wPtr) windows
	| not found
		= (False,Nothing,Nothing,windows,ioState)
	# (able,wsH)					= getWindowStateHandleSelect wsH
	| not able
		= (True,Nothing,Nothing,setWindowHandlesWindow wsH windows,ioState)
	# (wids,wsH)					= getWindowStateHandleWIDS wsH
	| wPtr==cPtr	-- The window lost the keyboard input
		# (ok,wsH)					= okWindowKeyLost wsH
		  deviceEvent				= if ok (Just (WindowKeyboardAction {wkWIDS=wids,wkKeyboardState=KeyLost})) Nothing
		= (True,Nothing,deviceEvent,setWindowHandlesWindow wsH windows,ioState)
	with
		okWindowKeyLost :: !(WindowStateHandle .pst) -> (!Bool,!WindowStateHandle .pst)
		okWindowKeyLost wsH=:{wshHandle=Just {wlsHandle={whKind,whAtts}}}
			| whKind==IsDialog
				= (False,wsH)
			| otherwise
				= (okKeyAtt,wsH)
		where
			(filter,selectState,_)	= getWindowKeyboardAtt (snd (cselect isWindowKeyboard (WindowKeyboard (const False) Unable undef) whAtts))
			okKeyAtt				= filter KeyLost && selectState==Able
		okWindowKeyLost _
			= windoweventFatalError "okWindowKeyLost" "placeholder not expected"
	| otherwise		-- One of the window controls lost the keyboard input
		# (ok,itemNr,wsH)			= okControlItemNrsKeyLost cPtr wsH
		  info						= {	ckWIDS			= wids
									  ,	ckItemNr		= itemNr
									  ,	ckItemPtr		= cPtr
									  ,	ckKeyboardState	= KeyLost
									  }
		  deviceEvent				= if (ok && itemNr>0) (Just (ControlKeyboardAction info)) Nothing
		= (True,Nothing,deviceEvent,setWindowHandlesWindow wsH windows,ioState)
	with
		okControlItemNrsKeyLost :: !OSWindowPtr !(WindowStateHandle .pst) -> (!Bool,!Int,!WindowStateHandle .pst)
		okControlItemNrsKeyLost itemPtr wsH=:{wshHandle=Just wlsH=:{wlsHandle=wH=:{whItems}}}
			# (_,ok,itemNr,itemHs)	= okControlsItemNrKeyLost True itemPtr whItems
			= (ok,itemNr,{wsH & wshHandle=Just {wlsH & wlsHandle={wH & whItems=itemHs}}})
		where
			okControlsItemNrKeyLost :: !Bool !OSWindowPtr ![WElementHandle .ls .pst] -> (!Bool,!Bool,!Int,![WElementHandle .ls .pst])
			okControlsItemNrKeyLost contextAble itemPtr [itemH:itemHs]
				# (found,ok,itemNr,itemH)		= okControlItemNrKeyLost contextAble itemPtr itemH
				| found
					= (found,ok,itemNr,[itemH:itemHs])
				| otherwise
					# (found,ok,itemNr,itemHs)	= okControlsItemNrKeyLost contextAble itemPtr itemHs
					= (found,ok,itemNr,[itemH:itemHs])
			where
				okControlItemNrKeyLost :: !Bool !OSWindowPtr !(WElementHandle .ls .pst) -> (!Bool,!Bool,!Int,!WElementHandle .ls .pst)
				okControlItemNrKeyLost contextAble itemPtr (WItemHandle itemH=:{wItemPtr,wItemNr,wItemSelect,wItemShow,wItemAtts,wItems})
					| itemPtr<>wItemPtr
						| wItemShow
							# (found,okKey,itemNr,itemHs)	= okControlsItemNrKeyLost contextAble1 itemPtr wItems
							= (found,okKey,itemNr,WItemHandle {itemH & wItems=itemHs})
						-- otherwise
							= (False,False,0,WItemHandle itemH)
					| otherwise
						= (True,okKeyAtt,wItemNr,WItemHandle itemH)
				where
					contextAble1= contextAble && wItemSelect
					(filter,selectState,_)
								= getControlKeyboardAtt (snd (cselect isControlKeyboard (ControlKeyboard (const False) Unable undef) wItemAtts))
					okKeyAtt	= contextAble1 && enabled selectState && filter KeyLost
									
				okControlItemNrKeyLost contextAble itemPtr (WListLSHandle itemHs)
					# (found,okKey,itemNr,itemHs)	= okControlsItemNrKeyLost contextAble itemPtr itemHs
					= (found,okKey,itemNr,WListLSHandle itemHs)
				
				okControlItemNrKeyLost contextAble itemPtr (WExtendLSHandle wExH=:{wExtendItems=itemHs})
					# (found,okKey,itemNr,itemHs)	= okControlsItemNrKeyLost contextAble itemPtr itemHs
					= (found,okKey,itemNr,WExtendLSHandle {wExH & wExtendItems=itemHs})
				
				okControlItemNrKeyLost contextAble itemPtr (WChangeLSHandle wChH=:{wChangeItems=itemHs})
					# (found,okKey,itemNr,itemHs)	= okControlsItemNrKeyLost contextAble itemPtr itemHs
					= (found,okKey,itemNr,WChangeLSHandle {wChH & wChangeItems=itemHs})
			
			okControlsItemNrKeyLost _ _ []
				= (False,False,0,[])
		
		okControlItemNrsKeyLost _ _
			= windoweventFatalError "okControlItemNrsKeyLost" "placeholder not expected"
-}
{-
filterLOSTMOUSE :: OSWindowMetrics -> OSEvent -> WindowHandles IF_MVAR(,ps) -> IO (Bool,Maybe [Int],Maybe DeviceEvent)
filterLOSTMOUSE _ (CrossCallInfo {p1=wPtr,p2=cPtr}) windows ioState
	# (found,wsH,windows)			= getWindowHandlesWindow (toWID wPtr) windows
	| not found
		= (False,Nothing,Nothing,windows,ioState)
	# (able,wsH)					= getWindowStateHandleSelect wsH
	| not able
		= (True,Nothing,Nothing,setWindowHandlesWindow wsH windows,ioState)
	# (wids,wsH)					= getWindowStateHandleWIDS wsH
	| wPtr==cPtr	-- The window lost the mouse input
		# (ok,wsH)					= okWindowMouseLost wsH
		  deviceEvent				= if ok (Just (WindowMouseAction {wmWIDS=wids,wmMouseState=MouseLost})) Nothing
		= (True,Nothing,deviceEvent,setWindowHandlesWindow wsH windows,ioState)
	with
		okWindowMouseLost :: !(WindowStateHandle .pst) -> (!Bool,!WindowStateHandle .pst)
		okWindowMouseLost wsH=:{wshHandle=Just {wlsHandle={whKind,whAtts}}}
			| whKind==IsDialog
				= (False,wsH)
			| otherwise
				= (okMouseAtt,wsH)
		where
			(filter,selectState,_)	= getWindowMouseAtt (snd (cselect isWindowMouse (WindowMouse (const False) Unable undef) whAtts))
			okMouseAtt				= filter MouseLost && selectState==Able
		okWindowMouseLost _
			= windoweventFatalError "okWindowMouseLost" "placeholder not expected"
	| otherwise		-- One of the window controls lost the mouse input
		# (ok,itemNr,wsH)			= okControlItemNrsMouseLost cPtr wsH
		  info						= {	cmWIDS			= wids
									  ,	cmItemNr		= itemNr
									  ,	cmItemPtr		= cPtr
									  ,	cmMouseState	= MouseLost
									  }
		  deviceEvent				= if (ok && itemNr>0) (Just (ControlMouseAction info)) Nothing
		= (True,Nothing,deviceEvent,setWindowHandlesWindow wsH windows,ioState)
	with
		okControlItemNrsMouseLost :: !OSWindowPtr !(WindowStateHandle .pst) -> (!Bool,!Int,!WindowStateHandle .pst)
		okControlItemNrsMouseLost itemPtr wsH=:{wshHandle=Just wlsH=:{wlsHandle=wH=:{whItems}}}
			# (_,ok,itemNr,itemHs)	= okControlsItemNrMouseLost True itemPtr whItems
			= (ok,itemNr,{wsH & wshHandle=Just {wlsH & wlsHandle={wH & whItems=itemHs}}})
		where
			okControlsItemNrMouseLost :: !Bool !OSWindowPtr ![WElementHandle .ls .pst] -> (!Bool,!Bool,!Int,![WElementHandle .ls .pst])
			okControlsItemNrMouseLost contextAble itemPtr [itemH:itemHs]
				# (found,ok,itemNr,itemH)		= okControlItemNrMouseLost contextAble itemPtr itemH
				| found
					= (found,ok,itemNr,[itemH:itemHs])
				| otherwise
					# (found,ok,itemNr,itemHs)	= okControlsItemNrMouseLost contextAble itemPtr itemHs
					= (found,ok,itemNr,[itemH:itemHs])
			where
				okControlItemNrMouseLost :: !Bool !OSWindowPtr !(WElementHandle .ls .pst) -> (!Bool,!Bool,!Int,!WElementHandle .ls .pst)
				okControlItemNrMouseLost contextAble itemPtr (WItemHandle itemH=:{wItemPtr,wItemNr,wItemSelect,wItemShow,wItemAtts,wItems})
					| itemPtr<>wItemPtr
						| wItemShow
							# (found,ok,itemNr,itemHs)	= okControlsItemNrMouseLost contextAble1 itemPtr wItems
							= (found,ok,itemNr,WItemHandle {itemH & wItems=itemHs})
						-- otherwise
							= (False,False,0,WItemHandle itemH)
					| otherwise
						= (True,okMouseAtt,wItemNr,WItemHandle itemH)
				where
					contextAble1= contextAble && wItemSelect
					(filter,selectState,_)
								= getControlMouseAtt (snd (cselect isControlMouse (ControlMouse (const False) Unable undef) wItemAtts))
					okMouseAtt	= contextAble1 && enabled selectState && filter MouseLost
									
				okControlItemNrMouseLost contextAble itemPtr (WListLSHandle itemHs)
					# (found,ok,itemNr,itemHs)	= okControlsItemNrMouseLost contextAble itemPtr itemHs
					= (found,ok,itemNr,WListLSHandle itemHs)
				
				okControlItemNrMouseLost contextAble itemPtr (WExtendLSHandle wExH=:{wExtendItems=itemHs})
					# (found,ok,itemNr,itemHs)	= okControlsItemNrMouseLost contextAble itemPtr itemHs
					= (found,ok,itemNr,WExtendLSHandle {wExH & wExtendItems=itemHs})
				
				okControlItemNrMouseLost contextAble itemPtr (WChangeLSHandle wChH=:{wChangeItems=itemHs})
					# (found,ok,itemNr,itemHs)	= okControlsItemNrMouseLost contextAble itemPtr itemHs
					= (found,ok,itemNr,WChangeLSHandle {wChH & wChangeItems=itemHs})
			
			okControlsItemNrMouseLost _ _ []
				= (False,False,0,[])
		
		okControlItemNrsMouseLost _ _
			= windoweventFatalError "okControlItemNrsMouseLost" "placeholder not expected"
-}
{-
filterMOUSE :: OSWindowMetrics -> OSEvent -> WindowHandles IF_MVAR(,ps) -> IO (Bool,Maybe [Int],Maybe DeviceEvent)
filterMOUSE _ (CrossCallInfo {p1=wPtr,p2=cPtr,p3=action,p4=x,p5=y,p6=mods}) windows ioState
	# (found,wsH,windows)			= getWindowHandlesWindow (toWID wPtr) windows
	| not found
		= (False,Nothing,Nothing,windows,ioState)
	# (able,wsH)					= getWindowStateHandleSelect wsH
	| not able
		= (True,Nothing,Nothing,setWindowHandlesWindow wsH windows,ioState)
	# (wids,wsH)					= getWindowStateHandleWIDS wsH
	| wPtr==cPtr	-- The mouse action takes place in the window
		# (inputTrack,ioState)		= ioStGetInputTrack ioState
		  (ok,mouse,wsH,inputTrack)	= okWindowMouseState action {x=x,y=y} wsH inputTrack
		  deviceEvent				= if ok (Just (WindowMouseAction {wmWIDS=wids,wmMouseState=mouse})) Nothing
		# ioState					= ioStSetInputTrack inputTrack ioState
		= (True,Nothing,deviceEvent,setWindowHandlesWindow wsH windows,ioState)
	with
		okWindowMouseState :: !Int !Point2 !(WindowStateHandle .pst) !(Maybe InputTrack)
					   -> (!Bool,MouseState,!WindowStateHandle .pst, ! Maybe InputTrack)
		okWindowMouseState action eventPos wsH=:{wshHandle=Just {wlsHandle={whKind,whWindowInfo,whAtts}}} inputTrack
			| whKind==IsDialog
				= (False,undef,wsH,inputTrack)
			| trackingMouse wPtr 0 inputTrack					-- Window is already handling Mouse(Drag/Up)
				| isDownButton || buttonstate==ButtonStillUp	-- Ignore all mouse down and mouse move events
					= (False,undef,wsH,inputTrack)
				| buttonstate==ButtonUp							-- Clear mouse tracking
					= (okMouseAtt,mousestate,wsH,untrackMouse inputTrack)
				-- otherwise
					= (okMouseAtt,mousestate,wsH,inputTrack)
			| isDownButton										-- Mouse down event sets input track
				= (okMouseAtt,mousestate,wsH,trackMouse wPtr 0 inputTrack)
			| isMember buttonstate [ButtonStillDown,ButtonUp]	-- Ignore all mouse drag and up events when not tracking
				= (False,undef,wsH,inputTrack)
			| otherwise
				= (okMouseAtt,mousestate,wsH,inputTrack)
		where
			origin					= (getWindowInfoWindowData whWindowInfo).windowOrigin
			mousestate				= mouseState action (eventPos+origin)
			buttonstate				= getMouseStateButtonState mousestate
			isDownButton			= isMember buttonstate [ButtonDown,ButtonDoubleDown,ButtonTripleDown]
			(filter,selectState,_)	= getWindowMouseAtt (snd (cselect isWindowMouse (WindowMouse (const False) Unable undef) whAtts))
			okMouseAtt				= filter mousestate && selectState==Able
		okWindowMouseState _ _ _ _
			= windoweventFatalError "okWindowMouseState" "placeholder not expected"
	| otherwise				-- The mouse action takes place in a control
		# (inputTrack,ioState)		= ioStGetInputTrack ioState
		  (ok,itemNr,mouse,wsH,inputTrack)
		  							= okControlItemsNrMouseState wPtr cPtr action {x=x,y=y} wsH inputTrack
		# ioState					= ioStSetInputTrack inputTrack ioState
		  info						= {	cmWIDS			= wids
									  ,	cmItemNr		= itemNr
									  ,	cmItemPtr		= cPtr
									  ,	cmMouseState	= mouse
									  }
		  deviceEvent				= if ok (Just (ControlMouseAction info)) Nothing
		= (True,Nothing,deviceEvent,setWindowHandlesWindow wsH windows,ioState)
	with
		okControlItemsNrMouseState :: !OSWindowPtr !OSWindowPtr !Int !Point2 !(WindowStateHandle .pst) !(Maybe InputTrack)
												   -> (!Bool,!Int,MouseState, !WindowStateHandle .pst, ! Maybe InputTrack)
		okControlItemsNrMouseState wPtr itemPtr action eventPos wsH=:{wshHandle=Just wlsH=:{wlsHandle=wH=:{whItems}}} inputTrack
			# (_,ok,itemNr,itemPos,itemHs,inputTrack)
									= okControlsItemNrMouseState True wPtr itemPtr action eventPos whItems inputTrack
			= (ok,itemNr,itemPos,{wsH & wshHandle=Just {wlsH & wlsHandle={wH & whItems=itemHs}}},inputTrack)
		where
			okControlsItemNrMouseState :: !Bool !OSWindowPtr !OSWindowPtr !Int !Point2 ![WElementHandle .ls .pst] !(Maybe InputTrack)
													   -> (!Bool,!Bool,!Int,MouseState,![WElementHandle .ls .pst], !Maybe InputTrack)
			okControlsItemNrMouseState contextAble wPtr itemPtr action eventPos [itemH:itemHs] inputTrack
				# (found,ok,itemNr,itemPos,itemH,inputTrack)
										= okControlItemNrMouseState contextAble wPtr itemPtr action eventPos itemH   inputTrack
				| found
					= (found,ok,itemNr,itemPos,[itemH:itemHs],inputTrack)
				| otherwise
					# (found,ok,itemNr,itemPos,itemHs,inputTrack)	= okControlsItemNrMouseState contextAble wPtr itemPtr action eventPos itemHs inputTrack
					= (found,ok,itemNr,itemPos,[itemH:itemHs],inputTrack)
			where
				okControlItemNrMouseState :: !Bool !OSWindowPtr !OSWindowPtr !Int !Point2 !(WElementHandle .ls .pst) !(Maybe InputTrack)
														   -> (!Bool,!Bool,!Int,MouseState,!WElementHandle .ls .pst,  !Maybe InputTrack)
				okControlItemNrMouseState contextAble wPtr itemPtr action eventPos 
										  (WItemHandle itemH=:{wItemPtr,wItemSelect,wItemKind,wItemNr,wItemShow,wItemAtts,wItems,wItemInfo})
										  inputTrack
					| itemPtr<>wItemPtr
						| wItemShow
							# (found,ok,itemNr,mousestate,itemHs,inputTrack)	= okControlsItemNrMouseState contextAble1 wPtr itemPtr action eventPos wItems inputTrack
							= (found,ok,itemNr,mousestate,WItemHandle {itemH & wItems=itemHs},inputTrack)
						-- otherwise
							= (False,False,0,undef,WItemHandle itemH,inputTrack)
					| trackingMouse wPtr itemPtr inputTrack				-- Control is already handling Mouse(Drag/Up)
						| isDownButton || buttonstate==ButtonStillUp	-- Ignore all mouse down and mouse move events
							= (True,False,0,undef,WItemHandle itemH,inputTrack)
						| buttonstate==ButtonUp							-- Clear mouse tracking
							= (True,okMouseAtt,wItemNr,mousestate,WItemHandle itemH,untrackMouse inputTrack)
						-- otherwise
							= (True,okMouseAtt,wItemNr,mousestate,WItemHandle itemH,inputTrack)
					| isDownButton										-- Mouse down event sets input track
						= (True,okMouseAtt,wItemNr,mousestate,WItemHandle itemH,trackMouse wPtr itemPtr inputTrack)
					| isMember buttonstate [ButtonStillDown,ButtonUp]	-- Ignore all mouse drag and up events when not tracking
						= (True,False,0,undef,WItemHandle itemH,inputTrack)
					| otherwise
						= (True,okMouseAtt,wItemNr,mousestate,WItemHandle itemH,inputTrack)
				where
					contextAble1= contextAble && wItemSelect
					(filter,selectState,_)
								= getControlMouseAtt (snd (cselect isControlMouse (ControlMouse (const False) Unable undef) wItemAtts))
					okMouseAtt	= contextAble1 && enabled selectState && filter mousestate
					mousestate	= mouseState action (origin+eventPos)
					buttonstate	= getMouseStateButtonState mousestate
					isDownButton= isMember buttonstate [ButtonDown,ButtonDoubleDown,ButtonTripleDown]
					origin		= case wItemKind of
									IsCustomButtonControl	-> zero
									IsCustomControl			-> zero
									IsCompoundControl		-> (getWItemCompoundInfo wItemInfo).compoundOrigin
									_						-> windoweventFatalError "okControlItemsNrMouseState" "mouse event generated for unexpected control"
									
				okControlItemNrMouseState contextAble wPtr itemPtr action eventPos (WListLSHandle itemHs) inputTrack
					# (found,ok,itemNr,mousestate,itemHs,inputTrack)	= okControlsItemNrMouseState contextAble wPtr itemPtr action eventPos itemHs inputTrack
					= (found,ok,itemNr,mousestate,WListLSHandle itemHs,inputTrack)
				
				okControlItemNrMouseState contextAble wPtr itemPtr action eventPos (WExtendLSHandle wExH=:{wExtendItems=itemHs}) inputTrack
					# (found,ok,itemNr,mousestate,itemHs,inputTrack)	= okControlsItemNrMouseState contextAble wPtr itemPtr action eventPos itemHs inputTrack
					= (found,ok,itemNr,mousestate,WExtendLSHandle {wExH & wExtendItems=itemHs},inputTrack)
				
				okControlItemNrMouseState contextAble wPtr itemPtr action eventPos (WChangeLSHandle wChH=:{wChangeItems=itemHs}) inputTrack
					# (found,ok,itemNr,mousestate,itemHs,inputTrack)	= okControlsItemNrMouseState contextAble wPtr itemPtr action eventPos itemHs inputTrack
					= (found,ok,itemNr,mousestate,WChangeLSHandle {wChH & wChangeItems=itemHs},inputTrack)
			
			okControlsItemNrMouseState _ _ _ _ _ [] inputTrack
				= (False,False,0,undef,[],inputTrack)
		
		okControlItemsNrMouseState _ _ _ _ _ _
			= windoweventFatalError "okControlItemsNrMouseState" "placeholder not expected"
where
	modifiers				= toModifiers mods
	nrDown					= case action of
								BUTTONDOWN			-> 1
								BUTTONDOUBLEDOWN	-> 2
							 	_					-> 3
	mouseState action pos	= case action of
								BUTTONSTILLUP		-> MouseMove pos modifiers
								BUTTONUP			-> MouseUp   pos modifiers
								BUTTONSTILLDOWN		-> MouseDrag pos modifiers
								_					-> MouseDown pos modifiers nrDown
-}
{-
{-	The ccWmPAINT message is generated to update the indicated rectangle of the argument window.
-}
filterPAINT :: OSWindowMetrics -> OSEvent -> WindowHandles IF_MVAR(,ps) -> IO (Bool,Maybe [Int],Maybe DeviceEvent)
filterPAINT _ (CrossCallInfo {p1=wptr,p2=left,p3=top,p4=right,p5=bottom,p6=gc}) windows
	| not found
		= return (False,Nothing,Nothing)
	| otherwise
		= let (wids,wsH1) = getWindowStateHandleWIDS wsH
		      updRect     = fromTuple4 (left,top,right,bottom)
		      updateInfo  = UpdateInfo
		                       { updWIDS       = wids
		                       , updWindowArea = updRect
		                       , updControls   = []
		                       , updGContext   = if gc==0 then Nothing else Just gc
		                       }
		  in  return (True,Nothing,Just (WindowUpdate updateInfo))
	where
		(found,wsH,_)     = getWindowHandlesWindow (toWID wptr) windows
-}
{-
filterSCROLLBARACTION :: OSWindowMetrics -> OSEvent -> WindowHandles IF_MVAR(,ps) -> IO (Bool,Maybe [Int],Maybe DeviceEvent)
filterSCROLLBARACTION _ (CrossCallInfo {p1=wPtr,p2=cPtr,p3=iBar,p4=action,p5=osThumb}) windows ioState
	# (found,wsH,windows)	= getWindowHandlesWindow (toWID wPtr) windows
	| not found
		= (False,Nothing,Nothing,windows,ioState)
	# (able,wsH)			= getWindowStateHandleSelect wsH
	| not able
		= (True,Nothing,Nothing,setWindowHandlesWindow wsH windows,ioState)
	| otherwise
		# (wids,wsH)		= getWindowStateHandleWIDS wsH
		  (sliderEvent,wsH)	= getSlidersEvent wids iBar osThumb cPtr wsH
		= (True,Nothing,Just sliderEvent,setWindowHandlesWindow wsH windows,ioState)
where
	getSlidersEvent :: !WIDS !Int !Int !OSWindowPtr !(WindowStateHandle .pst) -> (!DeviceEvent,!WindowStateHandle .pst)
	getSlidersEvent wids iBar osThumb itemPtr wsH=:{wshHandle=Just wlsH=:{wlsHandle=wH=:{whWindowInfo,whItems,whSize={w,h}}}}
		| wids.wPtr==itemPtr
			= (WindowScrollAction info,wsH)
		with
			info				= {	wsaWIDS			= wids
								  ,	wsaSliderMove	= move min max view osThumb
								  ,	wsaDirection	= if isHorizontal Horizontal Vertical
								  }
			windowInfo			= getWindowInfoWindowData whWindowInfo
			domainRect			= windowInfo.windowDomain
			isHorizontal		= iBar==SB_HORZ
			(min,max,view)		= if isHorizontal
									(domainRect.rleft,domainRect.rright, w)
									(domainRect.rtop, domainRect.rbottom,h)
		# (found,sliderEvent,itemHs)= getSlidersEvent wids iBar osThumb itemPtr whItems
		| found
			= (sliderEvent,{wsH & wshHandle=Just {wlsH & wlsHandle={wH & whItems=itemHs}}})
		| otherwise
			= windoweventFatalError "getSlidersEvent" "SliderControl could not be located"
	where
		getSlidersEvent :: !WIDS !Int !Int !OSWindowPtr ![WElementHandle .ls .pst] -> (!Bool,!DeviceEvent,![WElementHandle .ls .pst])
		getSlidersEvent wids iBar osThumb itemPtr [itemH:itemHs]
			# (found,sliderEvent,itemH)		= getSliderEvent wids iBar osThumb itemPtr itemH
			| found
				= (found,sliderEvent,[itemH:itemHs])
			| otherwise
				# (found,sliderEvent,itemHs)= getSlidersEvent wids iBar osThumb itemPtr itemHs
				= (found,sliderEvent,[itemH:itemHs])
		where
			getSliderEvent :: !WIDS !Int !Int !OSWindowPtr !(WElementHandle .ls .pst) -> (!Bool,!DeviceEvent,!WElementHandle .ls .pst)
			getSliderEvent wids iBar osThumb itemPtr (WItemHandle itemH=:{wItemPtr,wItemNr,wItemKind,wItemShow,wItems,wItemInfo,wItemSize})
				| itemPtr<>wItemPtr
					| wItemShow
						# (found,sliderEvent,itemHs)	= getSlidersEvent wids iBar osThumb itemPtr wItems
						= (found,sliderEvent,WItemHandle {itemH & wItems=itemHs})
					-- otherwise
						= (False,ControlSliderAction dummySlidersEvent,WItemHandle itemH)
				| wItemKind==IsCompoundControl
					= (True,CompoundScrollAction info,WItemHandle itemH)
				with
					info			= {	csaWIDS			= wids
									  ,	csaItemNr		= wItemNr
									  ,	csaItemPtr		= itemPtr
									  ,	csaSliderMove	= move min max view osThumb
									  ,	csaDirection	= if isHorizontal Horizontal Vertical
									  }
					compoundInfo	= getWItemCompoundInfo wItemInfo
					domainRect		= compoundInfo.compoundDomain
					isHorizontal	= iBar==SB_HORZ
					(min,max,view)	= if isHorizontal
										(domainRect.rleft,domainRect.rright, wItemSize.w)
										(domainRect.rtop, domainRect.rbottom,wItemSize.h)
				| otherwise
					= (True,ControlSliderAction info,WItemHandle itemH)
				with
					info			= {	cslWIDS			= wids
									  ,	cslItemNr		= wItemNr
									  ,	cslItemPtr		= itemPtr
									  ,	cslSliderMove	= move sliderState.sliderMin sliderState.sliderMax 0 osThumb
									  }
					sliderInfo		= getWItemSliderInfo wItemInfo
					sliderState		= sliderInfo.sliderInfoState
			
			getSliderEvent wids iBar osThumb itemPtr (WListLSHandle itemHs)
				# (found,sliderEvent,itemHs)	= getSlidersEvent wids iBar osThumb itemPtr itemHs
				= (found,sliderEvent,WListLSHandle itemHs)
			
			getSliderEvent wids iBar osThumb itemPtr (WExtendLSHandle wExH=:{wExtendItems=itemHs})
				# (found,sliderEvent,itemHs)	= getSlidersEvent wids iBar osThumb itemPtr itemHs
				= (found,sliderEvent,WExtendLSHandle {wExH & wExtendItems=itemHs})
			
			getSliderEvent wids iBar osThumb itemPtr (WChangeLSHandle wChH=:{wChangeItems=itemHs})
				# (found,sliderEvent,itemHs)	= getSlidersEvent wids iBar osThumb itemPtr itemHs
				= (found,sliderEvent,WChangeLSHandle {wChH & wChangeItems=itemHs})
		
		getSlidersEvent _ _ _ _ []
			= (False,ControlSliderAction dummySlidersEvent,[])
		
		dummySlidersEvent	= { cslWIDS=wids,cslItemNr=0,cslItemPtr=0,cslSliderMove=SliderIncSmall }
	
	getSlidersEvent _ _ _ _ _
		= windoweventFatalError "getSlidersEvent" "placeholder not expected"
	
	move :: !Int !Int !Int !Int -> SliderMove
	move min max view osThumb
		= case action of
			SB_LINEUP		-> SliderDecSmall
			SB_LINEDOWN		-> SliderIncSmall
			SB_PAGEUP		-> SliderDecLarge
			SB_PAGEDOWN		-> SliderIncLarge
			SB_THUMBPOSITION-> SliderThumb (fromOSscrollbarRange (min,max) osThumb)
			SB_THUMBTRACK	-> SliderThumb (fromOSscrollbarRange (min,max) osThumb)
			SB_TOP			-> SliderThumb min
			SB_BOTTOM		-> SliderThumb (max-view)
			SB_ENDSCROLL	-> SliderThumb (fromOSscrollbarRange (min,max) osThumb)
-}
{-
filterSETFOCUS :: OSWindowMetrics -> OSEvent -> WindowHandles IF_MVAR(,ps) -> IO (Bool,Maybe [Int],Maybe DeviceEvent)
filterSETFOCUS _ (CrossCallInfo {p1=wPtr,p2=cPtr}) windows ioState
	# (found,wsH,windows)	= getWindowHandlesWindow (toWID wPtr) windows
	| not found
		= (False,Nothing,Nothing,windows,ioState)
	# (wids,wsH)			= getWindowStateHandleWIDS wsH
	  (found,itemNr,wsH)	= getControlKeyFocusItemNr True cPtr wsH
	  windows				= setWindowHandlesWindow wsH windows
	| not found
		= (True,Nothing,Nothing,windows,ioState)
	| otherwise
		= (True,Nothing,Just (ControlGetKeyFocus {ckfWIDS=wids,ckfItemNr=itemNr,ckfItemPtr=cPtr}),windows,ioState)
-}
{-
filterSIZE :: OSWindowMetrics -> OSEvent -> WindowHandles IF_MVAR(,ps) -> IO (Bool,Maybe [Int],Maybe DeviceEvent)
filterSIZE wMetrics (CrossCallInfo {p1=wptr,p2=w,p3=h,p4=usersizing}) windows
	| not found
		= return (False,Nothing,Nothing,windows1)
	| wKind==IsDialog		-- This alternative should never occur
		= windoweventFatalError "filterSIZE" "WindowSizeAction event generated for Dialog"
	| otherwise
		= let (wids,wsH2) = getWindowStateHandleWIDS wsH1
		      (info,wsH3) = getWindowStateHandleSize wids w h (usersizing\=0) wsH2
		      windows2    = setWindowHandlesWindow wsH3 windows1
		  in  return (True,Nothing,Just (WindowSizeAction info),windows2)
	where
		(found,wsH,windows1) = getWindowHandlesWindow (toWID wptr) windows
		(wKind,wsH1)         = getWindowStateHandleWindowKind wsH
		
		getWindowStateHandleSize :: WIDS -> Int -> Int -> Bool -> WindowStateHandle IF_MVAR(,ps)
		                                 -> (WindowSizeActionInfo,WindowStateHandle IF_MVAR(,ps))
#if MVAR
		getWindowStateHandleSize wids newW newH usersizing wsH@(WindowStateHandle {wshHandle=Just wH})   =:{whSize,whWindowInfo}}}
#else
		getWindowStateHandleSize wids newW newH usersizing wsH@(WindowStateHandle wids (Just (WindowLSHandle {wlsHandle=wH})))
#endif
			= (WindowSizeActionInfo
			      { wsWIDS=wids
			      , wsSize=Size {w=newW',h=newH'}
			      , wsUpdateAll=not usersizing
			      }
			  , wsH
			  )
			where
				windowInfo = getWindowInfoWindowData (whWindowInfo wH)
				domainRect = windowDomain windowInfo
				hasScrolls = (isJust (windowHScroll windowInfo),isJust (windowVScroll windowInfo))
				(visHScroll,visVScroll)
				           = osScrollbarsAreVisible wMetrics domainRect (toTuple whSize) hasScrolls
				newW'      = if visVScroll (newW+osmVSliderWidth wMetrics)  newW	-- Correct newW in case of visible vertical   scrollbar
				newH'      = if visHScroll (newH+osmHSliderHeight wMetrics) newH	-- Correct newH in case of visible horizontal scrollbar
		getWindowStateHandleSize _ _ _ _ _
			= windoweventFatalError "getWindowStateHandleSize" "placeholder not expected"
-}
{-
filterSPECIALBUTTON :: OSWindowMetrics -> OSEvent -> WindowHandles IF_MVAR(,ps) -> IO (Bool,Maybe [Int],Maybe DeviceEvent)
filterSPECIALBUTTON _ (CrossCallInfo {p1=wPtr,p2=okOrCancel}) windows ioState
	# (found,wsH,windows)	= getWindowHandlesWindow (toWID wPtr) windows
	| not found
		= (False,Nothing,Nothing,windows,ioState)
	| otherwise
		# (wids,wsH)		= getWindowStateHandleWIDS      wsH
		  (okId,wsH)		= getWindowStateHandleDefaultId wsH
		  (cancelId,wsH)	= getWindowStateHandleCancelId  wsH
		  okOrCancelEvent	= if (okOrCancel==ISOKBUTTON)     (if (isJust okId)     (Just (WindowOK     wids)) Nothing)
		  					 (if (okOrCancel==ISCANCELBUTTON) (if (isJust cancelId) (Just (WindowCANCEL wids)) Nothing)
		  					 								  (windoweventFatalError "filterOSEvent (ccWmSPECIALBUTTON)" "incorrect argument"))
		= (True,Nothing,okOrCancelEvent,setWindowHandlesWindow wsH windows,ioState)
-}

toModifiers :: Int -> Modifiers
toModifiers i
	= Modifiers
		{ shiftDown   = shifton
		, optionDown  = alton
		, commandDown = ctrlon
		, controlDown = ctrlon
		, altDown     = alton
		}
	where
		shifton = (i1 .&. (i32 shiftBIT)) /= 0
		alton   = (i1 .&. (i32 altBIT))   /= 0
		ctrlon  = (i1 .&. (i32 ctrlBIT))  /= 0
		i1      = i32 i

{-
getControlKeyFocusItemNr :: !Bool !OSWindowPtr !(WindowStateHandle .pst) -> (!Bool,!Int,!WindowStateHandle .pst)
getControlKeyFocusItemNr activated cPtr wsH=:{wshHandle=Just wlsH=:{wlsHandle=wH}}
	# (found,itemNr,itemHs)	= getControlsKeyFocusItemNr` activated cPtr wH.whItems
	= (found,itemNr,{wsH & wshHandle=Just {wlsH & wlsHandle={wH & whItems=itemHs}}})
where
	getControlsKeyFocusItemNr` :: !Bool !OSWindowPtr ![WElementHandle .ls .pst] -> (!Bool,!Int,![WElementHandle .ls .pst])
	getControlsKeyFocusItemNr` activated cPtr []
		= (False,0,[])
	getControlsKeyFocusItemNr` activated cPtr [itemH:itemHs]
		# (found,itemNr,itemH)		= getControlKeyFocusItemNr` activated cPtr itemH
		| found
			= (found,itemNr,[itemH:itemHs])
		| otherwise
			# (found,itemNr,itemHs)	= getControlsKeyFocusItemNr` activated cPtr itemHs
			= (found,itemNr,[itemH:itemHs])
	where
		getControlKeyFocusItemNr` :: !Bool !OSWindowPtr !(WElementHandle .ls .pst) -> (!Bool,!Int,!WElementHandle .ls .pst)
		getControlKeyFocusItemNr` activated cPtr (WItemHandle itemH=:{wItemPtr,wItemNr,wItemKind,wItemSelect,wItemAtts,wItems})
			| cPtr==wItemPtr
				| not (isMember wItemKind [IsCompoundControl,IsCustomControl,IsEditControl,IsPopUpControl])
					= (True,0,WItemHandle itemH)
				-- otherwise
					= (True,wItemNr,WItemHandle itemH)
			| not (isRecursiveControl wItemKind)
				= (False,0,WItemHandle itemH)
			| otherwise
				# (found,itemNr,itemHs)	= getControlsKeyFocusItemNr` activated cPtr wItems
				= (found,itemNr,WItemHandle {itemH & wItems=itemHs})
		
		getControlKeyFocusItemNr` activated cPtr (WListLSHandle itemHs)
			# (found,itemNr,itemHs)	= getControlsKeyFocusItemNr` activated cPtr itemHs
			= (found,itemNr,WListLSHandle itemHs)
		
		getControlKeyFocusItemNr` activated cPtr (WExtendLSHandle wExH=:{wExtendItems=itemHs})
			# (found,itemNr,itemHs)	= getControlsKeyFocusItemNr` activated cPtr itemHs
			= (found,itemNr,WExtendLSHandle {wExH & wExtendItems=itemHs})
		
		getControlKeyFocusItemNr` activated cPtr (WChangeLSHandle wChH=:{wChangeItems=itemHs})
			# (found,itemNr,itemHs)	= getControlsKeyFocusItemNr` activated cPtr itemHs
			= (found,itemNr,WChangeLSHandle {wChH & wChangeItems=itemHs})
getControlKeyFocusItemNr _ _ _
	= windoweventFatalError "getControlKeyFocusItemNr" "window placeholder not expected"
-}

{-	The following operations on InputTrack are ignored.
--	Access operations on InputTrack:

trackingMouse :: !OSWindowPtr !OSWindowPtr !(Maybe InputTrack) -> Bool
trackingMouse wPtr cPtr (Just {itWindow,itControl,itKind={itkMouse}})
	= wPtr==itWindow && cPtr==itControl && itkMouse
trackingMouse _ _ _
	= False

trackingKeyboard :: !OSWindowPtr !OSWindowPtr !(Maybe InputTrack) -> Bool
trackingKeyboard wPtr cPtr (Just {itWindow,itControl,itKind={itkKeyboard}})
	= wPtr==itWindow && cPtr==itControl && itkKeyboard
trackingKeyboard _ _ _
	= False

trackMouse :: !OSWindowPtr !OSWindowPtr !(Maybe InputTrack) -> Maybe InputTrack
trackMouse wPtr cPtr (Just it=:{itWindow,itControl,itKind=itk})
	| wPtr<>itWindow || cPtr<>itControl
		= windoweventFatalError "trackMouse" "incorrect window/control parameters"
	| otherwise
		= Just {it & itKind={itk & itkMouse=True}}
trackMouse wPtr cPtr nothing
	= Just {itWindow=wPtr,itControl=cPtr,itKind={itkMouse=True,itkKeyboard=False}}

untrackMouse :: !(Maybe InputTrack) -> Maybe InputTrack
untrackMouse (Just it=:{itKind=itk})
	| itk.itkKeyboard
		= Just {it & itKind={itk & itkMouse=False}}
	| otherwise
		= Nothing
untrackMouse nothing
	= nothing

untrackKeyboard :: !(Maybe InputTrack) -> Maybe InputTrack
untrackKeyboard (Just it=:{itKind=itk})
	| itk.itkMouse
		= Just {it & itKind={itk & itkKeyboard=False}}
	| otherwise
		= Nothing
untrackKeyboard nothing
	= nothing

trackKeyboard :: !OSWindowPtr !OSWindowPtr !(Maybe InputTrack) -> Maybe InputTrack
trackKeyboard wPtr cPtr (Just it=:{itWindow,itControl,itKind=itk})
	| wPtr<>itWindow || cPtr<>itControl
		= windoweventFatalError "trackKeyboard" "incorrect window/control parameters"
	| otherwise
		= Just {it & itKind={itk & itkKeyboard=True}}
trackKeyboard wPtr cPtr nothing
	= Just {itWindow=wPtr,itControl=cPtr,itKind={itkMouse=False,itkKeyboard=True}}
-}
