module StdReceiver ( Receivers(..)
                   , closeReceiver
                   , getReceivers
                   , enableReceivers, disableReceivers, getReceiverSelectState
                   , SendReport(..), asyncSend
                   ) where


--	********************************************************************************
--	Clean Standard Object I/O library, version 1.2
--	
--	StdReceiver specifies all receiver operations.
--	********************************************************************************


import CleanStdList
import CleanStdMisc
import Commondef
import Deviceevents
import Id
import IOExts (fixIO)
import IOstate
import Osevent
import Receiveraccess
import Receiverdefaccess
import Receiverdevice
import Receiverhandle
import Receiverid
import Receivertable
import Scheduler
import StdReceiverAttribute
import Trace_12
#include "preprocess.h"
         -- Compile with -cpp; set MVAR=1 for MVAR version


stdReceiverFatalError :: String -> String -> x
stdReceiverFatalError rule error
	= dumpFatalError rule "StdReceiver" error


--	Open one-way receiver:

receiverStateIdentified :: Id -> ReceiverStateHandle IF_MVAR(,ps) -> Bool
receiverStateIdentified id (ReceiverStateHandle IF_MVAR(,_) rH) = receiverIdentified id rH

class Receivers rdef where
#if MVAR
	openReceiver :: rdef -> GUI ErrorReport
#else
	openReceiver :: ls -> rdef ls ps -> ps -> GUI ps (ErrorReport,ps)
#endif

instance Receivers (Receiver m) where
	openReceiver IF_MVAR(,ls) rDef IF_MVAR(,ps)
		= do {
			IF_MVAR(,ps1 <-) dOpen receiverFunctions IF_MVAR(,ps);
			idtable      <-  ioStGetIdTable;
			if   memberIdTable id idtable
#if MVAR
			then return ErrorIdsInUse
#else
			then return (ErrorIdsInUse,ps1)
#endif
			else 
			do {
				rt   <- ioStGetReceiverTable;
				let maybe_parent = getReceiverTableEntry id rt
				in
				if   isJust maybe_parent	-- This condition should not occur: IdTable didn't contain Id while ReceiverTable does.
				then stdReceiverFatalError "openReceiver (Receiver)" "inconsistency detected between IdTable and ReceiverTable"
				else
				do {
					(found,rDevice) <- accIOEnv (ioStGetDevice ReceiverDevice);
					if   not found		-- This condition should not occur: ReceiverDevice has just been 'installed'
					then stdReceiverFatalError "openReceiver (Receiver)" "could not retrieve ReceiverSystemState from IOSt"
					else
					let  rsHs       = rReceivers (receiverSystemStateGetReceiverHandles rDevice)
#if MVAR
					     rsH        = newReceiverStateHandle rid select connectedIds f
					     receivers2 = ReceiverHandles {rReceivers = rsH : rsHs}
					in
					do {
						appIOEnv (ioStSetDevice (ReceiverSystemState receivers2));
						bindRId id select id ReceiverDevice;
						ioId <- accIOEnv ioStGetIOId;
						ioStSetIdTable (snd (addIdToIdTable id (IdParent {idpIOId=ioId,idpDevice=ReceiverDevice,idpId=id}) idtable));
						receiverInit;
						return NoError
					}
#else
					in
					do {
						bindRId id select id ReceiverDevice;
						ioId <- accIOEnv ioStGetIOId;
						ioStSetIdTable (snd (addIdToIdTable id (IdParent {idpIOId=ioId,idpDevice=ReceiverDevice,idpId=id}) idtable));
						ps2  <- toGUI (doInitIO (receiverInit (ls,ps1))
						                        (\ls ioState -> ioStSetDevice
						                                            (ReceiverSystemState
						                                                 (ReceiverHandles
						                                                     {rReceivers = (newReceiverStateHandle rid ls select connectedIds f) : rsHs}))
						                                            ioState));
						return (NoError,ps2)
					}
#endif
				}
			}
		  }
		where
			rid            = receiverDefRId         rDef
			select         = receiverDefSelectState rDef
			f              = receiverDefFunction    rDef
			rDefAttributes = receiverDefAttributes  rDef
			receiverInit   = getReceiverInitFun (snd (cselect isReceiverInit (ReceiverInit IF_MVAR((return ()),return)) rDefAttributes))
		--	connectedIds   = getReceiverConnectedReceivers (snd (cselect isReceiverConnectedReceivers (ReceiverConnectedReceivers []) rDefAttributes))
			connectedIds   = []
			id             = rIdtoId rid

{-	Bi-directional receivers not yet implemented.
instance Receivers (Receiver2 m r) where
	openReceiver IF_MVAR(,ls) rDef IF_MVAR(,ps)
		= do {
			IF_MVAR(,ps1 <-) dOpen receiverFunctions IF_MVAR(,ps);
			idtable      <-  ioStGetIdTable;
			if   memberIdTable id idtable
#if MVAR
			then return ErrorIdsInUse
#else
			then return (ErrorIdsInUse,ps1)
#endif
			else 
			do {
				rt   <-  ioStGetReceiverTable;
				let maybe_parent = getReceiverTableEntry id rt
				in
				if   isJust maybe_parent	-- This condition should not occur: IdTable didn't contain Id while ReceiverTable does.
				then stdReceiverFatalError "openReceiver (Receiver2)" "inconsistency detected between IdTable and ReceiverTable"
				else
				do {
					(found,rDevice) <- accIOEnv (ioStGetDevice ReceiverDevice);
					if   not found		-- This condition should not occur: ReceiverDevice has just been 'installed'
					then stdReceiverFatalError "openReceiver (Receiver2)" "could not retrieve ReceiverSystemState from IOSt"
					else
					let  rsHs       = rReceivers (receiverSystemStateGetReceiverHandles rDevice)
#if MVAR
					     rsH        = newReceiverStateHandle2 id select connectedIds f
					     receivers2 = ReceiverHandles {rReceivers = rsH : rsHs}
					in
					do {
						appIOEnv (ioStSetDevice (ReceiverSystemState receivers2));
						bindRId id select id ReceiverDevice;
						ioId   <- accIOEnv ioStGetIOId;
						ioStSetIdTable (snd (addIdToIdTable id (IdParent {idpIOId=ioId,idpDevice=ReceiverDevice,idpId=id})));
						receiverInit;
						return NoError
					}
#else
					in
					do {
						ps2 <- toGUI (doInitIO (receiverInit (ls,ps1))
						                       (\ls ioState -> ioStSetDevice
						                                           (ReceiverSystemState
						                                                (ReceiverHandles
						                                                    {rReceivers = (newReceiverStateHandle2 id ls select connectedIds f) : rsHs}))
						                                           ioState));
						return (NoError,ps2)
					}
#endif
				}
			}
		  }
		where
			r2id           = receiver2DefR2Id        rDef
			select         = receiver2DefSelectState rDef
			f              = receiver2DefFunction    rDef
			rDefAttributes = receiver2DefAttributes rDef
			receiverInit   = getReceiverInitFun (snd (cselect isReceiverInit (ReceiverInit IF_MVAR((return ()),return)) rDefAttributes))
		--	connectedIds   = getReceiverConnectedReceivers (snd (cselect isReceiverConnectedReceivers (ReceiverConnectedReceivers []) rDefAttributes))
			connectedIds   = []
			id             = r2IdtoId r2id
-}

#if MVAR
#else
doInitIO :: GUI ps (ls,ps) -> (ls -> IOSt ps -> IOSt ps) -> IOSt ps -> IO (ps,IOSt ps)
doInitIO initGUI setReceiverLS ioState
	= do {
		r <- fixIO (\st -> fromGUI initGUI (setReceiverLS (fst (fst st)) ioState));
		let ((_,ps1),ioState1) = r
		in  return (ps1,ioState1)
	  }
#endif


--	Closing receivers.

closeReceiver :: Id -> GUI IF_MVAR(,ps) ()
closeReceiver id
	= do {
		closed <- accIOEnv ioStClosed;
		if   closed
		then return ()
		else 
		if   not (isCustomRId id || isCustomR2Id id || isCustomId id)
		then return ()
		else
		do {
			(found,rDevice) <- accIOEnv (ioStGetDevice ReceiverDevice);
			if   not found
			then return ()
			else 
			let  rsHs              = rReceivers (receiverSystemStateGetReceiverHandles rDevice)
			     (found,rsH,rsHs1) = remove (receiverStateIdentified id) undef rsHs
			in
			do {
				appIOEnv (ioStSetDevice (ReceiverSystemState (ReceiverHandles {rReceivers=rsHs1})));
				if   not found
				then return ()
				else 
				do {
					idtable <- ioStGetIdTable;
					ioStSetIdTable (snd (removeIdFromIdTable id idtable));
					unbindRId id;
				{-	TCP not supported yet
					appIOEnv (ioStSetRcvDisabled True);
					sequence_ [closeReceiver id | id <- rConnected $ rHandle $ rsH];
					case rInetInfo $ rHandle $ rsH of
					     Nothing               -> return ()
					     Just (_,_,_,closefun) -> closefun
				-}
				}
			}
		}
	  }


--	Get the Ids and ReceiverTypes of all current receivers:

getReceivers :: GUI IF_MVAR(,ps) [(Id,ReceiverType)]
getReceivers
	= do {
		(found,rDevice) <- accIOEnv (ioStGetDevice ReceiverDevice);
		if   not found
		then return []
		else 
		let  (idstypes,rsHs) = getreceivers $ rReceivers $ receiverSystemStateGetReceiverHandles rDevice
		in   appIOEnv (ioStSetDevice (ReceiverSystemState (ReceiverHandles {rReceivers=rsHs}))) >> return idstypes
	  }
	where
		getreceivers :: [ReceiverStateHandle IF_MVAR(,ps)] -> ([(Id,ReceiverType)],[ReceiverStateHandle IF_MVAR(,ps)])
		getreceivers (rsH:rsHs)
			= let (idtype, rsH1)  = getreceiver  rsH
			      (idtypes,rsHs1) = getreceivers rsHs
			  in  (idtype : idtypes, rsH1 : rsHs1)
			where
				getreceiver :: ReceiverStateHandle IF_MVAR(,ps) -> ((Id,ReceiverType),ReceiverStateHandle IF_MVAR(,ps))
				getreceiver rsH@(ReceiverStateHandle IF_MVAR(,_) (ReceiverHandle {rId=rId,rOneWay=rOneWay}))
					= ((rId,if rOneWay then "Receiver" else "Receiver2"),rsH)
		getreceivers _
			= ([],[])


--	Changing attributes:

enableReceivers :: [Id] -> GUI IF_MVAR(,ps) ()
enableReceivers ids
	= changeReceivers (receiverSetSelectState Able) (receiverEntrySetSelectState Able) ids

disableReceivers :: [Id] -> GUI IF_MVAR(,ps) ()
disableReceivers ids
	= -- appIOEnv (ioStSetRcvDisabled True) >> (
	     changeReceivers (receiverSetSelectState Unable) (receiverEntrySetSelectState Unable) ids -- )

receiverEntrySetSelectState :: SelectState -> ReceiverTableEntry -> ReceiverTableEntry
receiverEntrySetSelectState select rte
	= rte {rteSelectState=select}

changeReceivers :: IdFun (ReceiverStateHandle IF_MVAR(,ps)) -> IdFun ReceiverTableEntry -> [Id] -> GUI IF_MVAR(,ps) ()
changeReceivers changeReceiverState changeReceiverEntry ids
	| isEmpty okids			-- There aren't any receiver ids in the list
		= return ()
	| otherwise
		= do {
			(found,rDevice) <- accIOEnv (ioStGetDevice ReceiverDevice);
			if   not found
			then return ()
			else
			let  rsHs          = rReceivers $ receiverSystemStateGetReceiverHandles rDevice
			     allIds        = okids -- getConnectedIds okids [] rsHs rsHs
			     (myids,rsHs1) = changereceiverstates changeReceiverState allIds rsHs
			in
			do {
				appIOEnv (ioStSetDevice (ReceiverSystemState (ReceiverHandles {rReceivers=rsHs1})));
				if   isEmpty myids	-- No receivers were changed
				then return ()
				else ioStGetReceiverTable >>= (\rt -> 
					ioStSetReceiverTable (changereceiverentries changeReceiverEntry myids rt))
			}
		  }
	where
		okids = filter (\id->isCustomRId id || isCustomR2Id id || isCustomId id) ids
		
	{-	TCP not supported yet:
		getConnectedIds :: [Id] -> [Id] -> [ReceiverStateHandle ps] -> [ReceiverStateHandle ps] -> [Id]
		getConnectedIds ids _ [] _
			= ids
		getConnectedIds ids alreadyHandled (rsH@(ReceiverStateHandle {rHandle=ReceiverHandle {rId=rId,rConnected=rConnected}}) : rsHs) allStateHandles
			| not (isMember rId ids) || isEmpty rConnected || isMember rId alreadyHandled
				= getConnectedIds ids alreadyHandled rsHs allStateHandles
			| otherwise	-- search again in the whole set of receivers
				= getConnectedIds (removeDup (rConnected++ids)) (rId:alreadyHandled) allStateHandles allStateHandles
	-}	
		changereceiverstates :: IdFun (ReceiverStateHandle IF_MVAR(,ps)) -> [Id] -> [ReceiverStateHandle IF_MVAR(,ps)]
		                     -> ([Id],[ReceiverStateHandle IF_MVAR(,ps)])
		changereceiverstates _ [] _
			= ([],[])
		changereceiverstates _ _ []
			= ([],[])
		changereceiverstates f ids (rsH@(ReceiverStateHandle IF_MVAR(,_) (ReceiverHandle {rId=rId})) : rsHs)
			| hadId       = (rId:rIds, (f rsH): rsHs1)
			| otherwise   = (    rIds,    rsH : rsHs1)
			where
				(hadId,_,ids1) = remove ((==) rId) (dummy "changereceiverstates") ids
				(rIds,rsHs1)   = changereceiverstates f ids1 rsHs
		
		changereceiverentries :: IdFun ReceiverTableEntry -> [Id] -> ReceiverTable -> ReceiverTable
		changereceiverentries f (id:ids) rt
			= case getReceiverTableEntry id rt of
			       Nothing  -> changereceiverentries f ids rt
			       Just rte -> changereceiverentries f ids (setReceiverTableEntry (f rte) rt)
		changereceiverentries _ _ rt
			= rt


--	Get the SelectState of a receiver:

getReceiverSelectState :: Id -> GUI IF_MVAR(,ps) (Maybe SelectState)
getReceiverSelectState id
	| not (isCustomRId id || isCustomR2Id id || isCustomId id)
		= return Nothing
	| otherwise
		= do {
			(found,rDevice) <- accIOEnv (ioStGetDevice ReceiverDevice);
			if   not found
			then return Nothing
			else 
			let  rsHs           = rReceivers $ receiverSystemStateGetReceiverHandles rDevice
			     (select,rsHs1) = getselectstate id rsHs
			in   appIOEnv (ioStSetDevice (ReceiverSystemState (ReceiverHandles {rReceivers=rsHs1}))) >> return select
		  }
	where
		getselectstate :: Id -> [ReceiverStateHandle IF_MVAR(,ps)] -> (Maybe SelectState,[ReceiverStateHandle IF_MVAR(,ps)])
		getselectstate id (rsH@(ReceiverStateHandle IF_MVAR(,_) rH@(ReceiverHandle {rSelect=rSelect})) : rsHs)
			| receiverIdentified id rH = (Just rSelect, rsH:rsHs )
			| otherwise                = (select,       rsH:rsHs1)
			where
				(select,rsHs1)     = getselectstate id rsHs
		getselectstate _ _
			= (Nothing,[])


--	Message passing:

data	SendReport
 =	SendOk
 |	SendUnknownReceiver
 |	SendUnableReceiver
 |	SendDeadlock
 |	OtherSendReport String
	deriving (Eq,Show)


{-	Asynchronous, uni-directional, message passing.
	If the receiver could not be found in the global ReceiverTable, 
		then return SendUnknownReceiver and do nothing.
	If the receiver could be found, then increase the length of the 
		asynchronous message queue of the receiver in the global 
		ReceiverTable.
		Add the message to the asynchronous message queue of the 
		receiver using the scheduler.
-}
#if MVAR
asyncSend :: RId msg -> msg -> GUI SendReport
#else
asyncSend :: RId msg -> msg -> ps -> GUI ps (SendReport,ps)
#endif
asyncSend rid msg IF_MVAR(,ps)
	= do {
		rt <- ioStGetReceiverTable;
		case getReceiverTableEntry (rIdtoId rid) rt of
#if MVAR
		     Nothing  -> return SendUnknownReceiver
#else
		     Nothing  -> return (SendUnknownReceiver,ps)
#endif
		     Just rte
		          -> do {
		                  liftIO (writeChan (getRIdIn rid) msg);	-- Put the msg in the async queue of the receiver
		                  osEvents <- ioStGetEvents;			-- Notify the receiver a message is waiting
		                  ioStSetEvents (osAppendEvents [ScheduleMsgEvent (ASyncMessage {asmRecLoc=rteLoc rte})] osEvents);
#if MVAR
		                  return SendOk
#else
		                  return (SendOk,ps)
#endif
		             }
	  }

{-	NOT GOING TO DO SYNCHRONOUS MESSAGE PASSING YET.
{-	Synchronous, uni-directional, message passing.
	If the receiver could not be found in the global ReceiverTable,
		then return SendUnknownReceiver and do nothing.
	If the receiver could be found, then let the receiver handle the
		synchronous message using the scheduler.
-}
syncSend :: !(RId msg) msg !(PSt .l) -> (!SendReport, !PSt .l)
syncSend rid msg pState
	# id             = RIdtoId rid
	# (rt,ioState)   = IOStGetReceiverTable pState.io
	  maybe_parent   = getReceiverTableEntry id rt
	| isNothing maybe_parent
		= (SendUnknownReceiver,{pState & io=ioState})
	# parent         = fromJust maybe_parent
	  rteLoc         = parent.rteLoc
	  pid            = rteLoc.rlIOId
	  sEvent         = { smRecLoc = rteLoc
	                   , smMsg    = openDynamic (RIdtoDId rid) msg
	                   , smResp   = []
	                   , smError  = []
	                   }
	# (ioid,ioState) = IOStGetIOId ioState
	# pState         = {pState & io=ioState}
	| pid==ioid
		= PStHandleSyncMessage sEvent pState
	| otherwise
		# (opt_error,_,pState) = cswitchProcess pid (ScheduleMsgEvent (SyncMessage sEvent)) pState
		  report               = if (isJust opt_error) (toSendError (fromJust opt_error)) SendOk
		= (report,pState)
where
	PStHandleSyncMessage :: !SyncMessage !(PSt .l) -> (!SendReport, !PSt .l)
	PStHandleSyncMessage sm pState
		# (_,schedulerEvent,pState) = handleOneEventForDevices (ScheduleMsgEvent (SyncMessage sm)) pState
		  sm                        = case schedulerEvent of
		                                  (ScheduleMsgEvent (SyncMessage sm)) -> sm
		                                  _                                   -> stdReceiverFatalError "syncSend" "unexpected scheduler event"
		  errors                    = sm.smError
		  report                    = if (isEmpty errors)
		                                  SendOk
		                                 (case (hd errors) of
		                                      ReceiverUnable  -> SendUnableReceiver
		                                      ReceiverUnknown -> SendUnknownReceiver
		                                 )
		= (report,pState)


{-	Synchronous, bi-directional, message passing.
	If the receiver could not be found in the global ReceiverTable,
		then return SendUnknownReceiver and do nothing.
	If the receiver could be found, then let the receiver handle the
		synchronous message using the scheduler.
-}
syncSend2 :: !(R2Id msg resp) msg !(PSt .l) -> (!(!SendReport,!Maybe resp), !PSt .l)
syncSend2 r2id msg pState
	# id           = R2IdtoId r2id
	# (rt,pState)  = accPIO IOStGetReceiverTable pState
	  maybe_parent = getReceiverTableEntry id rt
	| isNothing maybe_parent
		= ((SendUnknownReceiver,Nothing),pState)
	# parent       = fromJust maybe_parent
	  rteLoc       = parent.rteLoc
	  pid          = rteLoc.rlIOId
	  sEvent       = { smRecLoc = rteLoc
	                 , smMsg    = openDynamic (R2IdtoDId r2id) msg
	                 , smResp   = []
	                 , smError  = []
	                 }
	# (ioid,pState)= accPIO IOStGetIOId pState
	| pid==ioid
		= PStHandleSync2Message (R2IdtoDId` r2id) sEvent pState
	# (opt_error,resp,pState) = cswitchProcess pid (ScheduleMsgEvent (SyncMessage sEvent)) pState
	| isJust opt_error
		= ((toSendError (fromJust opt_error),Nothing),pState)
	| isEmpty resp
		= stdReceiverFatalError "syncSend2" "no response received from bi-directional receiver"
	# maybe_response          = readDynamic (R2IdtoDId` r2id) (hd resp)
	| isNothing maybe_response
		= ((OtherSendReport "incorrect response received from bi-directional receiver",Nothing),pState)
	| otherwise
		= ((SendOk,maybe_response),pState)
where
	PStHandleSync2Message :: !(DId resp) !SyncMessage !(PSt .l) -> (!(!SendReport,!Maybe resp), !PSt .l)
	PStHandleSync2Message did sm pState
		# (_,schedulerEvent,pState) = handleOneEventForDevices (ScheduleMsgEvent (SyncMessage sm)) pState
		  sm                        = case schedulerEvent of
		                                  (ScheduleMsgEvent (SyncMessage sm)) -> sm
		                                  _                                   -> stdReceiverFatalError "syncSend2" "unexpected scheduler event"
		  errors                    = sm.smError
		  resps                     = sm.smResp
		| not (isEmpty errors)
			# sendReport        = case (hd errors) of
			                           ReceiverUnable  -> SendUnableReceiver
			                           ReceiverUnknown -> SendUnknownReceiver
			= ((sendReport,Nothing),pState)
		| isEmpty resps
			= stdReceiverFatalError "syncSend2" "no response received from bi-directional receiver"
		# maybe_response            = readDynamic did (hd resps)
		| isNothing maybe_response
			= ((OtherSendReport "incorrect response received from bi-directional receiver",Nothing),pState)
		| otherwise
			= ((SendOk,maybe_response),pState)

toSendError :: SwitchError -> SendReport
toSendError SwitchToYourself             = SendUnknownReceiver
toSendError SwitchToDoesNotExist         = SendUnknownReceiver
toSendError SwitchToReceiverDoesNotExist = SendUnknownReceiver
toSendError SwitchReceiverUnable         = SendUnableReceiver
toSendError SwitchEndsUpInDeadlock       = SendDeadlock
-}
