module WindowdeviceLS ( windowFunctions ) where


--	********************************************************************************
--	Clean to Haskell Standard Object I/O library, version 1.2
--	
--	WindowdeviceLS defines the window device event handlers for the LS version only.
--	********************************************************************************


import CleanStdList
import CleanStdMisc
import CleanStdTuple
import Commondef
import Controlcreate
import Deviceevents
import Id
import IOExts (fixIO)
import IOstate
import Oswindow
import StdControlAttribute
import StdWindowAttribute
import Windowaccess
import Windowcreate (bufferDelayedEvents)
import Windowdispose
import Windowevent
import Windowhandle


windowdeviceFatalError :: String -> String -> x
windowdeviceFatalError function error
	= dumpFatalError function "Windowdevice" error


windowFunctions :: DeviceFunctions ps
windowFunctions
	= DeviceFunctions
		{ dDevice = WindowDevice
		, dEvent  = windowEvent
		, dDoIO   = windowIO
		, dOpen   = windowOpen
		, dClose  = windowClose
		}

{-	windowOpen initialises the window device for this interactive process.
-}
windowOpen :: ps -> GUI ps ps
windowOpen ps
	= do {
		hasWindow <- accIOEnv (ioStHasDevice WindowDevice);
		if   hasWindow
		then return ps
		else do {
			xDI <- accIOEnv ioStGetDocumentInterface;
			let bound   = case xDI of
			                NDI -> Finite 0
			                SDI -> Finite 1
			                MDI -> Infinite
			    windows = WindowHandles {whsWindows=[],whsNrWindowBound=bound}
			in  appIOEnv ( ioStSetDevice (WindowSystemState windows) . ioStSetDeviceFunctions windowFunctions ) >> return ps
		     }
	  }


{-	windowClose closes all windows associated with this interactive process. 
	System resources are released.
	Note that the window device is not removed from the IOSt because there still might be
	a modal dialog which final state has to be retrieved. 
-}
windowClose :: ps -> GUI ps ps
windowClose ps
	= do {	(found,wDevice) <- accIOEnv (ioStGetDevice WindowDevice);
		if   not found
		then return ps
		else 
		do {	osdinfo <- accIOEnv ioStGetOSDInfo;
			let	windows         = windowSystemStateGetWindowHandles wDevice
				(wsHs,windows1) = (whsWindows windows,windows {whsWindows=[]})
			in
			do {	appIOEnv (ioStSetDevice (WindowSystemState windows1));
				(freeIdss,ps1) <- ssequence [disposeWindowStateHandle osdinfo wsH | wsH<-wsHs] ps;
				idtable        <- ioStGetIdTable;
				ioStSetIdTable (snd (removeIdsFromIdTable (flatten freeIdss) idtable));
				return ps1
			}
		}
	  }


{-	windowIO handles the DeviceEvents that have been filtered by windowEvent.
	The only events that are handled in this implementation are:
		ControlKeyboardAction
		ControlSelection
		WindowActivation
		WindowDeactivation
		WindowInitialise
		WindowRequestClose
-}
windowIO :: DeviceEvent -> ps -> GUI ps ps
windowIO deviceEvent ps
	= do {
		hasDevice <- accIOEnv (ioStHasDevice WindowDevice);
		if   not hasDevice
		then windowdeviceFatalError "dDoIO windowFunctions" "could not retrieve WindowSystemState from IOSt"
		else windowIO' deviceEvent ps
	  }
	where
		windowIO' :: DeviceEvent -> ps -> GUI ps ps
		windowIO' (ControlKeyboardAction info) ps
			= do {
				(_,wDevice) <- accIOEnv (ioStGetDevice WindowDevice);
				let windows              = windowSystemStateGetWindowHandles wDevice
				    (found,wsH,windows1) = getWindowHandlesWindow (toWID (wId (ckWIDS info))) windows
				in
				if   not found
				then windowdeviceFatalError "windowIO (ControlKeyboardAction _) _" "window could not be found"
				else toGUI (windowStateControlKeyboardActionIO info wsH windows1 ps)
			  }
		
		windowIO' (ControlSelection info) ps
			= do {
				(_,wDevice) <- accIOEnv (ioStGetDevice WindowDevice);
				let windows              = windowSystemStateGetWindowHandles wDevice
				    (found,wsH,windows1) = getWindowHandlesWindow (toWID (wId (csWIDS info))) windows
				in
				if   not found
				then windowdeviceFatalError "windowIO (ControlSelection _) _" "window could not be found"
				else toGUI (windowStateControlSelectionIO info wsH windows1 ps)
			  }
		
		windowIO' (WindowActivation wids) ps
			= do {
				(_,wDevice) <- accIOEnv (ioStGetDevice WindowDevice);
				let windows              = windowSystemStateGetWindowHandles wDevice
				    wid                  = toWID (wId wids)
				    (found,wsH,windows1) = getWindowHandlesWindow wid windows
				in
				if   not found
				then windowdeviceFatalError "windowIO (WindowActivation _)" "window could not be found"
				else let (_,_,windows2)  = removeWindowHandlesWindow wid windows1	-- Remove the placeholder from windows1
				     in  toGUI (windowStateActivationIO wsH windows2 ps)
			  }
		
		windowIO' (WindowCreateControls wids) ps
			= do {
				(_,wDevice) <- accIOEnv (ioStGetDevice WindowDevice);
				let windows              = windowSystemStateGetWindowHandles wDevice
				    (found,wsH,windows1) = removeWindowHandlesWindow (toWID (0::OSWindowPtr)) windows
				in
				if   not found
				then windowdeviceFatalError "windowIO (WindowCreateControls _)" "window could not be found"
				else do {
					wMetrics        <- accIOEnv ioStGetOSWindowMetrics;
					(cPtr,wsH1)     <- liftIO (createDialogControls wMetrics wsH);
					delayInfo       <- liftIO (osActivateControl (wPtr wids) cPtr);
					bufferDelayedEvents delayInfo;
					let wsH2         = setWindowStateHandleWIDS wids wsH1
					    windows2     = addWindowHandlesActiveWindow wsH2 windows1
					in  appIOEnv (ioStSetDevice (WindowSystemState windows2)) >> return ps
				     }
			  }
			where
				createDialogControls :: OSWindowMetrics -> WindowStateHandle ps -> IO (OSWindowPtr,WindowStateHandle ps)
				createDialogControls wMetrics wsH@(WindowStateHandle wids (Just wlsH@(WindowLSHandle {wlsHandle=wH})))
					= do {
						itemHs <- createControls wMetrics {-(whDefaultId wH) (whCancelId wH)-}Nothing Nothing True (wPtr wids) (whItems wH);
						let (itemPtr,wH1) = getInitActiveControl (wH {whItems=itemHs})
						in  return (itemPtr,WindowStateHandle wids (Just (wlsH {wlsHandle=wH1})))
					  }
				createDialogControls _ _
					= windowdeviceFatalError "windowIO (WindowCreateControls _)" "placeholder not expected"
		
		windowIO' (WindowDeactivation wids) ps
			= do {
				(_,wDevice) <- accIOEnv (ioStGetDevice WindowDevice);
				let windows              = windowSystemStateGetWindowHandles wDevice
				    (found,wsH,windows1) = getWindowHandlesWindow (toWID (wId wids)) windows
				in
				if   not found
				then windowdeviceFatalError "windowIO (WindowDeactivation _)" "window could not be found"
				else toGUI (windowStateDeactivationIO wsH windows1 ps)
			  }
		
		windowIO' (WindowInitialise wids) ps
			= do {
				(_,wDevice) <- accIOEnv (ioStGetDevice WindowDevice);
				let windows              = windowSystemStateGetWindowHandles wDevice
				    (found,wsH,windows1) = getWindowHandlesWindow (toWID (wId wids)) windows
				in
				if   not found
				then windowdeviceFatalError "windowIO (WindowInitialise _)" "window could not be found"
				else toGUI (windowStateInitialiseIO wsH windows1 ps)
			  }
	
		windowIO' (WindowRequestClose wids) ps
			= do {
				(_,wDevice) <- accIOEnv (ioStGetDevice WindowDevice);
				let windows              = windowSystemStateGetWindowHandles wDevice
				    (found,wsH,windows1) = getWindowHandlesWindow (toWID (wId wids)) windows
				in
				if   not found
				then windowdeviceFatalError "windowIO (WindowRequestClose _)" "window could not be found"
				else toGUI (windowStateRequestCloseIO wsH windows1 ps)
			  }


{-	windowStateControlKeyboardActionIO handles the keyboard actions of (Edit/Custom/PopUp)Controls 
	and CompoundControls (not yet).
	In this implementation only EditControls are taken into account.
-}
windowStateControlKeyboardActionIO :: ControlKeyboardActionInfo -> WindowStateHandle ps -> WindowHandles ps -> ps -> IOSt ps -> IO (ps,IOSt ps)
windowStateControlKeyboardActionIO info wsH@(WindowStateHandle wids (Just (WindowLSHandle {wlsState=ls,wlsHandle=wH}))) windows ps ioState
	= do {
		r <- windowControlKeyboardActionIO (\wH st ioState -> ioStSetDevice
		                                                          (WindowSystemState
		                                                              (setWindowHandlesWindow
		                                                                  (WindowStateHandle wids (Just (WindowLSHandle {wlsState=fst st,wlsHandle=wH})))
		                                                                  windows))
		                                                          ioState)
		                                   wH (ls,ps) ioState;
		let ((_,ps1),ioState1) = r
		in  return (ps1,ioState1)
	  }
	where
		windowControlKeyboardActionIO :: (WindowHandle ls ps -> (ls,ps) -> IOSt ps -> IOSt ps)
		                               -> WindowHandle ls ps -> (ls,ps) -> IOSt ps -> IO ((ls,ps),IOSt ps)
		windowControlKeyboardActionIO build wH@(WindowHandle {whItems=whItems}) ls_ps ioState
			= do {
				r <- elementsControlKeyboardActionIO (\itemHs st ioState -> build (wH {whItems=itemHs}) st ioState) whItems ls_ps ioState;
				let (_,ls_ps1,ioState1) = r
				in  return (ls_ps1,ioState1)
			  }
			where
				elementsControlKeyboardActionIO :: ([WElementHandle ls ps] -> (ls,ps) -> IOSt ps -> IOSt ps)
				                                 -> [WElementHandle ls ps] -> (ls,ps) -> IOSt ps -> IO (Bool,(ls,ps),IOSt ps)
				elementsControlKeyboardActionIO build (itemH:itemHs) ls_ps ioState
					= do {
						r <- elementControlKeyboardActionIO (\itemH st ioState -> build (itemH:itemHs) st ioState) itemH ls_ps ioState;
						let (done,ls_ps1,ioState1) = r
						in  if   done
						    then return (done,ls_ps1,ioState1)
						    else elementsControlKeyboardActionIO (\itemHs st ioState -> build (itemH:itemHs) st ioState) itemHs ls_ps1 ioState1
					  }
					where
						elementControlKeyboardActionIO :: (WElementHandle ls ps -> (ls,ps) -> IOSt ps -> IOSt ps)
						                                -> WElementHandle ls ps -> (ls,ps) -> IOSt ps -> IO (Bool,(ls,ps),IOSt ps)
						
						elementControlKeyboardActionIO build (WListLSHandle itemHs) ls_ps ioState
							= elementsControlKeyboardActionIO (\itemHs st ioState -> build (WListLSHandle itemHs) st ioState) itemHs ls_ps ioState
						
						elementControlKeyboardActionIO build (WExtendLSHandle addLS itemHs) (ls,ps) ioState
							= do {
								r <- elementsControlKeyboardActionIO (\itemHs st ioState -> build (WExtendLSHandle (fst (fst st)) itemHs) (snd (fst st),snd st) ioState)
								                                     itemHs ((addLS,ls),ps) ioState;
								let (done,((_,ls1),ps1),ioState1) = r
								in  return (done,(ls1,ps1),ioState1)
							  }
						
						elementControlKeyboardActionIO build (WChangeLSHandle newLS itemHs) (ls,ps) ioState
							= do {
								r <- elementsControlKeyboardActionIO (\itemHs st ioState -> build (WChangeLSHandle (fst st) itemHs) (ls,snd st) ioState)
								                                     itemHs (newLS,ps) ioState;
								let (done,(_,ps1),ioState1) = r
								in  return (done,(ls,ps1),ioState1)
							  }
						
						elementControlKeyboardActionIO build itemH ls_ps ioState
							| ckItemNr info /= wItemNr itemH
								= return (False,ls_ps,build itemH ls_ps ioState)
							| otherwise
								= do {
									r <- itemControlKeyboardActionIO build itemH ls_ps ioState;
									let (ls_ps1,ioState1) = r
									in  return (True,ls_ps1,ioState1)
								  }
							where
								itemControlKeyboardActionIO :: (WElementHandle ls ps -> (ls,ps) -> IOSt ps -> IOSt ps)
								                             -> WElementHandle ls ps -> (ls,ps) -> IOSt ps -> IO ((ls,ps),IOSt ps)
								itemControlKeyboardActionIO build itemH ls_ps ioState
									= fixIO (\st -> fromGUI (f (ckKeyboardState info) ls_ps) (build itemH (fst st) ioState))
									where
										(_,_,f) = getControlKeyboardAtt (snd (cselect isControlKeyboard undef (wItemAtts itemH)))
				
				elementsControlKeyboardActionIO build nil ls_ps ioState
					= return (False,ls_ps,build nil ls_ps ioState)

windowStateControlKeyboardActionIO _ _ _ _ _
	= windowdeviceFatalError "windowStateControlKeyboardActionIO" "unexpected window placeholder"


{-	windowStateControlSelectionIO handles the selection of the control.
-}
windowStateControlSelectionIO :: ControlSelectInfo -> WindowStateHandle ps -> WindowHandles ps -> ps -> IOSt ps -> IO (ps,IOSt ps)
windowStateControlSelectionIO info (WindowStateHandle wids (Just (WindowLSHandle {wlsState=ls,wlsHandle=wH}))) windows ps ioState
	= do {
		r <- windowControlSelectionIO (\wH st ioState -> ioStSetDevice
		                                                     (WindowSystemState
		                                                         (setWindowHandlesWindow
		                                                             (WindowStateHandle wids (Just (WindowLSHandle {wlsState=fst st,wlsHandle=wH})))
		                                                             windows))
		                                                     ioState)
		                              wH (ls,ps) ioState;
		let ((_,ps1),ioState1) = r
		in  return (ps1,ioState1)
	  }
	where
		windowControlSelectionIO :: (WindowHandle ls ps -> (ls,ps) -> IOSt ps -> IOSt ps)
		                          -> WindowHandle ls ps -> (ls,ps) -> IOSt ps -> IO ((ls,ps),IOSt ps)
		windowControlSelectionIO build wH@(WindowHandle {whItems=whItems}) ls_ps ioState
			= do {
				r <- elementsControlSelectionIO (\itemHs st ioState -> build (wH {whItems=itemHs}) st ioState) whItems ls_ps ioState;
				let (_,ls_ps1,ioState1) = r
				in  return (ls_ps1,ioState1)
			  }
			where
				elementsControlSelectionIO :: ([WElementHandle ls ps] -> (ls,ps) -> IOSt ps -> IOSt ps)
				                            -> [WElementHandle ls ps] -> (ls,ps) -> IOSt ps -> IO (Bool,(ls,ps),IOSt ps)
				elementsControlSelectionIO build (itemH:itemHs) ls_ps ioState
					= do {
						r <- elementControlSelectionIO (\itemH st ioState -> build (itemH:itemHs) st ioState) itemH ls_ps ioState;
						let (done,ls_ps1,ioState1) = r
						in  if   done
						    then return (done,ls_ps1,ioState1)
						    else elementsControlSelectionIO (\itemHs st ioState -> build (itemH:itemHs) st ioState) itemHs ls_ps1 ioState1
					  }
					where
						elementControlSelectionIO :: (WElementHandle ls ps -> (ls,ps) -> IOSt ps -> IOSt ps)
						                           -> WElementHandle ls ps -> (ls,ps) -> IOSt ps -> IO (Bool,(ls,ps),IOSt ps)
						
						elementControlSelectionIO build (WListLSHandle itemHs) ls_ps ioState
							= elementsControlSelectionIO (\itemHs st ioState -> build (WListLSHandle itemHs) st ioState) itemHs ls_ps ioState
						
						elementControlSelectionIO build (WExtendLSHandle addLS itemHs) (ls,ps) ioState
							= do {
								r <- elementsControlSelectionIO (\itemHs st ioState -> build (WExtendLSHandle (fst (fst st)) itemHs) (snd (fst st),snd st) ioState)
								                                itemHs ((addLS,ls),ps) ioState;
								let (done,((_,ls1),ps1),ioState1) = r
								in  return (done,(ls1,ps1),ioState1)
							  }
						
						elementControlSelectionIO build (WChangeLSHandle newLS itemHs) (ls,ps) ioState
							= do {
								r <- elementsControlSelectionIO (\itemHs st ioState -> build (WChangeLSHandle (fst st) itemHs) (ls,snd st) ioState)
								                                itemHs (newLS,ps) ioState;
								let (done,(_,ps1),ioState1) = r
								in  return (done,(ls,ps1),ioState1)
							  }
						
						elementControlSelectionIO build itemH ls_ps ioState
							| csItemNr info /= wItemNr itemH
								= return (False,ls_ps,build itemH ls_ps ioState)
							| otherwise
								= do {
									r <- itemControlSelectionIO build itemH ls_ps ioState;
									let (ls_ps1,ioState1) = r
									in  return (True,ls_ps1,ioState1)
								  }
							where
								itemControlSelectionIO :: (WElementHandle ls ps -> (ls,ps) -> IOSt ps -> IOSt ps)
								                        -> WElementHandle ls ps -> (ls,ps) -> IOSt ps -> IO ((ls,ps),IOSt ps)
								itemControlSelectionIO build itemH@(WItemHandle {wItemKind=wItemKind}) ls_ps ioState
									| wItemKind==IsButtonControl
										= if   hasAtt
										  then fixIO (\st -> fromGUI (f ls_ps) (build itemH (fst st) ioState))
										  else return (ls_ps,build itemH ls_ps ioState)
									| otherwise
										= return (ls_ps,build itemH ls_ps ioState)
									where
										atts          = wItemAtts itemH
										(hasAtt,fAtt) = cselect (\att->isControlFunction att || isControlModsFunction att) undef atts
										f             = case fAtt of
										                    (ControlFunction     f) -> f
										                    (ControlModsFunction f) -> f (csModifiers info)
										                    wrongAttribute          -> windowdeviceFatalError "windowStateControlSelectionIO" "argument is not a function attribute"
					
				elementsControlSelectionIO build nil ls_ps ioState
					= return (False,ls_ps,build nil ls_ps ioState)

windowStateControlSelectionIO _ _ _ _ _
	= windowdeviceFatalError "windowStateControlSelectionIO" "unexpected window placeholder"


{-	windowStateActivationIO handles the activation of the window/dialog.
-}
windowStateActivationIO :: WindowStateHandle ps -> WindowHandles ps -> ps -> IOSt ps -> IO (ps,IOSt ps)
windowStateActivationIO (WindowStateHandle wids (Just wlsH@(WindowLSHandle {wlsState=ls,wlsHandle=wH}))) windows ps ioState
	= do {
		r <- fixIO (\st -> fromGUI (f (ls,ps))
		                           (ioStSetDevice 
		                               (WindowSystemState
		                                   (addWindowHandlesActiveWindow
		                                       (WindowStateHandle (wids {wActive=True})
		                                                          (Just (wlsH {wlsState=fst (fst st)})))
		                                       windows))
		                               ioState));
		let ((_,ps1),ioState1) = r
		in  return (ps1,ioState1)
	  }
	where
		(hasAtt,att) = cselect isWindowActivate undef (whAtts wH)
		f            = if hasAtt then getWindowActivateFun att else return
windowStateActivationIO _ _ _ _
	= windowdeviceFatalError "windowStateActivationIO" "unexpected window placeholder"


{-	windowStateDeactivationIO handles the deactivation of the window/dialog.
-}
windowStateDeactivationIO :: WindowStateHandle ps -> WindowHandles ps -> ps -> IOSt ps -> IO (ps,IOSt ps)
windowStateDeactivationIO (WindowStateHandle wids (Just wlsH@(WindowLSHandle {wlsState=ls,wlsHandle=wH}))) windows ps ioState
	= do {
		r <- fixIO (\st -> fromGUI (f (ls,ps))
		                           (ioStSetDevice 
		                               (WindowSystemState
		                                  (setWindowHandlesWindow
		                                       (WindowStateHandle (wids {wActive=True})
		                                                          (Just (wlsH {wlsState=fst (fst st)})))
		                                       windows))
		                               ioState));
		let ((_,ps1),ioState1) = r
		in  return (ps1,ioState1)
	  }
	where
		(hasAtt,att) = cselect isWindowDeactivate undef (whAtts wH)
		f            = if hasAtt then getWindowDeactivateFun att else return
windowStateDeactivationIO _ _ _ _
	= windowdeviceFatalError "windowStateDeactivationIO" "unexpected window placeholder"


{-	windowStateInitialiseIO handles the initialisation of the window/dialog.
-}
windowStateInitialiseIO :: WindowStateHandle ps -> WindowHandles ps -> ps -> IOSt ps -> IO (ps,IOSt ps)
windowStateInitialiseIO (WindowStateHandle wids (Just wlsH@(WindowLSHandle {wlsState=ls,wlsHandle=wH}))) windows ps ioState
	= do {
		r <- fixIO (\st -> fromGUI (f (ls,ps))
		                           (ioStSetDevice
		                               (WindowSystemState 
		                                   (setWindowHandlesWindow 
		                                       (WindowStateHandle wids (Just (wlsH {wlsState=fst (fst st),wlsHandle=wH1})))
		                                       windows))
		                               ioState));
		let ((_,ps1),ioState1) = r
		in  return (ps1,ioState1)
	  }
	where
		(hasAtt,initAtt,atts) = remove isWindowInit undef (whAtts wH)
		f                     = if hasAtt then getWindowInitFun initAtt else return
		wH1                   = wH {whAtts=atts}
windowStateInitialiseIO _ _ _ _
	= windowdeviceFatalError "windowStateInitialiseIO" "unexpected window placeholder"


{-	windowStateRequestCloseIO handles the request to close the window/dialog.
-}
windowStateRequestCloseIO :: WindowStateHandle ps -> WindowHandles ps -> ps -> IOSt ps -> IO (ps,IOSt ps)
windowStateRequestCloseIO (WindowStateHandle wids (Just wlsH@(WindowLSHandle {wlsState=ls,wlsHandle=wH}))) windows ps ioState
	= do {
		r <- fixIO (\st -> fromGUI (f (ls,ps))
		                           (ioStSetDevice
		                               (WindowSystemState
		                                   (setWindowHandlesWindow
		                                       (WindowStateHandle wids (Just (wlsH {wlsState=fst (fst st)})))
		                                       windows))
		                               ioState));
		let ((_,ps1),ioState1) = r
		in  return (ps1,ioState1)
	  }
	where
		(hasAtt,closeAtt) = cselect isWindowClose undef (whAtts wH)
		f                 = if hasAtt then getWindowCloseFun closeAtt else return
windowStateRequestCloseIO _ _ _ _
	= windowdeviceFatalError "windowStateRequestCloseIO" "unexpected window placeholder"


--	Auxiliary function (move to Commondef??):
selectedAtIndex :: Cond x -> x -> [x] -> (Index, x)		-- if index==0 then not found; item was found at index
selectedAtIndex cond dummy xs
	= (if found then i else 0,x)
	where
		(found,i,x) = selected cond dummy xs 1
		
		selected :: Cond x -> x -> [x] -> Int -> (Bool,Int,x)
		selected cond dummy (x:xs) i
			| cond x    = (True,i,x)
			| otherwise = selected cond dummy xs (i+1)
		selected _ dummy _ i
			= (False,i,dummy)
