module WindowdeviceMVAR ( windowFunctions ) where


--	********************************************************************************
--	Clean to Haskell Standard Object I/O library, version 1.2
--	
--	WindowdeviceMVAR defines the window device event handlers for the MVAR version 
--	only.
--	********************************************************************************


import CleanStdList
import CleanStdMisc
import CleanStdTuple
import Commondef
import Controlcreate
import Deviceevents
import Id
import IOstate
import Oswindow
import StdControlAttribute
import StdWindowAttribute
import Windowaccess
import Windowcreate (bufferDelayedEvents)
import Windowdispose
import Windowevent
import Windowhandle


windowdeviceFatalError :: String -> String -> x
windowdeviceFatalError function error
	= dumpFatalError function "Windowdevice" error


windowFunctions :: DeviceFunctions
windowFunctions
	= DeviceFunctions
		{ dDevice = WindowDevice
		, dEvent  = windowEvent
		, dDoIO   = windowIO
		, dOpen   = windowOpen
		, dClose  = windowClose
		}

{-	windowOpen initialises the window device for this interactive process.
-}
windowOpen :: GUI ()
windowOpen
	= do {
		hasWindow <- accIOEnv (ioStHasDevice WindowDevice);
		if   hasWindow
		then return ()
		else do {
			xDI <- accIOEnv ioStGetDocumentInterface;
			let bound   = case xDI of
			                NDI -> Finite 0
			                SDI -> Finite 1
			                MDI -> Infinite
			    windows = WindowHandles {whsWindows=[],whsNrWindowBound=bound}
			in  appIOEnv ( ioStSetDevice (WindowSystemState windows) . ioStSetDeviceFunctions windowFunctions )
		     }
	  }


{-	windowClose closes all windows associated with this interactive process. 
	System resources are released.
	Note that the window device is not removed from the IOSt because there still might be
	a modal dialog which final state has to be retrieved. 
-}
windowClose :: GUI ()
windowClose
	= do {	(found,wDevice) <- accIOEnv (ioStGetDevice WindowDevice);
		if   not found
		then return ()
		else 
		do {	osdinfo <- accIOEnv ioStGetOSDInfo;
			let	windows         = windowSystemStateGetWindowHandles wDevice
				(wsHs,windows1) = (whsWindows windows,windows {whsWindows=[]})
			in
			do {	appIOEnv (ioStSetDevice (WindowSystemState windows1));
				freeIdss <- sequence [disposeWindowStateHandle osdinfo wsH | wsH<-wsHs];
				idtable  <- ioStGetIdTable;
				ioStSetIdTable (snd (removeIdsFromIdTable (flatten freeIdss) idtable))
			}
		}
	  }


{-	windowIO handles the DeviceEvents that have been filtered by windowEvent.
	The only events that are handled in this implementation are:
		ControlKeyboardAction
		ControlSelection
		WindowRequestClose
		WindowInitialise
		ControlKeyboardAction
-}
windowIO :: DeviceEvent -> GUI ()
windowIO deviceEvent
	= do {
		hasDevice <- accIOEnv (ioStHasDevice WindowDevice);
		if   not hasDevice
		then windowdeviceFatalError "dDoIO windowFunctions" "could not retrieve WindowSystemState from IOSt"
		else windowIO' deviceEvent
	  }
	where
		windowIO' :: DeviceEvent -> GUI ()
		windowIO' (ControlKeyboardAction info)
			= do {
				(_,wDevice) <- accIOEnv (ioStGetDevice WindowDevice);
				let windows              = windowSystemStateGetWindowHandles wDevice
				    (found,wsH,windows1) = getWindowHandlesWindow (toWID (wId (ckWIDS info))) windows
				in
				if   not found
				then windowdeviceFatalError "windowIO (ControlKeyboardAction _) _" "window could not be found"
				else windowStateControlKeyboardActionIO info wsH windows1
			  }

		windowIO' (ControlSelection info)
			= do {
				(_,wDevice) <- accIOEnv (ioStGetDevice WindowDevice);
				let windows              = windowSystemStateGetWindowHandles wDevice
				    (found,wsH,windows1) = getWindowHandlesWindow (toWID (wId (csWIDS info))) windows
				in
				if   not found
				then windowdeviceFatalError "windowIO (ControlSelection _) _" "window could not be found"
				else windowStateControlSelectionIO info wsH windows1
			  }
		
		windowIO' (WindowActivation wids)
			= do {
				(_,wDevice) <- accIOEnv (ioStGetDevice WindowDevice);
				let windows              = windowSystemStateGetWindowHandles wDevice
				    wid                  = toWID (wId wids)
				    (found,wsH,windows1) = getWindowHandlesWindow wid windows
				in
				if   not found
				then windowdeviceFatalError "windowIO (WindowActivation _)" "window could not be found"
				else let (_,_,windows2)  = removeWindowHandlesWindow wid windows1	-- Remove the placeholder from windows1
				     in  windowStateActivationIO wsH windows2
			  }
		
		windowIO' (WindowCreateControls wids)
			= do {
				(_,wDevice) <- accIOEnv (ioStGetDevice WindowDevice);
				let windows              = windowSystemStateGetWindowHandles wDevice
				    (found,wsH,windows1) = removeWindowHandlesWindow (toWID (0::OSWindowPtr)) windows
				in
				if   not found
				then windowdeviceFatalError "windowIO (WindowCreateControls _)" "window could not be found"
				else do {
					wMetrics        <- accIOEnv ioStGetOSWindowMetrics;
					(cPtr,wsH1)     <- liftIO (createDialogControls wMetrics wsH);
					delayInfo       <- liftIO (osActivateControl (wPtr wids) cPtr);
					bufferDelayedEvents delayInfo;
					let wsH2         = setWindowStateHandleWIDS wids wsH1
					    windows2     = addWindowHandlesActiveWindow wsH2 windows1
					in  appIOEnv (ioStSetDevice (WindowSystemState windows2))
				     }
			  }
			where
				createDialogControls :: OSWindowMetrics -> WindowStateHandle -> IO (OSWindowPtr,WindowStateHandle)
				createDialogControls wMetrics wsH@(WindowStateHandle {wshHandle=Just wH})
					= do {
						itemHs <- createControls wMetrics {-(whDefaultId wH) (whCancelId wH)-}Nothing Nothing True (wPtr wids) (whItems wH);
						let (itemPtr,wH1) = getInitActiveControl (wH {whItems=itemHs})
						in  return (itemPtr,wsH {wshHandle=Just wH1})
					  }
				createDialogControls _ _
					= windowdeviceFatalError "windowIO (WindowCreateControls _)" "placeholder not expected"
		
		windowIO' (WindowDeactivation wids)
			= do {
				(_,wDevice) <- accIOEnv (ioStGetDevice WindowDevice);
				let windows              = windowSystemStateGetWindowHandles wDevice
				    (found,wsH,windows1) = getWindowHandlesWindow (toWID (wId wids)) windows
				in
				if   not found
				then windowdeviceFatalError "windowIO (WindowDeactivation _)" "window could not be found"
				else windowStateDeactivationIO wsH windows1
			  }
		
		windowIO' (WindowInitialise wids)
			= do {
				(_,wDevice) <- accIOEnv (ioStGetDevice WindowDevice);
				let windows              = windowSystemStateGetWindowHandles wDevice
				    (found,wsH,windows1) = getWindowHandlesWindow (toWID (wId wids)) windows
				in
				if   not found
				then windowdeviceFatalError "windowIO (WindowInitialise _)" "window could not be found"
				else windowStateInitialiseIO wsH windows1
			  }
	
		windowIO' (WindowRequestClose wids)
			= do {
				(_,wDevice) <- accIOEnv (ioStGetDevice WindowDevice);
				let windows              = windowSystemStateGetWindowHandles wDevice
				    (found,wsH,windows1) = getWindowHandlesWindow (toWID (wId wids)) windows
				in
				if   not found
				then windowdeviceFatalError "windowIO (WindowRequestClose _)" "window could not be found"
				else windowStateRequestCloseIO wsH windows1
			  }


{-	windowStateControlKeyboardActionIO handles the keyboard actions of (Edit/Custom/PopUp)Controls 
	and CompoundControls (not yet).
	In this implementation only EditControls are taken into account.
-}
windowStateControlKeyboardActionIO :: ControlKeyboardActionInfo -> WindowStateHandle -> WindowHandles -> GUI ()
windowStateControlKeyboardActionIO info wsH windows
	| isNothing maybeGUI
		= windowdeviceFatalError "windowStateControlKeyboardActionIO" "could not locate selected control"
	| otherwise
		= let windows1 = setWindowHandlesWindow wsH1 windows
		  in  appIOEnv (ioStSetDevice (WindowSystemState windows1)) >> fromJust maybeGUI
	where
		(maybeGUI,wsH1) = accessWItemHandle "windowStateControlKeyboardActionIO"
		                                    (\itemH->wItemNr itemH==ckItemNr info)
		                                    (getGUI info) wsH
		
		getGUI :: ControlKeyboardActionInfo -> WElementHandle -> (GUI (),WElementHandle)
		getGUI info itemH@(WItemHandle {wItemAtts=atts})
			= (thd3 (getControlKeyboardAtt (snd (cselect isControlKeyboard undef atts))) (ckKeyboardState info),itemH)


{-	windowStateControlSelectionIO handles the selection of the control.
-}
windowStateControlSelectionIO :: ControlSelectInfo -> WindowStateHandle -> WindowHandles -> GUI ()
windowStateControlSelectionIO info wsH windows
	| isNothing maybeGUI
		= windowdeviceFatalError "windowStateControlSelectionIO" "could not locate selected control"
	| otherwise
		= let windows1 = setWindowHandlesWindow wsH1 windows
		  in  appIOEnv (ioStSetDevice (WindowSystemState windows1)) >> fromJust maybeGUI
	where
		(maybeGUI,wsH1) = accessWItemHandle "windowStateControlSelectionIO"
		                                    (\itemH->wItemNr itemH==csItemNr info)
		                                    (getGUI info) wsH
		
		getGUI :: ControlSelectInfo -> WElementHandle -> (GUI (),WElementHandle)
		getGUI info itemH@(WItemHandle {wItemKind=IsButtonControl,wItemAtts=atts})
			| hasAtt
				= (f,itemH)
			| otherwise
				= (return (),itemH)
			where
				(hasAtt,fAtt) = cselect (\att->isControlFunction att || isControlModsFunction att) undef atts
				f             = case fAtt of
				                   ControlFunction     fun -> fun
				                   ControlModsFunction fun -> fun (csModifiers info)
				                   wrongAttribute          -> windowdeviceFatalError "windowStateControlSelectionIO" "argument is not a function attribute"


{-	windowStateActivationIO handles the activation of the window/dialog.
-}
windowStateActivationIO :: WindowStateHandle -> WindowHandles -> GUI ()
windowStateActivationIO wsH@(WindowStateHandle {wshIds=wids,wshHandle=Just wH}) windows
	= appIOEnv (ioStSetDevice (WindowSystemState windows1)) >> f
	where
		(hasAtt,att) = cselect isWindowActivate undef (whAtts wH)
		f            = if hasAtt then getWindowActivateFun att else return ()
		wsH1         = wsH {wshIds=wids {wActive=True}}
	--	windows1     = setWindowHandlesWindow wsH1 windows
		windows1     = addWindowHandlesActiveWindow wsH1 windows
windowStateActivationIO _ _
	= windowdeviceFatalError "windowStateActivationIO" "unexpected window placeholder"


{-	windowStateDeactivationIO handles the deactivation of the window/dialog.
-}
windowStateDeactivationIO :: WindowStateHandle -> WindowHandles -> GUI ()
windowStateDeactivationIO wsH@(WindowStateHandle {wshIds=wids,wshHandle=Just wH}) windows
	= appIOEnv (ioStSetDevice (WindowSystemState windows1)) >> f
	where
		(hasAtt,att) = cselect isWindowDeactivate undef (whAtts wH)
		f            = if hasAtt then getWindowDeactivateFun att else return ()
		wsH1         = wsH {wshIds=wids {wActive=False}}
		windows1     = setWindowHandlesWindow wsH1 windows
windowStateDeactivationIO _ _
	= windowdeviceFatalError "windowStateDeactivationIO" "unexpected window placeholder"


{-	windowStateInitialiseIO handles the initialisation of the window/dialog.
-}
windowStateInitialiseIO :: WindowStateHandle -> WindowHandles -> GUI ()
windowStateInitialiseIO wsH@(WindowStateHandle {wshHandle=Just wH}) windows
	= appIOEnv (ioStSetDevice (WindowSystemState windows1)) >> f
	where
		(hasAtt,initAtt,atts) = remove isWindowInit undef (whAtts wH)
		f                     = if hasAtt then getWindowInitFun initAtt else return ()
		wH1                   = wH {whAtts=atts}
		wsH1                  = wsH {wshHandle=Just wH1}
		windows1              = setWindowHandlesWindow wsH1 windows
windowStateInitialiseIO _ _
	= windowdeviceFatalError "windowStateInitialiseIO" "unexpected window placeholder"


{-	windowStateRequestCloseIO handles the request to close the window/dialog.
-}
windowStateRequestCloseIO :: WindowStateHandle -> WindowHandles -> GUI ()
windowStateRequestCloseIO wsH@(WindowStateHandle {wshHandle=Just wH}) windows
	= appIOEnv (ioStSetDevice (WindowSystemState windows1)) >> f
	where
		(hasAtt,closeAtt) = cselect isWindowClose undef (whAtts wH)
		f                 = if hasAtt then getWindowCloseFun closeAtt else return ()
		windows1          = setWindowHandlesWindow wsH windows
windowStateRequestCloseIO _ _
	= windowdeviceFatalError "windowStateRequestCloseIO" "unexpected window placeholder"
