module Windowdispose (disposeWindow, disposeWindowStateHandle) where


--	********************************************************************************
--	Clean to Haskell Standard Object I/O library, version 1.2
--	
--	Windowdispose disposes all system resources associated with the indicated window
--	if it exists.
--	********************************************************************************


import Commondef
import Controlvalidate
import Id
import IOstate
import Ostoolbox
import Oswindow
import Scheduler
import Windowaccess
import Windowhandle
#include "preprocess.h"
         -- Compile with -cpp; set MVAR=1 for MVAR version


windowdisposeFatalError :: String -> String -> x
windowdisposeFatalError function error
	= dumpFatalError function "Windowdispose" error


{-	disposeWindow disposes all system resources associated with the indicated window if it exists.
	Inactive modal dialogues are not removed.
	If the window belongs to an SDI process, then only the SDI client is removed, not the SDI frame.
	It removes the indicated window from the window device administration.
	Because the window may contain controls that are 'logically' disposed, but not 'physically' 
	disposeWindow also applies the init function contained in the IOSt.
-}
disposeWindow :: WID -> IF_MVAR(,ps ->) GUI IF_MVAR((),ps ps)
disposeWindow wid IF_MVAR(,ps)
	= do {
		(found,wDevice) <- accIOEnv (ioStGetDevice WindowDevice);
		if   not found
		then return IF_MVAR((),ps)
		else let
			windows               = windowSystemStateGetWindowHandles wDevice
			(found,wsH,windows1)  = getWindowHandlesWindow wid windows
		in 
		if   not found                -- The window could not be found
		then appIOEnv (ioStSetDevice (WindowSystemState windows1)) >> return IF_MVAR((),ps)
		else let
			(alreadyClosing,wsH1) = getWindowStateHandleClosing wsH
		in
		if   alreadyClosing           -- The window is already in the act of being closed
		then appIOEnv (ioStSetDevice (WindowSystemState (setWindowHandlesWindow wsH1 windows1))) >> return IF_MVAR((),ps)
		else let                      -- Any modeless window can be disposed
			(wKind,wsH2)          = getWindowStateHandleWindowKind wsH1
			(wids, wsH3)          = getWindowStateHandleWIDS wsH2
		in do {
		        xDI <- accIOEnv ioStGetDocumentInterface;
			-- Of a SDI process, the SDI client should be closed, not the SDI frame (which is closed by closeProcess)
		        if   xDI==SDI && wKind==IsWindow
		        then dispose wids wsH3 (incWindowBound windows1) IF_MVAR(,ps)
		        else dispose wids wsH3 windows1 IF_MVAR(,ps)
		   }
	  }
	where
		incWindowBound :: WindowHandles IF_MVAR(,ps) -> WindowHandles IF_MVAR(,ps)
		incWindowBound wHs@(WindowHandles {whsNrWindowBound=whsNrWindowBound})
			= wHs {whsNrWindowBound=incBound whsNrWindowBound}

		dispose :: WIDS -> WindowStateHandle IF_MVAR(,ps) -> WindowHandles IF_MVAR(,ps -> ps) -> GUI IF_MVAR((),ps ps)
		dispose wids@(WIDS {wId=wId}) wsH windows IF_MVAR(,ps)
			= do {
				appIOEnv (ioStSetDevice (WindowSystemState windows1));
				disposeFun   <- accIOEnv ioStGetInitIO;
				IF_MVAR(,ps1 <-) disposeFun IF_MVAR(,ps);
				osdinfo      <- accIOEnv ioStGetOSDInfo;
#if MVAR
				ids          <- disposeWindowStateHandle osdinfo wsH;
#else
				(ids,ps2)    <- disposeWindowStateHandle osdinfo wsH ps1;
#endif
				idtable      <- ioStGetIdTable;
				ioStSetIdTable (snd (removeIdsFromIdTable ids idtable));
				return IF_MVAR((),ps2)
			  }
			where
				(_,_,windows1)    = removeWindowHandlesWindow (toWID wId) windows	-- Remove window placeholder


{-	disposeWindowStateHandle disposes all system resources associated with the given WindowStateHandle.
	The return [Id] are the Ids of the other controls.
	When timers are part of windows, also timer ids should be returned.
-}
#if MVAR
disposeWindowStateHandle :: OSDInfo -> WindowStateHandle -> GUI [Id]
disposeWindowStateHandle osdinfo wsH@(WindowStateHandle {wshIds=wids,wshHandle=Just wH})
#else
disposeWindowStateHandle :: OSDInfo -> WindowStateHandle ps -> ps -> GUI ps ([Id],ps)
disposeWindowStateHandle osdinfo (WindowStateHandle wids (Just wlsH@(WindowLSHandle {wlsHandle=wH}))) ps
#endif
	= do {
		(ids,fs) <- liftIO (disposeWElementHandles (wPtr wids) (whItems wH));
		liftIO fs;
	--	ioState  <- accIOEnv (toSt id);
		context  <- accIOEnv ioStGetContext;
#if MVAR
	--	(delayInfo,ioState1) <- liftIO (osDestroyWindow osdinfo False (whKind wH==IsWindow) (wPtr wids) handleContextOSEvent ioState);
		(delayInfo,_) <- liftIO (osDestroyWindow osdinfo False (whKind wH==IsWindow) (wPtr wids) handleContextOSEvent context);
	--	appIOEnv (const ioState1);
		return ((wId wids) : ids)
#else
	--	(delayInfo,pState1)  <- liftIO (osDestroyWindow osdinfo False (whKind wH==IsWindow) (wPtr wids) handleContextOSEvent (PSt {ls=ps,io=ioState}));
		(delayInfo,_)  <- liftIO (osDestroyWindow osdinfo False (whKind wH==IsWindow) (wPtr wids) handleContextOSEvent context);
	--	appIOEnv (const (io pState1));
	--	return ((wId wids):ids, ls pState1)
		return ((wId wids):ids, ps)
#endif
	  }
disposeWindowStateHandle _ _ IF_MVAR(,_)
	= windowdisposeFatalError "disposeWindowStateHandle" "window expected instead of placeholder"

{-	disposeWElementHandle(s) (recursively) hides all system resources associated with the given 
	WElementHandle(s). The argument OSWindowPtr must be the parent window.
	The (IO ()) action must be used to actually dispose the controls.
	It returns all freed receiver and control ids.
	When timers are part of windows, also timer ids should be returned.
-}
disposeWElementHandles :: OSWindowPtr -> [WElementHandle IF_MVAR(,ls ps)] -> IO ([Id],IO ())
disposeWElementHandles wptr (itemH:itemHs)
	= do {
		(ids, fs)  <- disposeWElementHandle  wptr itemH;
		(idss,fss) <- disposeWElementHandles wptr itemHs;
		return (ids++idss,fs>>fss)
	  }
disposeWElementHandles _ []
	= return ([],return ())

{-	disposeWElementHandle (recursively) hides all system resources associated with the given WItemHandle. 
	The OSWindowPtr argument must identify the parent window.
	The (IO ()) function must be used to actually dispose the controls.
	It returns all freed ids.
	When timers are part of windows, also timer ids should be returned.
-}
disposeWElementHandle :: OSWindowPtr -> WElementHandle IF_MVAR(,ls ps) -> IO ([Id],IO ())
#if MVAR
#else
disposeWElementHandle wptr (WExtendLSHandle _ itemHs)
	= disposeWElementHandles wptr itemHs
disposeWElementHandle wptr (WChangeLSHandle _ itemHs)
	= disposeWElementHandles wptr itemHs
#endif
disposeWElementHandle wptr (WListLSHandle itemHs)
	= disposeWElementHandles wptr itemHs
disposeWElementHandle wptr (WItemHandle {wItemKind=wItemKind,wItemId=wItemId,wItemPtr=wItemPtr,wItemPos=wItemPos,wItemSize=wItemSize})
	= hide wptr wItemPtr (posSizeToRect wItemPos wItemSize) False >> return (maybeToList wItemId,dispose wItemPtr)
	where
		(hide,dispose)	= case wItemKind of
					IsTextControl   -> (osSetTextControlShow,   osDestroyTextControl)
					IsEditControl   -> (osSetEditControlShow,   osDestroyEditControl)
					IsButtonControl -> (osSetButtonControlShow, osDestroyButtonControl)
					_               -> windowdisposeFatalError "disposeWItemHandle" ("unmatched ControlKind: "++show wItemKind)

maybeToList :: Maybe x -> [x]
maybeToList (Just x) = [x]
maybeToList _        = []
