module Windowvalidate ( validateWindowId
                      , validateWindow
                      , exactWindowSize
                      , exactWindowPos
                      , validateViewDomain
                      ) where


--	********************************************************************************
--	Clean to Haskell Standard Object I/O library, version 1.2
--	
--	Windowvalidate contains window validation functions.
--	********************************************************************************


import Prelude hiding (Either(..))	-- Either = Left | Right must be hidden
import CleanStdMisc
import Commondef
import Controllayout
import Id
import IOstate
import Ossystem
import Oswindow
import StdId
import StdWindowAttribute
import Windowaccess
#include "preprocess.h"
         -- Compile with -cpp; set MVAR=1 for MVAR version


windowvalidateError :: String -> String -> x
windowvalidateError function message
	= dumpError function "windowvalidate" message

windowvalidateFatalError :: String -> String -> x
windowvalidateFatalError function message
	= dumpFatalError function "windowvalidate" message


{-	validateWindowId checks whether the Id of the window/dialogue has already been bound.
	If so, Nothing is returned; otherwise a proper Id value for the window/dialogue is returned.
	The Id is not bound.
-}
validateWindowId :: Maybe Id -> GUI IF_MVAR(,ps) (Maybe Id)
validateWindowId Nothing
	= do {
		wId <- openId;
		return (Just wId)
	  }
validateWindowId (Just id)
	= do {
		idtable <- ioStGetIdTable;
		if   memberIdTable id idtable
		then return Nothing
		else return (Just id)
	  }


{-	Validate the given window.
-}
validateWindow :: OSWindowMetrics -> OSDInfo -> WindowHandle IF_MVAR(,ls ps) -> WindowHandles IF_MVAR(,ps)
               -> IO (Index,Point2,Size,Vector2,WindowHandle IF_MVAR(,ls ps),   WindowHandles IF_MVAR(,ps))

validateWindow wMetrics _ wH@(WindowHandle {{-whMode=mode,-}whKind=IsDialog,whItemNrs=whItemNrs,whItems=whItems,whAtts=atts}) windows
	= do {
		(derSize,items) <- layoutControls wMetrics hMargins vMargins spaces reqSize minSize [(domain,zero)] whItems;
		let (itemNrs,   items1) = genWElementItemNrs whItemNrs items
		--  (focusItems,items2) = getWElementKeyFocusIds True items1
		    derSize1            = determineRequestedSize derSize sizeAtt
		    domain1             = sizeToRectangle derSize1
		    okSize              = exactWindowSize wMetrics domain1 derSize1 False False IsDialog
		in do {
			(okPos,windows3) <- exactWindowPos wMetrics okSize pos IsDialog {-mode-} windows2;
			return ( index
			       , okPos
			       , okSize
			       , zero
			       , wH { whItemNrs   = itemNrs
			--          , whKeyFocus  = newFocusItems focusItems
			            , whItems     = items1 -- should be items2 as soon as getWElementKeyFocusIds is incorporated
			--          , whSelect    = True
			            , whAtts      = atts3  -- should be atts4 as soon as validateWindowInitActive is incorporated
			--          , whDefaultId = defid
			--          , whCancelId  = canid
			            , whSize      = okSize
			            }
			       , windows3
			       )
		   }
	  }
	where
		atts1                 = filter isValidDialogAttribute         atts
		(index,atts2,windows1)= validateWindowIndex {-mode-}          atts1 windows
		(pos,  atts3,windows2)= validateWindowPos   {-mode-}          atts2 windows1
		sizeAtt               = attrSize                              atts3	-- Retrieve Window(View/Outer)Size (identical for Dialogs)
		(hMargins,vMargins)   = attrMargins         IsDialog wMetrics atts3
		spaces                = getWindowItemSpaces IsDialog wMetrics atts3
	--	(defid,whItems)       = getOkId                               atts3 whItems
	--	(canid,whItems)       = getCancelId                           atts3 whItems
	--	(atts4,whItems)       = validateWindowInitActive              atts3 whItems
		reqSize               = determineRequestedSize zero sizeAtt
		(minWidth,minHeight)  = osMinWindowSize
		minSize               = Size {w=minWidth,h=minHeight}
		domain                = sizeToRectangle reqSize
{-	Windows are not handled in this implementation. I've already transformed to Haskell syntax.
validateWindow wMetrics osdInfo wH@(WindowHandle {whKind=IsWindow,whItemNrs=whItemNrs,whItems=whItems,whAtts=atts}) windows
	= let
		atts1                  = filter isValidWindowAttribute atts
	--	mode                   = Modeless
		(domain,atts2)         = validateWindowDomain          atts1
		(maybe_hScroll,atts3)  = validateWindowHScroll         atts2
		(maybe_vScroll,atts4)  = validateWindowVScroll         atts3
		(sysLook,look, atts5)  = validateWindowLook            atts4
	  in
	  do {
		(reqSize,atts4)       <- validateWindowSize wMetrics domain isMDI True (isJust maybe_hScroll,isJust maybe_vScroll) atts5;
		let
			(index,atts5,windows1)   = validateWindowIndex {-mode-}  atts4 windows
			(pos,  atts6,windows2)   = validateWindowPos   {-mode-}  atts5 windows1
			(penAtts,atts7)          = attrPen                       atts6
			(hMargins,vMargins)      = attrMargins IsWindow wMetrics atts7
			spaces                   = getWindowItemSpaces IsWindow wMetrics atts7
			isAble                   = attrSelectState                       atts7
			(defid,whItems1)         = getOkId                               atts7 whItems
			(canid,whItems2)         = getCancelId                           atts7 whItems1
			(atts8,whItems3)         = validateWindowInitActive              atts7 whItems2
			pen                      = stateMap2 setPenAttribute (reverse penAtts) defaultPen
		in
		do {
			(derSize,items)         <- layoutControls wMetrics hMargins vMargins spaces reqSize minSize [(domain,domain.corner1)] whItems3;
			let
				(itemNrs,items1)    = genWElementItemNrs whItemNrs items
			--	(focusItems,items2) = getWElementKeyFocusIds True items1
				(origin,atts9)      = validateOrigin derSize domain atts8
				okSize              = exactWindowSize wMetrics domain derSize (isJust maybe_hScroll) (isJust maybe_vScroll) IsWindow
			in
			do {
				(okPos,windows)    <- exactWindowPos wMetrics okSize pos IsWindow {-mode-} windows;
				let
					(hScroll,vScroll) = validScrollInfos wMetrics okSize maybe_hScroll maybe_vScroll
				in return ( index
	                                  , okPos
	                                  , okSize
	                                  , toVector (origin-domain.corner1)
	                                  , wH { whItemNrs    = itemNrs
	                                       , whKeyFocus   = newFocusItems focusItems
	                                       , whWindowInfo = WindowInfo
	                                                           { windowDomain  = rectangleToRect domain
	                                                           , windowOrigin  = domain.corner1
	                                                           , windowHScroll = hScroll
	                                                           , windowVScroll = vScroll
	                                                           , windowLook    = {lookFun=look,lookPen=pen,lookSysUpdate=sysLook}
	                                                           , windowClip    = {clipRgn=0,clipOk=False}
	                                                           }
	                                       , whItems      = items1  - should be items2 as soon as getWElementKeyFocusIds is incorporated
	                                       , whSelect     = isAble
	                                       , whAtts       = atts9
	                                       , whDefaultId  = defid
	                                       , whCancelId   = canid
	                                       , whSize       = okSize
	                                       }
	                                  , windows
	                                  )
			}
		}
	  }
	where
		minSize = fromTuple osMinWindowSize
		isMDI   = getOSDInfoDocumentInterface osdInfo == MDI
		
		validScrollInfos :: OSWindowMetrics -> Size -> Maybe ScrollFunction -> Maybe ScrollFunction
		                 -> (Maybe ScrollInfo,Maybe ScrollInfo)
		validScrollInfos wMetrics wSize maybe_hScroll maybe_vScroll
			= (fmap (scrollInfo hScrollRect) maybe_hScroll,fmap (scrollInfo vScrollRect) maybe_vScroll)
		where
			windowRect   = sizeToRect wSize
			hasScrolls   = (isJust maybe_hScroll,isJust maybe_vScroll)
			hScrollRect  = getWindowHScrollRect wMetrics hasScrolls windowRect
			vScrollRect  = getWindowVScrollRect wMetrics hasScrolls windowRect
			
			scrollInfo :: Rect -> ScrollFunction -> ScrollInfo
			scrollInfo r@(Rect {rleft,rtop}) scrollFun
				= ScrollInfo
				  { scrollFunction = scrollFun
				  , scrollItemPos  = Point2 {x=rleft,y=rtop}
	  			  , scrollItemSize = rectSize r
				  , scrollItemPtr  = osNoWindowPtr
				  }
-}


determineRequestedSize :: Size -> Maybe Size -> Size
determineRequestedSize size Nothing  = size
determineRequestedSize _ (Just size) = size


{-	validateWindowIndex validates the WindowIndex attribute. 
	The return Index is the validated Index. 
	The return WindowAttribute list does not contain a WindowIndex attribute.
-}
validateWindowIndex :: {-WindowMode -> -} [WindowAttribute IF_MVAR(,ls ps)] -> WindowHandles IF_MVAR(,ps)
                                -> (Index,[WindowAttribute IF_MVAR(,ls ps)],   WindowHandles IF_MVAR(,ps))
validateWindowIndex {-mode-} atts windows@(WindowHandles {whsWindows=whsWindows})
	= (okIndex,atts',windows {whsWindows=modal'++modeless'})
	where
	{-	(_,indexAtt,atts')     = remove isWindowIndex (WindowIndex 0) atts
		index                  = getWindowIndexAtt indexAtt
	REPLACED BY: -}
		(index,atts')          = (0,atts)
		(modal,modeless)       = uspan isModalWindow whsWindows
		(nrModals,modal')      = ulength modal
		(nrModeless,modeless') = ulength modeless
		okIndex                = if   False -- mode==Modal
		                         then 0							-- Open modal windows frontmost
		                         else setBetween index nrModals (nrModals+nrModeless)	-- Open modeless windows behind the modal windows
		
		isModalWindow :: WindowStateHandle IF_MVAR(,ps) -> (Bool,WindowStateHandle IF_MVAR(,ps))
		isModalWindow wsH
			= (False,wsH)
		{-	= let (mode,wsH1) = getWindowStateHandleWindowMode wsH
			  in  (mode==Modal,wsH1)
		-}


{-	validateWindowPos validates the WindowPos attribute.
	If no WindowPos attribute is given then Nothing is returned.
	If the WindowPos is relative, it is verified that the window relates to an existing window.
	If this is not the case, then Nothing is returned.
	The resulting attribute list does not contain the WindowPos attribute anymore.
-}
validateWindowPos :: {-WindowMode -> -} [WindowAttribute IF_MVAR(,ls ps)] -> WindowHandles IF_MVAR(,ps)
                      -> (Maybe ItemPos,[WindowAttribute IF_MVAR(,ls ps)],   WindowHandles IF_MVAR(,ps))
validateWindowPos {-mode-} atts windows
	| not hasPosAtt
		= (Nothing,atts',windows)
	| not isRelative
		= (Just itemPos,atts',windows)
	| otherwise
		= let (found,windows1) = hasWindowHandlesWindow (toWID relativeTo) windows
		  in  (if found then Just itemPos else Nothing,atts',windows1)
	where
	{-	(hasPosAtt,posAtt,atts') = remove isWindowPos undef atts
		itemPos                  = getWindowPosAtt posAtt
	REPLACED BY: -}
		(hasPosAtt,itemPos,atts') = (False,(Left,zero),atts)
		(isRelative,relativeTo)  = isRelativeItemPos itemPos


{-	The result ({corner1=A,corner2=B},_) of validateWindowDomain is such that A<B (point A lies to 
	the left of and above point B). If either A.x==B.x or A.y==B.y then the ViewDomain is illegal and 
	the computation is aborted. 
	The default ViewDomain is maximal and positive, i.e.:
		{viewDomainRange & corner1=zero}.
-}
{-	Used only for Window
validateWindowDomain :: [WindowAttribute IF_MVAR(,ls ps)] -> (ViewDomain,[WindowAttribute IF_MVAR(,ls ps)])
validateWindowDomain atts
	| not hasDomain
		= (viewDomainRange {corner1=zero},atts1)
	| isEmptyRectangle domain
		= windowvalidateError "validateWindowDomain" "Window has illegal ViewDomain argument"
	| otherwise
		= (validateViewDomain domain,atts1)
	where
		(hasDomain,domainAtt,atts1) = remove isWindowViewDomain undef atts
		domain                      = getWindowViewDomainAtt domainAtt
-}
validateViewDomain :: ViewDomain -> ViewDomain
validateViewDomain domain
	= Rectangle
	     { corner1 = Point2
	                    { x = setBetween dl rl rr
	                    , y = setBetween dt rt rb
	                    }
	     , corner2 = Point2
	                    { x = setBetween dr rl rr
	                    , y = setBetween db rt rb
	                    }
	     }
	where
		(Rect {rleft=dl,rtop=dt,rright=dr,rbottom=db}) = rectangleToRect domain
		(Rect {rleft=rl,rtop=rt,rright=rr,rbottom=rb}) = rectangleToRect viewDomainRange


{-	validateWindowSize wMetrics viewDomain isMDI isResizable (hasHScroll,hasVScroll) atts
		takes care that the Window(View/Outer)Size attribute fits on the current screen.
		The Boolean  isMDI should be True iff the window belongs to a MDI process.
		The Boolean  isResizable should be True iff the window is resizable. 
		The Booleans hasHScroll hasVScroll should be True iff the window has the WindowHScroll, WindowVScroll
		attribute set respectively. 
		In addition, the WindowOuterSize attribute is mapped to WindowViewSize attribute.
-}
{-	Used only for Window
validateWindowSize :: OSWindowMetrics -> ViewDomain -> Bool -> Bool -> (Bool,Bool) -> [WindowAttribute IF_MVAR(,ls ps)]
                                                                          -> IO (Size,[WindowAttribute IF_MVAR(,ls ps)])
validateWindowSize wMetrics domain isMDI isResizable hasScrolls atts
	| not hasSize
		= return (pictSize,(WindowViewSize pictSize):atts)
		where
			domainSize = rectangleSize domain
			pictSize   = Size {w=min (w domainSize) (w maxSize),h=min (h domainSize) (h maxSize)}
	| isWindowViewSize sizeAtt
		= return (size1,snd (creplace isWindowViewSize (WindowViewSize size1) atts))
		where
			size       = getWindowViewSizeAtt sizeAtt
			size1      = Size {w=setBetween (w size) (fst minSize) (w maxSize),h=setBetween (h size) (snd minSize) (h maxSize)}
	| otherwise
		= do {
			(dw,dh)   <- osStripOuterSize isMDI isResizable;
			let
				(w',h')     = (w outerSize-dw,h outerSize-dh)
				visScrolls  = osScrollbarsAreVisible wMetrics (rectangleToRect domain) (w',h') hasScrolls
				viewSize    = rectSize (getWindowContentRect wMetrics visScrolls (sizeToRect (Size {w=w',h=h'})))
				(_,_,atts1) = remove isWindowOuterSize undef atts
				(_,_,atts2) = remove isWindowViewSize  undef atts1
			in return (viewSize,(WindowViewSize viewSize):atts)
		  }
		where
			outerSize = getWindowOuterSizeAtt sizeAtt
	where
		(hasSize,sizeAtt) = cselect (\att->isWindowViewSize att || isWindowOuterSize att) undef atts
		minSize           = osMinWindowSize
		maxSize           = maxScrollWindowSize
-}


{-	validateOrigin takes care that the WindowOrigin attribute is a point in the rectangle
	formed by the left top of the (validated!) ViewDomain, and the width and height of the 
	(validated!) derived size.
-}
{-	Used only for Window
validateOrigin :: Size -> ViewDomain -> [WindowAttribute IF_MVAR(,ls ps)] -> (Point2,[WindowAttribute IF_MVAR(,ls ps)])
validateOrigin (Size {w=w,h=h}) domain@(Rectangle {corner1=Point2 {x=l,y=t},corner2=Point2 {x=r,y=b}}) atts
	= (Point2 {x=setBetween (x origin) l (max l (r-w)),y=setBetween (y origin) t (max t (b-h))},atts1)
	where
		(_,domainAtt,atts1) = remove isWindowOrigin (WindowOrigin (corner1 domain)) atts
		origin              = getWindowOriginAtt domainAtt
-}

{-	validateWindow(H/V)Scroll removes the Window(H/V)Scroll attribute from the attribute list. 
-}
{-	Used only for Window
validateWindowHScroll :: [WindowAttribute IF_MVAR(,ls ps)] -> (Maybe ScrollFunction,[WindowAttribute IF_MVAR(,ls ps)])
validateWindowHScroll atts
	| found     = (Just (getWindowHScrollFun scrollAtt),atts1)
	| otherwise = (Nothing,atts1)
	where
			(found,scrollAtt,atts1) = remove isWindowHScroll undef atts
-}

{-	Used only for Window
validateWindowVScroll :: [WindowAttribute IF_MVAR(,ls ps)] -> (Maybe ScrollFunction,[WindowAttribute IF_MVAR(,ls ps)])
validateWindowVScroll atts
	| found     = (Just (getWindowVScrollFun scrollAtt),atts1)
	| otherwise = (Nothing,atts1)
	where
		(found,scrollAtt,atts1) = remove isWindowVScroll undef atts
-}


{-	validateWindowLook takes care that the optional WindowLook attribute is removed from the attribute list.
	If no attribute was present, then a default look is provided that paints the window with the background colour
	using standard window update mechanism.
-}
{-	Used only for Window
validateWindowLook :: [WindowAttribute IF_MVAR(,ls ps)] -> (Bool,Look,[WindowAttribute IF_MVAR(,ls ps)])
validateWindowLook atts
	= (sysLook,lookFun,atts1)
	where
		(_,lookAtt,atts1) = remove isWindowLook (WindowLook True defaultlook) atts
		(sysLook,lookFun) = getWindowLookAtt lookAtt
		
		defaultlook :: SelectState -> UpdateState -> Picture -> Picture
		defaultlook _ updState picture = strictSeq (map unfill (updArea updState)) picture
-}


--	Retrieve (View/Outer)Size, Margins, ItemSpaces, SelectState, and PenAttributes from the attribute list.

attrSize :: [WindowAttribute IF_MVAR(,ls ps)] -> Maybe Size
attrSize atts
	| not hasSize          = Nothing
	| isWindowViewSize att = Just (getWindowViewSizeAtt  att)
--	| otherwise            = Just (getWindowOuterSizeAtt att)
	where
	--	(hasSize,att)  = cselect (\att->isWindowViewSize att || isWindowOuterSize att) undef atts
		(hasSize,att)  = cselect isWindowViewSize undef atts

attrMargins :: WindowKind -> OSWindowMetrics -> [WindowAttribute IF_MVAR(,ls ps)] -> ((Int,Int),(Int,Int))
attrMargins wKind wMetrics atts
	= (getWindowHMargins wKind wMetrics atts,getWindowVMargins wKind wMetrics atts)

{-	SelectState attribute not incorporated yet.
attrSelectState :: [WindowAttribute IF_MVAR(,ls ps)] -> Bool
attrSelectState atts
	= enabled (getWindowSelectStateAtt (snd (cselect isWindowSelectState (WindowSelectState Able) atts)))
-}

{-	Pen attribute not incorporated yet.
attrPen :: [WindowAttribute IF_MVAR(,ls ps)] -> ([PenAttribute],[WindowAttribute IF_MVAR(,ls ps)])
attrPen atts
	= (getWindowPenAtt penAtt,atts1)
	where
		(_,penAtt,atts1) = remove isWindowPen (WindowPen []) atts
-}

{-	get(Ok/Cancel)Id select the Id of the Window(Ok/Cancel) attribute, and checks
	whether this Id corresponds with a (Custom)ButtonControl.
-}
{-	WindowOk attribute not incorporated yet.
getOkId :: [WindowAttribute IF_MVAR(,ls ps)] -> [WElementHandle IF_MVAR(,ls ps)] -> (Maybe Id,[WElementHandle IF_MVAR(,ls ps)])
getOkId atts itemHs
	| not hasid
		= (Nothing,itemHs)
	| ok
		= (Just id,itemHs1)
	| otherwise
		= (Nothing,itemHs1)
	where
		(hasid,idAtt) = cselect isWindowOk undef atts
		id            = getWindowOkAtt idAtt
		(ok,itemHs1)  = isOkOrCancelControlId id itemHs
-}
{-	WindowCancel attribute not incorporated yet.
getCancelId :: [WindowAttribute IF_MVAR(,ls ps)] -> [WElementHandle IF_MVAR(,ls ps)] -> (Maybe Id,[WElementHandle IF_MVAR(,ls ps)])
getCancelId atts itemHs
	| not hasid
		= (Nothing,itemHs)
	| ok
		= (Just id,itemHs1)
	| otherwise
		= (Nothing,itemHs1)
	where
		(hasid,idAtt) = cselect isWindowCancel undef atts
		id            = getWindowCancelAtt idAtt
		(ok,itemHs1)  = isOkOrCancelControlId id itemHs
-}
{-	Used only by getOkId and getCancelId.
isOkOrCancelControlId :: Id -> [WElementHandle IF_MVAR(,ls ps)] -> (Bool,[WElementHandle IF_MVAR(,ls ps)])
isOkOrCancelControlId id itemHs
	| isNothing maybeKind
		= (False,itemHs1)
	| otherwise
		= (kind==IsButtonControl || kind==IsCustomButtonControl,itemHs1)
	where
		(maybeKind,itemHs1) = getControlKind id itemHs
		kind                = fromJust maybeKind
-}


{-	validateWindowInitActive checks if the WindowInitActive attribute corresponds with an existing control.
	If this is not the case, the attribute is removed from the attribute list.
-}
{-	WindowInitActive attribute not incorporated yet.
validateWindowInitActive :: [WindowAttribute IF_MVAR(,ls ps)] -> [WElementHandle IF_MVAR(,ls ps)]
                        -> ([WindowAttribute IF_MVAR(,ls ps)],   [WElementHandle IF_MVAR(,ls ps)])
validateWindowInitActive atts itemHs
	| not hasAtt
		= (atts1,itemHs)
	| isNothing kind
		= (atts1,itemHs1)
	| otherwise
		= (atts,itemHs1)
	where
		(hasAtt,att,atts1) = remove isWindowInitActive undef atts
		(kind,itemHs1)     = getControlKind (getWindowInitActiveAtt att) itemHs
-}

{-	getControlKind id itemHs
		returns (Just ControlKind) of the control in the item list. 
		If no such control could be found then Nothing is returned.
-}
{-	Used only by isOkOrCancelControlId and validateWindowInitActive.
getControlKind :: Id -> [WElementHandle IF_MVAR(,ls ps)] -> (Maybe ControlKind,[WElementHandle IF_MVAR(,ls ps)])
getControlKind id (itemH:itemHs)
	= let (maybe,itemH1) = getControlKind' id itemH
	  in  if   isJust maybe
	      then (maybe,itemH1:itemHs)
	      else let (maybe,itemHs1) = getControlKind id itemHs
	           in  (maybe,itemH1:itemHs1)
	where
		getControlKind' :: Id -> WElementHandle IF_MVAR(,ls ps) -> (Maybe ControlKind,WElementHandle IF_MVAR(,ls ps))
#if MVAR
#else
		getControlKind' id (WExtendLSHandle addLS itemHs)
			= let (kind,itemHs1) = getControlKind id itemHs
			  in  (kind,WExtendLSHandle addLS itemHs1)
		getControlKind' id (WChangeLSHandle newLS itemHs)
			= let (kind,itemHs1) = getControlKind id itemHs
			  in  (kind,WChangeLSHandle newLS itemHs1)
#endif
		getControlKind' id (WListLSHandle itemHs)
			= let (kind,itemHs1) = getControlKind id itemHs
			  in  (kind,WListLSHandle itemHs1)
		getControlKind' id itemH@(WItemHandle {wItemId=wItemId,wItemKind=wItemKind})	--,wItems=itemHs})
			| isJust wItemId && fromJust wItemId==id
				= (Just wItemKind,itemH)
		{-	| otherwise
				= let (kind,itemHs1) = getControlKind id itemHs
				  in  (kind,itemH {wItems=itemHs1})	-}
getControlKind _ _
	= (Nothing,[])
-}


{-	exactWindowSize determines the exact size of a window.
	The size is extended to fit in sliderbars if requested (argument 4 and 5).
-}
exactWindowSize :: OSWindowMetrics -> ViewDomain -> Size -> Bool -> Bool -> WindowKind -> Size
exactWindowSize wMetrics domain wSize@(Size {w=w,h=h}) hasHScroll hasVScroll wKind
	| wKind==IsDialog          = wSize
	| visHScroll && visVScroll = Size {w=w',h=h'}
	| visHScroll               = wSize {h=h'}
	| visVScroll               = wSize {w=w'}
	| otherwise                = wSize
	where
		visHScroll         = hasHScroll && osScrollbarIsVisible (minmax (x $ corner1 domain) (x $ corner2 domain)) w
		visVScroll         = hasVScroll && osScrollbarIsVisible (minmax (y $ corner1 domain) (y $ corner2 domain)) h
		w'                 = w+osmVSliderWidth  wMetrics
		h'                 = h+osmHSliderHeight wMetrics


{-	exactWindowPos determines the exact position of a window.
	The size argument must be the exact size as calculated by exactWindowSize of the window.
	The ItemPos argument must be the validated(!) ItemPos attribute of the window.
-}
exactWindowPos :: OSWindowMetrics -> Size -> Maybe ItemPos -> WindowKind -> {-WindowMode -> -} WindowHandles IF_MVAR(,ps)
                                                                            -> IO (Point2,WindowHandles IF_MVAR(,ps))
exactWindowPos wMetrics exactSize maybePos wKind {-wMode-} windows
	| wKind==IsDialog -- && wMode==Modal
		= do {
			screenRect <- osScreenrect;
			let
				screenSize = rectSize screenRect
				l          = rleft screenRect + round ((fromIntegral (w screenSize - w exactSize))/2.0)
				t          = rtop  screenRect + round ((fromIntegral (h screenSize - h exactSize))/3.0)
				pos        = Point2 {x=setBetween l (rleft screenRect) (rright screenRect),y=setBetween t (rtop screenRect) (rbottom screenRect)}
			in return (pos,windows)
		  }
	| isNothing maybePos
		= return (zero,windows)
	| otherwise
		= do {
			(pos,windows1) <- getItemPosPosition wMetrics exactSize (fromJust maybePos) windows;
			pos1           <- setWindowInsideScreen pos exactSize;
			return (pos1,windows1)
		  }
	where
	{-	getItemPosPosition calculates the exact position of the given window. 
		getItemPosPosition does not check whether this position will place the window off screen.
	-}
		getItemPosPosition :: OSWindowMetrics -> Size -> ItemPos -> WindowHandles IF_MVAR(,ps)
		                   -> IO (Point2,WindowHandles IF_MVAR(,ps))
		getItemPosPosition wMetrics size itemPos windows@(WindowHandles {whsWindows=wsHs})
			| isRelative
				= do {
					rect <- osScreenrect;
					let	unidentifyWindow :: WID -> WindowStateHandle IF_MVAR(,ps) -> (Bool,WindowStateHandle IF_MVAR(,ps))
						unidentifyWindow wid wsH
							= let (ids,wsH1) = getWindowStateHandleWIDS wsH
							  in  (not (identifyWIDS wid ids),wsH1)
						
						screenDomain        = rectToRectangle rect
						screenOrigin        = Point2 {x=rleft rect,y=rtop rect}
						(before,after)      = uspan (unidentifyWindow (toWID relativeTo)) wsHs
						(wptr,wsH1,after')  = case after of
						                          [] -> windowvalidateFatalError "getItemPosPosition" "target window could not be found"
#if MVAR
						                          (wsH:rest) -> (wPtr $ wshIds wsH,wsH,rest)
#else
						                          (wsH@(WindowStateHandle wids wlsH):rest) -> (wPtr wids,wsH,rest)
#endif
						(relativeSize,wsH2) = getWindowStateHandleSize wsH1
						windows1            = windows {whsWindows=before++(wsH2:after')}
					in
					do {
						(relativeX,relativeY) <- osGetWindowPos wptr;
						let
							(relativeW,relativeH) = toTuple relativeSize
							(exactW,exactH)       = (w size,h size)
							v                     = itemPosOffset (snd itemPos) screenDomain screenOrigin
							(vx',vy')             = (vx v,vy v)
							pos                   = case (fst itemPos) of
							                            (LeftOf  _) -> Point2 {x=relativeX+vx'-exactW,   y=relativeY+vy'}
							                            (RightTo _) -> Point2 {x=relativeX+vx'+relativeW,y=relativeY+vy'}
							                            (Above   _) -> Point2 {x=relativeX+vx',          y=relativeY+vy'-exactH}
							                            (Below   _) -> Point2 {x=relativeX+vx',          y=relativeY+vy'+relativeH}
							                            other       -> windowvalidateFatalError "getItemPosPosition" "unexpected ItemLoc alternative"
						in return (pos,windows1)
					}
				  }
			| isAbsolute
				= do {
					rect <- osScreenrect;
					let	screenDomain        = rectToRectangle rect
						screenOrigin        = Point2 {x=rleft rect,y=rtop rect}
					in return (movePoint (itemPosOffset offset screenDomain screenOrigin) zero,windows)
				  }
			| isCornerItemPos itemPos
				= do {
					rect <- osScreenrect;
					let	screenDomain        = rectToRectangle rect
						screenOrigin        = Point2 {x=rleft rect,y=rtop rect}
						(exactW,exactH)     = toTuple size
						v                   = itemPosOffset (snd itemPos) screenDomain screenOrigin
						(vx',vy')           = (vx v,vy v)
						pos                 = case (fst itemPos) of
						                           LeftTop     -> Point2 {x=rleft  rect + vx',        y=rtop    rect + vy'}
						                           RightTop    -> Point2 {x=rright rect + vx'-exactW, y=rtop    rect + vy'}
						                           LeftBottom  -> Point2 {x=rleft  rect + vx',        y=rbottom rect + vy'-exactH}
						                           RightBottom -> Point2 {x=rright rect + vx'-exactW, y=rbottom rect + vy'-exactH}
					in return (pos,windows)
				  }
			| otherwise
				= return  (zero,windows)
			where
				(isRelative,relativeTo) = isRelativeItemPos itemPos
				(isAbsolute,offset)     = isAbsoluteItemPos itemPos
		
	{-	setWindowInsideScreen makes sure that a window at the given position and given size will be on screen.
	-}
		setWindowInsideScreen :: Point2 -> Size -> IO Point2
		setWindowInsideScreen pos@(Point2 {x=x,y=y}) size@(Size {w=w,h=h})
			= do {
				screenRect <- osScreenrect;
				let
					(Size {w=screenW,h=screenH})
					        = rectSize screenRect
					(x',y') = (setBetween x (rleft screenRect) (rright screenRect-w),setBetween y (rtop screenRect) (rbottom screenRect-h))
					pos1    = if   w<=screenW && h<=screenH then Point2 {x=x',y=y'}		-- window fits entirely on screen
					          else if w<=screenW            then Point2 {x=x',y=0 }		-- window is to high
					          else if h<=screenH            then Point2 {x=0, y=y'}		-- window is to wide
					                                        else zero			-- window doesn't fit anyway
				in return pos1
			  }


--	itemPosOffset calculates the actual offset vector of the given ItemOffset value.

itemPosOffset :: ItemOffset -> ViewDomain -> Point2 -> Vector2
{-
itemPosOffset NoOffset _ _
	= zero
-}
itemPosOffset {-(OffsetVector v)-}v _ _
	= v
{-
itemPosOffset (OffsetFun i f) domain origin
	| i==1       = f (domain,origin)
	| otherwise  = windowvalidateError "calculating OffsetFun" ("illegal ParentIndex value: "++show i)
-}


--	Predicates on ItemPos:
isRelativeItemPos :: ItemPos -> (Bool,Id)
isRelativeItemPos (LeftOf  id,_) = (True, id)
isRelativeItemPos (RightTo id,_) = (True, id)
isRelativeItemPos (Above   id,_) = (True, id)
isRelativeItemPos (Below   id,_) = (True, id)
isRelativeItemPos _              = (False,undef)

isAbsoluteItemPos :: ItemPos -> (Bool,ItemOffset)
isAbsoluteItemPos (Fix,offset) = (True, offset)
isAbsoluteItemPos _            = (False,undef)

isCornerItemPos :: ItemPos -> Bool
isCornerItemPos (LeftTop,_)     = True
isCornerItemPos (RightTop,_)    = True
isCornerItemPos (LeftBottom,_)  = True
isCornerItemPos (RightBottom,_) = True
isCornerItemPos _               = False
